'use strict'

var logger = require('../../logger')
var semver = require('semver')
var shimmer = require('../shimmer')
var appendMsg = "Mongodb :"

module.exports = function (mongodb, agent, version, enabled) {


    // 2.1.0
    var COLLECTION_OPERATIONS = [
        'aggregate',
        'bulkWrite',
        'count',
        'createIndex',
        'deleteMany',
        'deleteOne',
        'distinct',
        'drop',
        'dropAllIndexes',
        'dropIndex',
        'ensureIndex',
        'findAndModify',
        'findAndRemove',
        'findOne',
        'findOneAndDelete',
        'findOneAndReplace',
        'findOneAndUpdate',
        'geoHaystackSearch',
        'geoNear',
        'group',
        'indexes',
        'indexExists',
        'indexInformation',
        'insert',
        'insertMany',
        'insertOne',
        'isCapped',
        'mapReduce',
        'options',
        'parallelCollectionScan',
        'reIndex',
        'remove',
        'rename',
        'replaceOne',
        'save',
        'stats',
        'update',
        'updateMany',
        'updateOne',
        'find'
    ]

    var CURSOR_OPS = [
        'nextObject',
        'next',
        'toArray',
        'count',
        'explain'
    ]

    var DB_OPS = [
        'addUser',
        'authenticate',
        'collection',
        'collectionNames',
        'collections',
        'command',
        'createCollection',
        'createIndex',
        'cursorInfo',
        'dereference',
        'dropCollection',
        'dropDatabase',
        'dropIndex',
        'ensureIndex',
        'eval',
        'executeDbAdminCommand',
        'indexInformation',
        'logout',
        'open',
        'reIndex',
        'removeUser',
        'renameCollection',
        'stats',
        '_executeInsertCommand',
        '_executeQueryCommand'
    ]


    logger.debug('shimming mongodb version: ' ,version )


    /**
       For aggregate if there is no callback is provided then 
       the cursor is returned
       1. First check if callback is provided. enable the boolean 
       2. else listen on 'close' 'end' events to close the connection
       cursor.operations totally different from collection operations
     */


    shimmer.massWrap(mongodb.Collection.prototype, COLLECTION_OPERATIONS, function (collection, name) {

        return function () {

            var span = agent.buildSpan()

            if (!span) {
                return collection.apply(this, arguments)
            }

            var last = arguments.length - 1;

            var hasCallback = false;

            if (typeof arguments[last] === 'function') {
                hasCallback = true;
                var cb = arguments[last];

                arguments[last] = function wrapCallback() {
                    span.end();
                    return cb.apply(this, arguments)
                }
            }

            span.start()

            var emit = collection.apply(this, arguments)

            span.type = 'mongodb' // 'DB.MONGODB.COLLECTIONS'
            span.name = 'fn=>' + name;
            span.options = getDbDetails(this);
            span.options.methodName = 'MONGODB.COLLECTIONS.' + name;
            span.options.query = JSON.stringify(arguments[0]) || ''
            span.options.serverType = 'MONGODB';
            if (!hasCallback && emit) {
                wrapFunction(emit, span);
            }

            return emit;
        }

    })



    if (mongodb.Cursor && mongodb.Cursor.prototype) {

        logger.debug(appendMsg,'wrapping mongodb cursors')
        // refer http://mongodb.github.io/node-mongodb-native/2.0/api/AggregationCursor.html
        var proto = mongodb.Cursor.prototype

        shimmer.massWrap(proto, CURSOR_OPS, function (cursor, cursorName) {

            return function () {

                var trace = agent.buildSpan()

                if (!trace) {
                    logger.debug(appendMsg,' mongoo cursor ',cursorName,' trace is null')
                    return cursor.apply(this, arguments)
                }

                var last = arguments.length - 1;

                if (typeof arguments[last] === 'function') {

                    var cb = arguments[last];

                    arguments[last] = function wrapCallback() {
                        logger.debug(appendMsg,'mongodb cursor end==>', cursorName)
                        trace.end()
                        return cb.apply(this, arguments)
                    }
                }

                trace.start()
                trace.name = cursorName;
                trace.type = 'mongodb' // 'MONGODB.CURSOR'
                trace.options = getDbDetails(this);
                trace.options.methodName = 'MONGODB.CURSOR.' + cursorName ;
                trace.options.query = JSON.stringify(arguments[0]) || ''
                trace.options.serverType = 'MONGODB';
                return cursor.apply(this, arguments)
            }

        })

    }

    logger.debug(appendMsg,'wrapping mongo - Db operations')

    if (mongodb.Db && mongodb.Db.prototype) {
        var proto = mongodb.Db.prototype

        shimmer.massWrap(proto, DB_OPS, function (dbOPS, name) {
            return function () {
                var span = agent.buildSpan()

                if (!span) {
                    logger.debug('span is null')
                    return dbOPS.apply(this, arguments)
                }

                var last = arguments.length - 1;

                var hasCallback = false;

                if (typeof arguments[last] === 'function') {
                    hasCallback = true;
                    var cb = arguments[last];

                    arguments[last] = function wrapCallback() {
                        logger.debug("mongodb call back ended")
                        span.end();
                        return cb.apply(this, arguments)
                    }
                }

                span.start();

                logger.debug(appendMsg,'mongodb function started', name, name == 'command' ? arguments : '')
                var emit = dbOPS.apply(this, arguments)

                span.type = 'mongodb' //'DB.MONGODB_OPERATION'
                span.name = name;
                span.options = getDbDetails(this);
                span.options.methodName = 'DB.MONGODB_OPERATION.' + name;
                span.options.query = JSON.stringify(arguments[0]) || ''
                span.options.serverType = 'MONGODB';
                if (!hasCallback && emit) {

                    wrapFunction(emit, span);


                    if (emit && emit.on && typeof emit.on === 'function') {
                        // refer http://mongodb.github.io/node-mongodb-native/2.0/api/AggregationCursor.html
                        if (emit.prependListener) {
                            emit.prependListener('close', function () {
                                logger.debug(appendMsg,'prependListener close')
                                span.end();
                            })
                        } else {
                            emit.on('close', function () {
                                span.end();
                            })
                        }

                        emit.prependListener('error', function () {
                            span.end();
                        })
                    } else if (typeof emit.then === 'function') {

                        return emit.then(
                            function (v) {
                                span.end();
                                return v
                            },
                            function (err) {
                                throw err
                            }
                        )
                       
                    }
                }

                return emit;
            }

        })
    }

    function wrapFunction(emit, span) {

        if (emit && emit.on && typeof emit.on === 'function') {
            // refer http://mongodb.github.io/node-mongodb-native/2.0/api/AggregationCursor.html
            if (emit.prependListener) {
                emit.prependListener('close', function () {
                    logger.debug(appendMsg,'prependListener close')
                    span.end();
                })
            } else {
                emit.on('close', function () {
                    span.end();
                })
            }

            emit.prependListener('error', function () {
                span.end();
            })
        } else if (emit.then && typeof emit.then === 'function') {

            return emit.then(
                function (v) {
                    span.end();
                    return v
                },
                function (err) {
                    span.end(err);
                    throw err
                }
            )
        }

    }

    return mongodb;

}


//Show these params for db details
/**
 Database name
 host
 port_path_or_id
 instance
 */
function getDbDetails(instance) {
    
    var options = {}
    options.error = ''
    options.nodeOrder = ''
    options.dbName = '';
    try {

        options.collectionName = instance.collectionName || 'unknown'

        if (instance.db && instance.db.serverConfig) {
            var serverConfig = instance.db.serverConfig
            options.dbName = serverConfig.db || serverConfig.dbInstance
            options.host = instance.db.serverConfig.host
            options.port = instance.db.serverConfig.port;


        } else if (instance.s && instance.s.topology && instance.s.topology.host) {
            options.host = instance.s.topology.host;
            options.port = instance.s.topology.port;
            options.tableName = instance.s.name || '';
            options.dbName = instance.s.dbName || instance.s.databaseName;

            if (!options.tableName) {

                if (instance.ns) {

                    var ns = instance.ns.split(/\./)
                    options.tableName = ns[1];
                    options.dbName = ns[0];
                }
            }

        }

    } catch (e) {
        logger.debug('error', e)
    }


    //Need to check how to collect other params
    // if (instance.db && instance.db.serverConfig) {
    //     host = instance.db.serverConfig.host 
    //     port = instance.db.serverConfig.port;
    // } else if (instance.s && instance.s.topology && instance.s.topology.isMasterDoc) {
    //     host = instance.s.topology.isMasterDoc.primary
    //     port = ''
    // }

    return options;
}