'use strict'

var symbols = require('./symbols')
var log = require('./logger')
var DEFAULT_CONFIG = require('./default_config')
var InfraManager = require('./infra-manager');
var Collector = require("./collector");
var Socket = require('./collector/socket-client')
var os = require('os')
var errorCapture = require('./instrumentation/modules/wrap_error')
var Instrumentation = require('./instrumentation')
var appendMsg = 'Agent :'

module.exports = Agent

var STATES = {
  'STARTING': 'starting',
  'RUNNING': 'running',
  'STOPPED': 'stopped',
  'ERROR': 'error'
};

function Agent() {
  this.version = '1.0.0'
  this.state = STATES.STOPPED
  this.config = DEFAULT_CONFIG
  this.infraManager = new InfraManager(this)
  this.socket = new Socket(this)
  this.collector = new Collector(this);

  this._instrumentation = new Instrumentation(this)
  this._httpClient = null
}

Object.defineProperty(Agent.prototype, 'currentTransaction', {
  get() {
    return this._instrumentation.currentTransaction
  }
})


/**
 * Start the agent
 * @param {Object} options contains config details appname, key, log level etc 
 * Load default config
 * Start agent to collecting process level metrics
 */
Agent.prototype.start = function (options) {

  if (global.EGURKHA) {
    log.warn(appendMsg,'Do not start ' + EGURKHA + ' more than once')
    return this;
  } else {

    log.info(appendMsg, 'started...')
    this.setConfig(options);
   
    appInfo()
    
    this._instrumentation = new Instrumentation(this)

    this.state = STATES.STARTING;

    if (!options || !options.component_id) {
      log.error(appendMsg, 'component_id is missing, please add the component id in the config and then start again..')
      log.error(appendMsg,'Agent is stopped')
      this.state = STATES.STOPPED;
      return this;
    } else {
      this.state = STATES.RUNNING;

      global.EGURKHA = 'EG-Node-Monitor';

      if (this.config.inframetrics_enabled) {
        this.infraManager.start();
      } else {
        log.info(appendMsg,'Inframetrics capturing disabled')
      }

      //Start btm instrumentation
      if (!this.config.disable_btm) {
        log.info(appendMsg,"BTM is enabled ")
        this._instrumentation.start()
      } else {
        log.info(appendMsg,"BTM is disabled")
      }

      if(this.config.capture_uncaught_exceptions){
        log.info(appendMsg,"Capturing uncaught exceptions is enabled ")
        errorCapture(this)
      }else{
        log.debug(appendMsg,"Capturing uncaught exceptions is disabled ")
      }
     

      log.info(appendMsg,'Config log level:', this.config.log_level ? this.config.log_level : log.level)
      log.setDirectory();

      if (this.config.log_level) {
        log.info(appendMsg,'Log level changed into:', this.config.log_level)
        log.setLevel(this.config.log_level);
      }

      this.socket.makeConnection(this.config.agent_host, this.config.agent_port);
    }

  }

  return this
}

/**
 * Stop the agent
 */
Agent.prototype.stop = function (cb) {
  try {
    
      this.state = STATES.STOPPED;

      //Stop infra manager stopping to collect metrics
     if(this.infraManager.isStarted){
        this.infraManager.stop();
     }
     
     log.info(appendMsg,'Agent stopped')

      if (cb && typeof cb === 'function') {
          cb();
      }

  } catch (e) {
      log.error(appendMsg,'error while agent stop ', e)
  }
}
/**
 * To start a new transaction
 */
Agent.prototype.startTransaction = function () {
  return this._instrumentation.startTransaction.apply(this._instrumentation, arguments)
}
/**
 * To end a running transaction
 */
Agent.prototype.endTransaction = function () {
  return this._instrumentation.endTransaction.apply(this._instrumentation, arguments)
}
/**
 * To set a transaction name for transaction
 * default is request url
 */
Agent.prototype.setTransactionName = function () {
  return this._instrumentation.setTransactionName.apply(this._instrumentation, arguments)
}

/**
 * To start a new span
 */
Agent.prototype.startSpan = function (name, type) {
  var span = this._instrumentation.buildSpan.apply(this._instrumentation)
  if (span) span.start(name, type)
  return span
}
/**
 * To get all spans for the current transaction
 */
Agent.prototype.buildSpan = function () {
  return this._instrumentation.buildSpan.apply(this._instrumentation, arguments)
}

/**
 * To set addional message for transaction
 * ex insert some message
 */
Agent.prototype.setContext = function (context) {
  var transaction = this.currentTransaction
  if (!transaction){
    return false
  } 
  transaction.setContext(context)
  return true
}

/**
 * Set an external config in Agent 
 * @param {Object} options
 */
Agent.prototype.setConfig = function (options) {

  log.info(appendMsg,"User config options:", options)

  try {
    var keys = Object.keys(options)

    for (var key in keys) {
      this.config[keys[key]] = options[keys[key]];
    }

  } catch (e) {
    log.error(appendMsg,'Agent setConfig ', e)
  }
}


function appInfo() {
  log.info('Nodejs process id :', process.pid)
  log.info('System uptime :', process.uptime())
  log.info('eG Start time :', Date())
  log.info('Nodejs version :', process.versions ? process.versions.node : '')
  log.info('V8 version :', process.versions ? process.versions.v8 : '')
  log.info('Libuv version :', process.versions ? process.versions.uv : '')
  log.info('Nodejs LTS :', process.release ? process.release.lts : '')
  log.info('OS Type:', os.type ? os.type() : '')
  log.info('OS Architecture :', os.arch ? os.arch() : '')
  log.info('OS Platform:', os.platform ? os.platform() : '')
  //Load default config 
  log.info(appendMsg,'after merged DEFAULT_CONFIG :', DEFAULT_CONFIG);
}