'use strict'

const logger = require('../logger');
const appUtils = require('../app-utils');

const appendMsg = 'Container-ID:';

async function getContainerID(config, containerType) {
  try {
    let containerId = null;
    let cmd = config.cmd_to_get_container_id;

    /**
     * If the our cmd to get container ID is not working then,
     * to give immediate solution we have to set correct cmd in
     * the enviromental value EG_CMD_TO_GET_CONTAINER_ID to get correct ID
     */
    if (cmd) {
      cmd = (Buffer.from(cmd, 'base64')).toString();
      if (cmd) containerId = await _findContainerID(cmd, "ENV_CMD");
    }

    if (!containerId) {
      cmd = containerType.getCmdForContainerID();
      const line = containerType.getContainerTypeDecidedLine();
      containerId = await _findContainerID(cmd, containerType.toString());
      if (!containerId) containerId = await _getContainerIDUsingMultiCmds(); // 2nd attempt using multiple cmd
      if ((!containerId || !isValid64CharId(containerId)) && line) // 3rd attempt using regex
        containerId = getContainerIDUsingRegEx(line);
    }

    if (!containerId) return;
    containerId = containerId.trim();
    if (containerId.length >= 64) {
      containerId = containerId.substring(0, 12);
    }

    return containerId || null;
  } catch (e) {
    logger.error(appendMsg, 'get error', e);
  }
}

async function _findContainerID(cmd, msg) {
  if (!cmd) return;

  try {
    const stdout = await appUtils.executeCmd(cmd, true) || null;
    logger.info(appendMsg, `${cmd} is runned to get Container ID for ${msg} and the output is`, stdout, logger.doConsole);
    return stdout;
  } catch (e) {
    logger.error(appendMsg, msg + ' - _findContainerID error for cmd:' + cmd, e);
  }
}

async function _getContainerIDUsingMultiCmds() {
  let containerId = null;
  // Getting docker id from Kubernetes pod env 
  let cmd = `cat /proc/self/cgroup | grep -e "cpu" | grep -o  -e "docker-.*.scope" | head -n 1 | sed "s/docker-\\(.*\\).scope/\\\\1/"`;
  containerId = await _findContainerID(cmd, "DOCKER");
  if (containerId) return containerId;


  // Checking For Crio - Container - Kubernetes 
  cmd = `cat /proc/self/cgroup | grep -e "cpu" | grep -o  -e "crio-.*" | head -n 1 | sed "s/crio-\\(.*\\)/\\\\1/"`;
  containerId = await _findContainerID(cmd, "CRI-O_K8S");
  if (containerId) return containerId;

  // Checking For Crio - Container - Openshift/Kubernetes
  cmd = `cat /proc/self/cgroup | grep -e "cpu" | grep -o  -e "crio-.*.scope" | head -n 1 | sed "s/crio-\\(.*\\).scope/\\\\1/"`;
  containerId = await _findContainerID(cmd, "CRI-O");
  if (containerId) return containerId;

  // Checking For AWS EKS & AWS ECS 
  cmd = `cat /proc/self/cgroup  | grep -e "cpu" | awk -F"/" '{print $NF}' | head -n 1`;
  containerId = await _findContainerID(cmd, "AWS-EKS-ECS");
  if (containerId) return containerId;

  cmd = `cat /proc/self/cgroup | grep -e "cpu" | grep -o  -e "docker.*" | head -n 1 | sed "s/docker\/\(.*\)/\\1/" | cut -c 1-13`;
  containerId = await _findContainerID(cmd, "DOCKER");
  if (containerId) return containerId;

  // Print cgroup entry for debugging
  cmd = `cat /proc/self/cgroup | head -1`;
  await _findContainerID(cmd, "CONTAINER");
  return containerId;
}

function isValid64CharId(containerId) {
  const findString = new RegExp("^([a-z0-9]){64}$");
  return findString.test(containerId);
}

function getContainerIDUsingRegEx(matcher) {
  const findString = new RegExp("([a-z0-9]){64}");
  const matched = findString.exec(matcher);
  if (matched && matched[0]) {
    logger.info(appendMsg, `Container ID by RegEx: ${matched[0]} from the line: ${matcher}`, logger.doConsole);
    return matched[0];
  }

  return '';
}

exports.get = getContainerID;