'use strict'

const fs = require('fs');

const logger = require('../logger');
const symbols = require('../symbols');
const shimmer = require('./shimmer');

const appendMsg = 'ErrorReporter:'
module.exports = ErrorReporter;
const errorFileName = 'fatal_error.json';

function ErrorReporter(_agent) {
  this._agent = _agent;
  this.isRunning = false;
  this.isFirstTime = true;
}

function setFatalErrorReport() {
  if (!process.report) return;

  process.report.compact = false;
  process.report.reportOnFatalError = true;
  process.report.directory = __dirname + '/../../../../';
  process.report.filename = errorFileName;
}

/** Just to make sure our config is not changed */
//setTimeout(setFatalErrorReport, 1000 * 60);

ErrorReporter.prototype.start = function () {
  const _this = this;

  // if (process.report) {
  //   setFatalErrorReport();
  //   this.reportIfAnyFatalError();
  //   logger.info(appendMsg, "process.report is enabled.");
  // } else {
  //   logger.error(appendMsg, "process.report is not supported.");
  // }

  if (this.isRunning) return;
  if (this.isFirstTime) {
    process.on('uncaughtException', function (err) {
      _this.captureError('uncaughtException', err);
    });

    process.on('unhandledRejection', function (err) {
      _this.captureError('unhandledRejection', err);
    });

    process.on('beforeExit', function (err) {
      _this.captureError('beforeExit', err);
    });

    process.on('uncaughtExceptionMonitor', function (err) {
      _this.captureError('beforeExit', err);
    });

    process.on('multipleResolves', function (err) {
      _this.captureError('multipleResolves', err);
    });
    this.isFirstTime = false;
  }

  if (process.emit) {
    const allowedReasons = ["warning"];

    shimmer.wrap(process, 'emit', function processEmitWrapper(originalFunc) {
      return function processEmitWrapped(reason, error) {
        if (allowedReasons.indexOf(reason) > -1 && error) {
          if (error.code && error.code.substring(0, 3) === 'DEP') {
            logger.warn(appendMsg, 'Deprecation Warning', error);
            return;
          }

          if (error.name === "ExperimentalWarning") {
            logger.warn(appendMsg, 'ExperimentalWarning', error);
            return;
          }

          _this.captureError(reason, error);
        }
        return originalFunc.apply(this, arguments);
      }
    });
    logger.info(appendMsg, "Wrapped process.emit to handle the errors");
  } else {
    logger.error(appendMsg, 'Process.emit not found');
  }

  logger.info(appendMsg, "Error reporter started successfully...!");
  this.isRunning = true;
}

ErrorReporter.prototype.captureError = function (reason, error) {
  if (!this.isRunning) return;
  if (!error || (error instanceof Error && error[symbols.errorReportedSymbol] === true)) return;
  logger.error(appendMsg, 'Type:', reason, 'Error:', error.message);
  this._agent.captureError(error);
}

ErrorReporter.prototype.stop = function () {
  if (!this.isRunning) return;
  this.isRunning = false;
  if (process.report) {
    process.report.reportOnFatalError = false;
  }

  if (process.emit) {
    shimmer.unwrap(process, 'emit');
  }
  logger.info(appendMsg, "Error reporter stoped successfully...!");
}

/**
 * if the application is down due to any fatal error (like heap out of memory),
 * then the nodejs will write the error details in the file.
 * this fn will process that error and send to agent.
 */
ErrorReporter.prototype.reportIfAnyFatalError = function () {
  if (!process.report) return;

  try {
    const path = process.report.directory + process.report.filename;
    let errReport = fs.readFileSync(path, 'utf8');
    if (!errReport) return;
    fs.unlinkSync(path);
    logger.info(appendMsg, "Fatal Error found. Going to process it.");

    errReport = JSON.parse(errReport);
    const header = errReport.header || {};
    const err = {
      message: header.event || header.trigger || 'Fatal Error',
      stack: ''
    };

    if (errReport.javascriptStack) {
      const jsStack = errReport.javascriptStack;
      if (jsStack.message && jsStack.message != 'No stack.') {
        err.message += ' ' + jsStack.message;
      }

      if (jsStack.stack && jsStack.stack.length > 0) {
        const stack = jsStack.stack.join('\n');
        if (stack && stack != 'Unavailable.') {
          err.stack = stack;
        }
      }
    }

    const nativeStack = errReport.nativeStack;
    if (nativeStack && nativeStack.length) {
      const stack = nativeStack.map(e => `${e.pc} - ${e.symbol}`).join('\n');
      if (stack && stack != 'Unavailable.') {
        if (err.stack) err.stack += '\n';
        err.stack += stack;
      }
    }

    err.details = {
      javascriptHeap: errReport.javascriptHeap,
      resourceUsage: errReport.resourceUsage,
      libuv: errReport.libuv,
    }
    err[symbols.errorParsedSymbol] = true;
    this._agent.captureError(err, {
      pid: header.processId,
      time: header.dumpEventTimeStamp && Number(header.dumpEventTimeStamp) || null
    });
  } catch (e) {
    /** if no file then you can ignore */
    if (e && e.code === 'ENOENT') return;
    logger.error(appendMsg, 'reportIfAnyFatalError:', e);
  }
}