'use strict'

const semver = require('semver');

const shimmer = require('../shimmer');
const logger = require('../../logger');
const utils = require('../instrumentation-utils');

let isWrapped = false;
const appendMsg = 'Mysql2:';

exports.start = function (mysql2, agent, version, enabled) {
  if (!enabled) return mysql2;
  const ins = agent._instrumentation;

  try {
    isWrapped = true;
    if (!semver.satisfies(version, '>=1 <3')) {
      logger.info(appendMsg, 'mysql2 version is not supported - aborting...', version);
      return mysql2;
    }

    logger.debug(appendMsg, 'wrapping mysql2.Connection.prototype.wrapQuery');
    shimmer.wrap(mysql2.Connection.prototype, 'query', wrapQuery);

    logger.debug(appendMsg, 'wrapping mysql2.Connection.prototype.execute');
    shimmer.wrap(mysql2.Connection.prototype, 'execute', wrapQuery);

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }
  return mysql2;

  function wrapQuery(original) {
    return function wrappedQuery(sql, values, cb) {
      const span = agent.startSpan('', 'mysql', 'MySql2');
      if (!span) return original.apply(this, arguments);

      const id = span.transaction.id
      let hasCallback = false
      let sqlStr = sql
      span.options = getParameters(this);

      switch (typeof sql) {
        case 'string':
          sqlStr = sql
          break
        case 'object':
          if (typeof sql.onResult === 'function') {
            sql.onResult = wrapCallback(sql.onResult)
          }
          sqlStr = sql.sql
          break
        case 'function':
          arguments[0] = wrapCallback(sql)
          break
      }

      if (sqlStr) {
        logger.debug(appendMsg, 'extracted sql from mysql2 query ', { id: id, sql: sqlStr });
        if (sqlStr && span.options) {
          span.options.query = sqlStr;
        }
        span.name = utils.getSpanNameFromQuery(sqlStr);
      }

      if (typeof values === 'function') {
        arguments[1] = wrapCallback(values)
      } else if (typeof cb === 'function') {
        arguments[2] = wrapCallback(cb)
      }

      const result = original.apply(this, arguments);

      if (result && !hasCallback) {
        ins.bindEmitter(result)
        shimmer.wrap(result, 'emit', function (original) {
          return function (event, err) {
            switch (event) {
              case 'error':
                if (err) {
                  logger.debug(appendMsg, 'error captured');
                  span.captureError(err);
                }
              case 'close':
              case 'end':
                span.end()
            }
            return original.apply(this, arguments)
          }
        })
      }

      return result

      function wrapCallback(cb) {
        hasCallback = true
        return agent.bindFunction(span ? wrappedCallback : cb)
        function wrappedCallback(err) {
          //Capture error 
          if (err) {
            logger.debug(appendMsg, 'error captured');
            span.captureError(err);
          }

          span.end()
          return cb.apply(this, arguments)
        }
      }
    }
  }

  function getParameters(sqlObj) {
    var params = {};
    if (sqlObj.config) {
      params.host = sqlObj.config.host
      params.port = sqlObj.config.port
      params.dbName = sqlObj.config.database
      params.serverType = 'mysql'
    }

    return params;
  }
}

exports.stop = function (mysql2, version) {
  if (!isWrapped) return;
  shimmer.unwrap(mysql2.Connection.prototype, 'query');
  shimmer.unwrap(mysql2.Connection.prototype, 'execute');
  isWrapped = false;
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}