'use strict'

const endOfStream = require('end-of-stream');

const shimmer = require('../shimmer');
const logger = require('../../logger');
const symbols = require('../../symbols');
const httpShared = require('../http-shared');
const { parseUrl } = require('../../parser');
const appConstant = require('../../app-constants');
const httpStatusText = require('../http-status-text');

const resTransformer = require('../http-res-transformer');

let isRunning = false;
const appendMsg = 'Http2:';
const eGGUID = appConstant.GUID_NAME;

exports.start = function (http2, agent, version, enabled) {
  if (!enabled) return http2;
  const ins = agent._instrumentation;
  isRunning = true;

  try {
    logger.debug(appendMsg, 'shimming http2.createServer function');
    shimmer.wrap(http2, 'createServer', wrapCreateServer);
    shimmer.wrap(http2, 'createSecureServer', wrapCreateServer);

    logger.debug(appendMsg, 'shimming http2.connect function');
    shimmer.wrap(http2, 'connect', wrapConnect);

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }
  return http2;

  // The `createServer` function will unpatch itself after patching
  // the first server prototype it patches.
  function wrapCreateServer(original) {
    return function wrappedCreateServer(options, handler) {
      const server = original.apply(this, arguments);
      shimmer.wrap(server.constructor.prototype, 'emit', wrapEmit);
      logger.debug(appendMsg, 'intercepted request event call to http2.Server.prototype.emit');
      wrappedCreateServer[symbols.unwrap]();
      return server;
    }
  }

  function reqIsHTTP1(req) {
    return req && typeof req.httpVersion === 'string' && req.httpVersion.startsWith('1.');
  }

  function wrapEmit(original) {
    let patched = false;
    return function wrappedEmit(event, stream, headers) {
      if (event === 'stream') {
        if (!patched) {
          patched = true;
          const proto = stream.constructor.prototype;
          shimmer.wrap(proto, 'pushStream', wrapPushStream);
          shimmer.wrap(proto, 'respondWithFile', wrapRespondWith);
          shimmer.wrap(proto, 'respondWithFD', wrapRespondWith);
          shimmer.wrap(proto, 'respond', wrapHeaders);
        }

        const url = headers[':path'];
        logger.debug(appendMsg, 'New incoming http2 request call url:', url);

        if (!isRunning || httpShared.isUrlExcluded(agent, url)) {
          logger.debug(appendMsg, 'ignoring excluded url', url);
          ignoreTheRequest();
        } else {
          const method = headers[':method'];
          const header = Object.assign({}, headers);
          const option = httpShared.getRequestIdentifier(agent, url, header, method, stream.session.socket.remoteAddress);
          option.method = method;
          const trans = agent.startTransaction('http2:inbound', option);

          if (trans) {
            trans.req = req;
            ins.bindEmitter(stream);

            endOfStream(stream, function () {
              logger.debug(appendMsg, "call ended with status code:", trans.requestIdentifier.statusCode, 'id:', trans.id);
              trans.end();
            });
          } else {
            ignoreTheRequest();
          }
        }
      } else if (event === 'request') {
        const req = stream;
        const res = headers;

        if (reqIsHTTP1(stream)) {
          // http2.createSecureServer() supports a `allowHTTP1: true` option.
          // When true, an incoming client request that supports HTTP/1.x
          httpShared.onNewRequest(agent, 'HTTP2', req, res);
        } else {
          const trans = agent.currTransaction();
          if (trans) {
            resTransformer.transform(agent, req, res, 'HTTP2');
          }
        }
      }

      return original.apply(this, arguments);
    }

    function ignoreTheRequest() {
      // don't leak previous transaction
      agent._instrumentation.supersedeWithEmptyRunContext();
      // for testing perpose, we are calling the fn. don't remove this.
      if (agent._externalTransport) agent._externalTransport();
    }
  }

  function updateHeaders(headers) {
    if (!isRunning) return;
    const trans = agent.currTransaction();
    if (!trans) return;

    const status = headers[':status'] || 200;
    trans.requestIdentifier.statusCode = status;
    trans.requestIdentifier.statusMessage = httpStatusText.get(status) || "";

    if (appConstant.REDIRECT_STATUS_CODE.indexOf(status) > -1 && headers['Location']) {
      trans.requestIdentifier.isRequestRedirect = true;
    }
  }

  function wrapHeaders(original) {
    return function (headers) {
      updateHeaders(headers);
      return original.apply(this, arguments);
    }
  }

  function wrapRespondWith(original) {
    return function (_, headers) {
      updateHeaders(headers);
      return original.apply(this, arguments);
    }
  }

  function wrapPushStream(original) {
    return function wrappedPushStream(...args) {
      if (!isRunning) return original.apply(this, args);
      // Note: Break the run context so that the wrapped `stream.respond` et al
      // for this pushStream do not overwrite outer transaction state.
      const callback = args.pop();
      args.push(agent._instrumentation.bindFunctionToEmptyRunContext(callback));
      return original.apply(this, args);
    }
  }

  function wrapConnect(orig) {
    return function (host) {
      const ret = orig.apply(this, arguments);
      shimmer.wrap(ret, 'request', orig => wrapRequest(orig, host));
      return ret;
    }
  }

  function wrapRequest(orig, host) {
    return function (headers) {
      if (!isRunning) return orig.apply(this, arguments);

      const urlObj = parseUrl(headers[':path']);
      const uri = host + urlObj.pathname;
      const method = headers[':method'] || 'GET';
      const span = agent.startSpan(method + ' ' + uri, 'http:outbound', 'Http2');
      const id = span && span.transaction.id;
      logger.debug(appendMsg, 'intercepted call to http2.request, id:', id);
      headers[eGGUID] = span.getNeweGGUID();
      const options = {};
      options.uri = uri;
      options.method = method;
      options.nodeOrder = span.nodeOrder;
      span.options = options;

      const req = orig.apply(this, arguments);
      if (!span) return req;

      ins.bindEmitter(req);
      req.on('response', (resHeaders) => {
        span.options.statusCode = resHeaders[':status'] || null;
      });

      req.on('error', function (err) {
        if (err) {
          logger.debug(appendMsg, 'error captured');
          span.captureError(err);
        }

        span.end();
      });

      req.on('end', () => {
        logger.debug(appendMsg, 'http outbound request event is completed', {
          id: id,
          url: span.options.uri
        })
        span.end();
      });


      return req;
    }
  }
}

exports.stop = function (http2, version) {
  isRunning = false;
  shimmer.unwrap(http2, 'createServer');
  shimmer.unwrap(http2, 'createSecureServer');
  shimmer.unwrap(http2, 'connect');
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}