'use strict'

const os = require('os');
const fs = require('fs');
const moment = require('moment-timezone');
const exec = require('child_process').exec;

let timezone = '';
let utcOffset = '';
const appendMsg = 'AppUtils:';
const logger = require('./logger');

try {
  utcOffset = moment().utcOffset();
  timezone = moment.tz(moment.tz.guess()).zoneAbbr();
} catch (e) {
  throw e;
}

function getTimezone() {
  return timezone;
}

function getUtcOffset() {
  return utcOffset;
}

function getTempDirPath() {
  try {
    return fs.realpathSync(os.tmpdir());
  } catch (e) {
    logger.error(appendMsg, 'error in getTempDirPath', e);
    return '';
  }
}

function format(num) {
  return (num < 10 ? '0' : '') + num;
};

/**
 * return timestamp in EG format
 * Example: 2018-04-12 13:09:37.528
 * @param {*} ts 
 * @returns date String - dd/MM/yyyy HH:mm:ss
 */
function formatDate(ts, sep) {
  sep = sep || '-';
  const now = ts ? new Date(ts) : new Date();

  return now.getFullYear() + sep +
    format(now.getMonth() + 1) + sep +
    format(now.getDate()) +
    ' ' + format(now.getHours()) +
    ':' + format(now.getMinutes()) +
    ':' + format(now.getSeconds()) +
    '.' + format(now.getMilliseconds());
};

/**
 * this function will slice the trace line if it is exceeds maxTraceLine
 * @param {*} err 
 * @param {*} maxTraceLine 
 * @returns 
 */
function spliceStackTrace(err, maxTraceLine) {
  if (!err || !err.stack) return err;

  if (maxTraceLine > -1) {
    const stack = err.stack.split('\n');
    if (stack.length > maxTraceLine) {
      err.stack = stack.slice(0, maxTraceLine);
      err.stack = err.stack.join('\n');
    }
  }

  return err;
}

/**
 * this will parse the black listed url pattern to regex
 * example: endsWith: new RegExp('^(/egurkha)'), endsWith: new RegExp('(.ttf|.otf)$')
 * @param {*} config 
 * @returns null
 */
function setBlackListedUrls(config) {
  config.black_listed_urls = null;
  const blackListedUrls = [];

  if (config.chars_to_exclude) {
    const chars = config.chars_to_exclude.split(",");
    if (chars.length) {
      chars.forEach(char => {
        blackListedUrls.push(`(${escapeSpecialChar(char)})`);
      });
    }
  }

  config.black_listed_urls = getRegexFromPattern(config.excluded_patterns, blackListedUrls);
}
exports.setBlackListedUrls = setBlackListedUrls;

function getRegexFromPattern(stringPattern, blackListedUrls) {
  blackListedUrls = blackListedUrls || [];
  if (!stringPattern) {
    return blackListedUrls.length ? new RegExp(blackListedUrls.join("|")) : null;
  }

  const patterns = stringPattern.split(",");
  if (!patterns.length) return;
  let endsWith = [];
  let contains = [];
  let startsWith = [];

  patterns.forEach(pattern => {
    if (pattern === '*') {
      blackListedUrls.push(`(.*?)`);
      return;
    }

    if (pattern[0] === '*' && pattern[pattern.length - 1] === '*') {
      contains.push(pattern.substring(1, pattern.length - 1));
      return;
    }

    const splited = pattern.split("*");
    if (splited[0]) startsWith.push(splited[0]);
    else if (splited[1]) endsWith.push(splited[1]);
  });

  if (endsWith.length) {
    if (endsWith.indexOf(".js") > -1) endsWith.push(".js.map");
    if (endsWith.indexOf(".css") > -1) endsWith.push(".css.map");
    blackListedUrls.push(`(((.*)\\${endsWith.join("|(.*)\\")})$)`);
  }

  if (contains.length) blackListedUrls.push(`(^.*(${contains.join(`|`)}).*)`);
  if (startsWith.length) blackListedUrls.push(`(^(${startsWith.join(`|`)}))`);
  return new RegExp(blackListedUrls.join("|"));
}
exports.getRegexFromPattern = getRegexFromPattern;

function escapeSpecialChar(str) {
  str = str || '';
  return String(str).replace(/[-[\]{}()*+?,\\/^$|#\s]/g, '\\$&');
}

function isEmptyObj(obj) {
  return !obj || Object.keys(obj).length === 0;
}

function getStringToBytes(string) {
  return Buffer.byteLength(string, 'utf8');
}

function executeCmd(command, returnFirstLine) {
  return new Promise((resolve, reject) => {
    exec(command, function (error, stdout, stderr) {
      if (error || stderr) {
        reject(error || stderr);
      } else {
        if (returnFirstLine && stdout) {
          const out = stdout.toString().split(/\r?\n/)[0];
          resolve(out && out.toString().trim() || "");
        } else resolve(stdout);
      }
    });
  });
}

function isValidIPaddress(ipaddress) {
  if (/^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/.test(ipaddress)) {
    return true;
  }
  return false;
}

function readFile(filePath) {
  try {
    const data = fs.readFileSync(filePath, 'utf-8');
    return data && data.toString().trim() || "";
  } catch (e) {
    logger.error(appendMsg, 'readFile error', e);
  }
}

function streamToString(stream) {
  const chunks = [];
  return new Promise((resolve, reject) => {
    stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)));
    stream.on('error', (err) => reject(err));
    stream.on('end', () => resolve(Buffer.concat(chunks).toString('utf8')));
  })
}

function _roundOff(no) {
  return Number(no.toFixed(2));
}
exports.roundOff = _roundOff

exports.setEventLoopUtilization = function (elu, stats) {
  stats.idleTime = _roundOff(elu.idle); // milliseconds
  stats.activeTime = _roundOff(elu.active);
  stats.utilization = _roundOff(elu.utilization * 100);
}

exports.copyObjectField = function (obj, keys) {
  if (!obj || !keys || !keys.length) return;

  const temp = {};
  keys.forEach(key => {
    temp[key] = obj[key];
  });
  return temp;
}

exports.getValueFromObj = function (obj, key) {
  if (!obj || !key) return;
  try {
    let objCopy = Object.assign({}, obj);
    const splitJsVariableArray = key.split('.');
    for (let i = 0; i < splitJsVariableArray.length; i++) {
      objCopy = objCopy[splitJsVariableArray[i]];
      if (objCopy === undefined) break;
    }
    if (objCopy === 'undefined') return '';
    if (typeof objCopy === 'string') return objCopy || '';
  } catch (error) {
    logger.error(appendMsg, 'getValueFromObj', error);
  }
  return '';
}

exports.readFile = readFile;
exports.isEmptyObj = isEmptyObj;
exports.formatDate = formatDate;
exports.executeCmd = executeCmd;
exports.getTimezone = getTimezone;
exports.getUtcOffset = getUtcOffset;
exports.getTempDirPath = getTempDirPath;
exports.streamToString = streamToString;
exports.spliceStackTrace = spliceStackTrace;
exports.getStringToBytes = getStringToBytes;
exports.isValidIPaddress = isValidIPaddress;
