import { eG } from "../eg-core-client"
import { eGReactNative } from "../eg-react-native";

export interface AsyncStorageOptions {
  ignore?: string[]
}

const PLUGIN_DEFAULTS: AsyncStorageOptions = {
  ignore: [],
}

export default <eGSubtype = eGReactNative>(options: AsyncStorageOptions) => (eg: eG<eGSubtype> & eGReactNative) => {
  // setup configuration
  const config = Object.assign({}, PLUGIN_DEFAULTS, options || {})
  const ignore = config.ignore || PLUGIN_DEFAULTS.ignore

  let swizzSetItem
  let swizzRemoveItem
  let swizzMergeItem
  let swizzClear
  let swizzMultiSet
  let swizzMultiRemove
  let swizzMultiMerge
  let isSwizzled = false

  const sendToEG = (action: string, data?: any) => {
    console.log("action:"+action+", data:"+data)
    // eg.send("asyncStorage.mutation", { action, data })
  }

  const setItem = async (key, value, callback) => {
    try {
      if (ignore.indexOf(key) < 0) {
        sendToEG("setItem", { key, value })
      }
    } catch (e) {}
    return swizzSetItem(key, value, callback)
  }

  const removeItem = async (key, callback) => {
    try {
      if (ignore.indexOf(key) < 0) {
        sendToEG("removeItem", { key })
      }
    } catch (e) {}
    return swizzRemoveItem(key, callback)
  }

  const mergeItem = async (key, value, callback) => {
    try {
      if (ignore.indexOf(key) < 0) {
        sendToEG("mergeItem", { key, value })
      }
    } catch (e) {}
    return swizzMergeItem(key, value, callback)
  }

  const clear = async callback => {
    try {
      sendToEG("clear")
    } catch (e) {}
    return swizzClear(callback)
  }

  const multiSet = async (pairs, callback) => {
    try {
      const shippablePairs = (pairs || []).filter(
        pair => pair && pair[0] && ignore.indexOf(pair[0]) < 0
      )
      if (shippablePairs.length > 0) {
        sendToEG("multiSet", { pairs: shippablePairs })
      }
    } catch (e) {}
    return swizzMultiSet(pairs, callback)
  }

  const multiRemove = async (keys, callback) => {
    try {
      const shippableKeys = (keys || []).filter(key => ignore.indexOf(key) < 0)
      if (shippableKeys.length > 0) {
        sendToEG("multiRemove", { keys: shippableKeys })
      }
    } catch (e) {}
    return swizzMultiRemove(keys, callback)
  }

  const multiMerge = async (pairs, callback) => {
    try {
      const shippablePairs = (pairs || []).filter(
        pair => pair && pair[0] && ignore.indexOf(pair[0]) < 0
      )
      if (shippablePairs.length > 0) {
        sendToEG("multiMerge", { pairs: shippablePairs })
      }
    } catch (e) {}
    return swizzMultiMerge(pairs, callback)
  }

  /**
   * Hijacks the AsyncStorage API.
   */
  const trackAsyncStorage = () => {
    if (isSwizzled) return

    swizzSetItem = eg.asyncStorageHandler.setItem
    eg.asyncStorageHandler.setItem = setItem

    swizzRemoveItem = eg.asyncStorageHandler.removeItem
    eg.asyncStorageHandler.removeItem = removeItem

    swizzMergeItem = eg.asyncStorageHandler.mergeItem
    eg.asyncStorageHandler.mergeItem = mergeItem

    swizzClear = eg.asyncStorageHandler.clear
    eg.asyncStorageHandler.clear = clear

    swizzMultiSet = eg.asyncStorageHandler.multiSet
    eg.asyncStorageHandler.multiSet = multiSet

    swizzMultiRemove = eg.asyncStorageHandler.multiRemove
    eg.asyncStorageHandler.multiRemove = multiRemove

    swizzMultiMerge = eg.asyncStorageHandler.multiMerge
    eg.asyncStorageHandler.multiMerge = multiMerge

    isSwizzled = true
  }

  const untrackAsyncStorage = () => {
    if (!isSwizzled) return

    eg.asyncStorageHandler.setItem = swizzSetItem
    eg.asyncStorageHandler.removeItem = swizzRemoveItem
    eg.asyncStorageHandler.mergeItem = swizzMergeItem
    eg.asyncStorageHandler.clear = swizzClear
    eg.asyncStorageHandler.multiSet = swizzMultiSet
    eg.asyncStorageHandler.multiRemove = swizzMultiRemove
    eg.asyncStorageHandler.multiMerge = swizzMultiMerge

    isSwizzled = false
  }

  // egShipStorageValues()
  if (eg.asyncStorageHandler) {
    trackAsyncStorage()
  }

  return {
    features: {
      trackAsyncStorage,
      untrackAsyncStorage,
    },
  }
}
