import * as WebSocket from "ws"
import validate from "./validate"
import logger from "./plugins/logger"
import image from "./plugins/image"
import benchmark from "./plugins/benchmark"
import stateResponses from "./plugins/state-responses"
import apiResponse from "./plugins/api-response"
import clear from "./plugins/clear"
import repl from "./plugins/repl"
import serialize from "./serialize"
import { start } from "./stopwatch"
import { ClientOptions } from "./client-options"

import {NativeModules} from 'react-native';
import DeviceInfo from './eg-react-device-info';

export const corePlugins = [
  image(),
  logger(),
  benchmark(),
  stateResponses(),
  apiResponse(),
  clear(),
  repl(),
]

const DEFAULT_OPTIONS: ClientOptions = {
  appToken : "",
  createSocket: null,
  host: "1.1.1.1",
  port: 0,
  loggingEnabled: true,
  logLevel: 1,
  name: "@eginnovations/react-native-eg-agent",
  secure: false,
  plugins: corePlugins,
  safeRecursion: true,
  onCommand: () => null,
  onConnect: () => null,
  onDisconnect: () => null,
}

// these are not for you.
const reservedFeatures = [
  "options",
  "connected",
  "socket",
  "plugins",
  "configure",
  "connect",
  "send",
  "use",
  "startTimer",
  "setUserName",
  "setAttribute"
]
const isReservedFeature = (value: string) => reservedFeatures.some(res => res === value)

export enum ArgType {
  String = "string",
}

export interface CustomCommandArg {
  name: string
  type: ArgType
}

export interface CustomCommand {
  id?: number
  command: string
  handler: (args?: any) => void

  title?: string
  description?: string
  args?: CustomCommandArg[]
}

export interface eGCore {
  setUserName: (name: string) => void
  setAttribute: (name: string, value: string) => void
 
  startTimer: () => () => number
  close: () => void
  send: (type: any, payload?: any, important?: boolean) => void
  display: (config?: any) => void
  reportError: (this: any, error: any) => void
  onCustomCommand: (config: CustomCommand | string, optHandler?: () => void) => () => void

  /* Provided by plugins */
  // API Response Plugin
  apiResponse?: (request: any, response: any, duration: any) => void
  
  // API Response Plugin
  // apiResponse?: (request: any, response: any, duration: any, host: any) => void

  // Benchmark Plugin
  benchmark?: (
    title: string
  ) => {
    step: (stepName: string) => void
    stop: (stopTitle: string) => void
    last: (stopTitle: string) => void
  }

  // Clear Plugin
  clear?: () => void

  // Image Plugin
  image?: (options: {
    uri: any
    preview: any
    filename: any
    width: any
    height: any
    caption: any
  }) => void

  // Logger Plugin
  log?: (...args: any[]) => void
  logImportant?: (...args: any[]) => void
  debug?: (message: any, important?: boolean) => void
  warn?: (message: any) => void
  error?: (message: any, stack: any) => void

  // State Plugin
  stateActionComplete?: (name: any, action: any, important?: boolean) => void
  stateValuesResponse?: (path: any, value: any, valid?: boolean) => void
  stateKeysResponse?: (path: any, keys: any, valid?: boolean) => void
  stateValuesChange?: (changes: any) => void
  stateBackupResponse?: (state: any) => void

  // REPL Plugin
  repl?: (name: string, value: object | Function | string | number) => void

  
  host: any

}

export interface eG<eGSubtype = eGCore> extends eGCore {
  /**
   * Set the configuration options.
   */
  configure: (options?: ClientOptions) => eG<eGSubtype> & eGSubtype

  use: (
    pluginCreator?: (client: eG<eGSubtype> & eGSubtype) => any
  ) => eG<eGSubtype> & eGSubtype

  connect: () => eG<eGSubtype> & eGSubtype

  setUserName: (name: string) => void
  setAttribute: (name: string, value : string) => void
}

export class eGImpl<eGSubtype = eGCore>
  implements eG<eGSubtype> {
  // the configuration options
  options: ClientOptions = Object.assign({}, DEFAULT_OPTIONS)

  /**
   * Are we connected to a server?
   */
  connected = false

  /**
   * The socket we're using.
   */
  socket: WebSocket = null

  /**
   * Available plugins.
   */
  plugins: any[] = []

  /**
   * Messages that need to be sent.
   */
  sendQueue: any[] = []

  /**
   * Are we ready to start communicating?
   */
  isReady = false

  /**
   * The last time we sent a message.
   */
  lastMessageDate = new Date()

  /**
   * The registered custom commands
   */
  customCommands: CustomCommand[] = []

  /**
   * The current ID for custom commands
   */
  customCommandLatestId = 1

  /**
   * Starts a timer and returns a function you can call to stop it and return the elapsed time.
   */
  startTimer = () => start()

  host: any

  deviceInfo: any

  appData: any

  resourceData: any

  userData: any

  /**
   * Set the user name.
   */
  setUserName(name : string) {
    NativeModules.eGAgent.setUserName(name);
  }

  setAttribute(name: string, value : string) {
    NativeModules.eGAgent.setUserAttribute(name, value);
  }

  /**
   * Set the configuration options.
   */
  configure(options: ClientOptions = {}): eG<eGSubtype> & eGSubtype {
    // options get merged & validated before getting set
    const newOptions = Object.assign({}, this.options, options)
    validate(newOptions)
    this.options = newOptions
    const {
      appToken,
      secure,
      host,
      loggingEnabled,
      logLevel,
      port,
    } = this.options

    	
  //  console.log("appToken::"+appToken);
  
    
  //  console.log("configure::$host::${host}"+host)

    const protocol = secure ? "https" : "http"
    // this.host = `${protocol}://${host}:${port}/` 
    if (host.startsWith("http")) {
      this.host = host + "/rumcollector/servlet/eGMobileRumCollector";
  } else {
      this.host = protocol + "://" + host + ":" + port + "/rumcollector/servlet/eGMobileRumCollector";
  }
  
 
  console.log("configure:host.toString(): "+host.toString())
  // console.log("configure:host.toString().indexOf('1.1.1.1'): "+host.toString().indexOf('1.1.1.1'))
  try {
      if (host.toString().indexOf('1.1.1.1') == -1) {
          // console.log("host URL ..."+this.host)
          if (host.startsWith("http")) {
              NativeModules.eGAgent.start({ appToken: appToken, hostUrl:  host , loggingEnabled: loggingEnabled, useSsl: secure, logLevel: logLevel }, function () { return console.log('EG Agent  Initialized'); });
          } else {
              NativeModules.eGAgent.start({ appToken: appToken, hostUrl: protocol + "://" + host + ":" + port + "/", loggingEnabled: loggingEnabled, useSsl: secure, logLevel: logLevel }, function () { return console.log('EG Agent  Initialized'); });
          }
        
          this.isReady = true;
      }
  } catch (e) {
        console.log("eg agent start::::"+e)
        console.error("Error:"+e);
      }
    // if we have plugins, let's add them here
    if (Array.isArray(this.options.plugins)) {
      this.options.plugins.forEach(p => this.use(p))
    }

    return (this as any) as eG<eGSubtype> & eGSubtype // cast needed to allow patching by other implementations like eG-react-native
  }

  close() {
    this.connected = false
    // this.socket && this.socket.close && this.socket.close()
  }

  
  /**
   * Connect to the eg-manager server.
   */
  connect(): eG<eGSubtype> & eGSubtype {
    this.connected = true
    this.isReady = true
    this.deviceInfo = NativeModules.eGAgent.getDevInfo();
    this.appData =NativeModules.eGAgent.getAppinfo();
    this.resourceData = NativeModules.eGAgent.getResInfo();
    this.userData = NativeModules.eGAgent.getUserInfo();
    console.log('connected...')

    return (this as any) as eG<eGSubtype> & eGSubtype // cast needed to allow patching by other implementations like eg-react-native
  }

  
  /**
   * Sends a command to the server
   */
  send = (type, data, important = false) => {
    // set the timing info
    console.log("send called to agent url..." + this.host)
    const date = new Date()
    let deltaTime = date.getTime() - this.lastMessageDate.getTime()
    // glitches in the matrix
    if (deltaTime < 0) {
      deltaTime = 0
    }
    this.lastMessageDate = date
    const {
      appToken
    } = this.options
   
    let instrumentationType = type;
    let appName = DeviceInfo.getApplicationName();
    let activityName = DeviceInfo.getCurrentActivityNameSync();
    let osName =DeviceInfo.getBaseOsSync();
    let osVersion = DeviceInfo.getSystemVersion();
    let deviceName =  DeviceInfo.getDeviceNameSync()
    // this.deviceInfo = NativeModules.eGAgent.getDevInfo();
    // this.appData =NativeModules.eGAgent.getAppinfo();
    // this.resourceData = NativeModules.eGAgent.getResInfo();
    
    let deviceInformation = JSON.parse(this.deviceInfo);
    let applicationInformation = JSON.parse(this.appData);
    let resourceInformation = JSON.parse(this.resourceData);
    let userInformation = JSON.parse(this.userData);

    let executionTime = deltaTime;
    let country = DeviceInfo.getUserCountrySync();

    let fullMessage = {};
    console.log("send called type..." + type)
    if (type == 'HTTP') {
      let httpInformation = data;
      fullMessage = {
        appName,
        instrumentationType,
        appToken:appToken,
        activityName,
        osName,
        osVersion,
        deviceName,
        deviceInformation,
        applicationInformation,
        resourceInformation,
        httpInformation,
        userInformation,
        executionTime,
        country
    }
  } else if (type == 'Page') {
    let pageInformation = data;
    fullMessage = {
      appName,
      instrumentationType,
      appToken:appToken,
      activityName,
      osName,
      osVersion,
      deviceName,
      deviceInformation,
      applicationInformation,
      resourceInformation,
      pageInformation,
      userInformation,
      executionTime,
      country
  }
} else if (type == 'JSError') {
    let message = data
    console.log("sending serialized message .."+message)
    fullMessage = {
      appName,
      instrumentationType,
      activityName,
      deviceName,
      osName,
      osVersion,
      appToken:appToken,
      deviceInformation,
      applicationInformation,
      resourceInformation,
      userInformation,
      errordetails:message,
      executionTime,
      country
  }
}
console.log("sending serialized data .."+serialize(data, this.options.proxyHack))
    
    console.log('url in send is.... '+this.host)
    
    let serializedMessage = serialize(fullMessage, this.options.proxyHack)
    console.log("sending serializedMessage.."+serializedMessage)
    console.log("this.isready .. "+this.isReady)
    if (this.isReady) {
      // send this command
      try {

        send(serializedMessage, this.host)
        while (this.sendQueue.length > 0) {
          const h = this.sendQueue[0]
          this.sendQueue = this.sendQueue.slice(1)
          send(h, this.host)
        }
      
      } catch(e) {
        this.isReady = false
        console.log("An error occured communicating with eG. Please reload your app", e)
      }
    } else {
      // queue it up until we can connect
      this.sendQueue.push(serializedMessage)
    }
   
  }


  /**
   * Sends a custom command to the server to displays nicely.
   */
  display(config: any = {}) {
    const { name, value, preview, image: img, important = false } = config
    const payload = {
      name,
      value: value || null,
      preview: preview || null,
      image: img || null,
    }
    this.send("display", payload, important)
  }

  /**
   * Client libraries can hijack this to report errors.
   */
  reportError(this: any, error) {
    this.error(error)
  }

  /**
   * Adds a plugin to the system
   */
  use(
    pluginCreator?: (client: eG<eGSubtype> & eGSubtype) => any
  ): eG<eGSubtype> & eGSubtype {
    // we're supposed to be given a function
    if (typeof pluginCreator !== "function") {
      throw new Error("plugins must be a function")
    }

    // execute it immediately passing the send function
    const plugin = pluginCreator.bind(this)(this)

    // ensure we get an Object-like creature back
    if (typeof plugin !== "object") {
      throw new Error("plugins must return an object")
    }

    // do we have features to mixin?
    if (plugin.features) {
      // validate
      if (typeof plugin.features !== "object") {
        throw new Error("features must be an object")
      }

      // here's how we're going to inject these in
      const inject = (key: string) => {
        // grab the function
        const featureFunction = plugin.features[key]

        // only functions may pass
        if (typeof featureFunction !== "function") {
          throw new Error(`feature ${key} is not a function`)
        }

        // ditch reserved names
        if (isReservedFeature(key)) {
          throw new Error(`feature ${key} is a reserved name`)
        }

        // ok, let's glue it up... and lose all respect from elite JS champions.
        this[key] = featureFunction
      }

      // let's inject
      Object.keys(plugin.features).forEach(key => inject(key))
    }

    // add it to the list
    this.plugins.push(plugin)

    // call the plugins onPlugin
    plugin.onPlugin && typeof plugin.onPlugin === "function" && plugin.onPlugin.bind(this)(this)

    // chain-friendly
    return (this as any) as eG<eGSubtype> & eGSubtype // cast needed to allow patching by other implementations like eG-react-native
  }

  onCustomCommand(config: CustomCommand | string, optHandler?: () => void): () => void {
    let command: string
    let handler: () => void
    let title: string
    let description: string
    let args: CustomCommandArg[]

    if (typeof config === "string") {
      command = config
      handler = optHandler
    } else {
      command = config.command
      handler = config.handler

      title = config.title
      description = config.description
      args = config.args
    }

    // Validations
    // Make sure there is a command
    if (!command) {
      throw new Error("A command is required")
    }

    // Make sure there is a handler
    if (!handler) {
      throw new Error(`A handler is required for command "${command}"`)
    }

    // Make sure the command doesn't already exist
    const existingCommands = this.customCommands.filter(cc => cc.command === command)
    if (existingCommands.length > 0) {
      existingCommands.forEach(command => {
        console.log(command)
        this.customCommands = this.customCommands.filter(cc => cc.id !== command.id)

        this.send("customCommand.unregister", {
          id: command.id,
          command: command.command,
        })
      })
    }

    if (args) {
      const argNames = []

      args.forEach(arg => {
        if (!arg.name) {
          throw new Error(`A arg on the command "${command}" is missing a name`)
        }

        if (argNames.indexOf(arg.name) > -1) {
          throw new Error(
            `A arg with the name "${arg.name}" already exists in the command "${command}"`
          )
        }

        argNames.push(arg.name)
      })
    }

    // Create this command handlers object
    const customHandler: CustomCommand = {
      id: this.customCommandLatestId,
      command,
      handler,
      title,
      description,
      args,
    }

    // Increment our id counter
    this.customCommandLatestId += 1

    // Add it to our array
    this.customCommands.push(customHandler)

    this.send("customCommand.register", {
      id: customHandler.id,
      command: customHandler.command,
      title: customHandler.title,
      description: customHandler.description,
      args: customHandler.args,
    })

    return () => {
      this.customCommands = this.customCommands.filter(cc => cc.id !== customHandler.id)

      this.send("customCommand.unregister", {
        id: customHandler.id,
        command: customHandler.command,
      })
    }
  }
}

// convenience factory function
export function createClient<eGSubtype = eGCore>(
  options?: ClientOptions
): eG<eGSubtype> & eGSubtype {
  const client = new eGImpl<eGSubtype>()
  client.configure(options)
  return (client as any) as eG<eGSubtype> & eGSubtype // cast needed to allow patching by other implementations like eg-react-native
}

function send(serializedMessage, host) {
  var myHeaders = new Headers();
        myHeaders.append("Content-Type", "application/json");
          var requestOptions = {
            method: 'POST',
            headers: myHeaders,
            body: serializedMessage,
            // redirect: 'follow'
          };
          console.log("JSON::"+serializedMessage)
          fetch(host, requestOptions)
          .then(function (response) { var text =response.text(); console.log("text::"+text); return text; })
          .then(function (result) {  console.log("result::"+result); return result;})
          .catch(function (error) {  console.log('error', error); return error;});
}


