/** @module Android */
var path = require("path");
var eg = require("../eginnovations");

/**
 * @desc Android object to add New Relic to the gradle tasks.
 * @type {{gradleBuildPath: string,
 * pluginConfig: (*),
 * nrTag: string,
 * slurpRootLevelGradle: module.exports.slurpRootLevelGradle,
 * writeRootLevelGradle: module.exports.writeRootLevelGradle,
 * injectAgentPlugin: module.exports.injectAgentPlugin,
 * injecteGInnovationsProperties: module.exports.injecteGInnovationsProperties,
 * removeAgentPlugin: module.exports.removeAgentPlugin}}
 */
module.exports = {

  /**
   * gradle build paths
   */
  gradleBuildPath: path.join("platforms", "android", "app", "build.gradle"),
  
  gradleRootBuildPath: path.join("platforms", "android",  "build.gradle"),

  /**
   * Android Plugin Config
   */
  pluginConfig: eg.getAndroidConfig(),

  /**
   * eG Innovations Gradle plugin invocation
   */
  egGradlePluginTag: "apply plugin: 'eg'\n",

  
  
  /**
   * Agent build extension
   */
  /*egTag: "\n// EG INNOVATIONS ADDED\n" + "buildscript { \t\n\trepositories	 {\n\t\t mavenLocal()\n\t\t mavenCentral()\n\t"+  "}\n\tdependencies {\n\t\t classpath 'com.eginnovations.agent.android:agent-gradle-plugin:{AGENT_VER}'\n\t\t classpath "+ "'com.eginnovations.agent.android:android-agent:{AGENT_VER}'\n\t}\n}\n"+
  + "{PLUGIN}" +
  "// EG INNOVATIONS ADDED\n",
*/
egTag: "\n// EG INNOVATIONS ADDED\n" + "buildscript { \t\n\trepositories{ \n\tflatDir { \n\t\tdirs 'libs'\n\t"+  "}\n\t }\n\t  \n\tdependencies {\n\t\t classpath fileTree(include: ['*.jar'], dir: 'libs')\n\t}\n}\n"+
  "{PLUGIN}" +
  "// EG INNOVATIONS ADDED\n",
  
  egRootTag: "\n// EG INNOVATIONS ADDED\n" + "buildscript { \t\n\trepositories{ \n\tflatDir { \n\t\tdirs 'app/libs'\n\t"+  "}\n\t }\n\t  \n\tdependencies {\n\t\t classpath fileTree(include: ['*.jar'], dir: 'app/libs')\n\t}\n}\n"+
  "{PLUGIN}" +
  "// EG INNOVATIONS ADDED\n",

  /**
   * Helper method to read the root gradle file.
   * @returns {String} Root Gradle File
   */
  slurpRootLevelGradle: function () {
    return eg.slurpFile(this.gradleBuildPath);
  },
  
  slurpLevelGradle: function () {
    return eg.slurpFile(this.gradleRootBuildPath);
  },
  

  /**
   * Write contents to the gradle build path
   * @param {String} contents - The contents to be written to the gradle file.
   */
  writeRootLevelGradle: function (contents) {
    eg.writeFile(this.gradleBuildPath, contents);
  },
  
  writeLevelGradle: function (contents) {
    eg.writeFile(this.gradleRootBuildPath, contents);
  },

  /**
   * Inject the eG Innovationsc agent into root gradle
   */
  injectAgentPlugin: function (context, setEnabled) {
    var buildGradle = this.slurpRootLevelGradle() + this.egTag.replace(/{AGENT_VER}/g, this.pluginConfig.agentVer);
	var buildRootGradle = this.slurpLevelGradle() + this.egRootTag.replace(/{AGENT_VER}/g, this.pluginConfig.agentVer);
    buildGradle = buildGradle.replace(/{PLUGIN}/g, setEnabled? this.egGradlePluginTag: "");
	buildRootGradle = buildRootGradle.replace(/{PLUGIN}/g, "");
	this.writeRootLevelGradle(buildGradle);
    this.writeLevelGradle(buildRootGradle);
  },

  /**
   * Inject the eG Innovations App Token into the eG Innovations Properties file.
   */
  injecteGInnovationsProperties: function () {
    var appToken = this.pluginConfig.appToken;
    var eGInnovationsPropertiesPath = path.join(".", "eginnovations.properties");
    var eGInnovationsProperties = "com.eginnovations.application_token=" + appToken + "\n";
    eg.writeFile(eGInnovationsPropertiesPath, eGInnovationsProperties);
  },

  /**
   * Remove eG Innovations from gradle.
   **/
  removeAgentPlugin: function () {
    var buildScript = this.slurpRootLevelGradle();
    this.writeRootLevelGradle(buildScript.replace(/\n\/\/ EG INNOVATIONS ADDED[\s\S]*\/\/ EG INNOVATIONS ADDED\n/, ""));
	buildScript = this.slurpLevelGradle();
    this.writeLevelGradle(buildScript.replace(/\n\/\/ EG INNOVATIONS ADDED[\s\S]*\/\/ EG INNOVATIONS ADDED\n/, ""));
  },

  /**
   * Return {boolean} - if this platform exists and has been configured with an application token
   */
  isPlatformConfigured: function() {
    var config = eg.getAndroidConfig();
    return eg.isPlatformConfigured(config);
  },

  /**
   * Verify the platform has been installed prior to running this task
   * @param context {Object} - Cordova context
   */
  verifyPlatformInstall: function (context) {
    if (!this.isPlatformConfigured()) {
      eg.warnPlatformNotConfigured('Android');
    }
  }

};