import { Platform, NativeModules } from "react-native"
import { createClient, eG } from "./eg-core-client"

import getReactNativeVersion from "./helpers/getReactNativeVersion"
import getReactNativeDimensions from "./helpers/getReactNativeDimensions"
import asyncStorage, { AsyncStorageOptions } from "./plugins/asyncStorage"
import overlay from "./plugins/overlay"
import openInEditor, { OpenInEditorOptions } from "./plugins/openInEditor"
import trackGlobalErrors, { TrackGlobalErrorsOptions } from "./plugins/trackGlobalErrors"
import networking, { NetworkingOptions } from "./plugins/networking"
import storybook from "./plugins/storybook"
import devTools from "./plugins/devTools"
import ConnectionManager from "./connection-manager"

const constants = NativeModules.PlatformConstants || {}

let tempClientId = null

const DEFAULTS = {
	appToken: "",
    createSocket: function (path) { return new ConnectionManager(path); },
    host: "1.1.1.1",
    port: 1,
	name: "React Native App",
  environment: process.env.NODE_ENV || (__DEV__ ? "development" : "production"),
  client: {
    eGLibraryName: "@eginnovations/react-native-eg-agent",
    eGLibraryVersion: "REACT_NATIVE_EG-AGENT_VERSION",
    platform: Platform.OS,
    platformVersion: Platform.Version,
    osRelease: constants.Release,
    model: constants.Model,
    serverHost: constants.ServerHost,
    forceTouch: constants.forceTouchAvailable || false,
    interfaceIdiom: constants.interfaceIdiom,
    systemName: constants.systemName,
    uiMode: constants.uiMode,
    serial: constants.Serial,
    androidId: constants.androidID,
    reactNativeVersion: getReactNativeVersion(),
    ...getReactNativeDimensions(),
  },
  /* eslint-disable @typescript-eslint/no-use-before-define */
  getClientId: () => {
    if (eGAgent.asyncStorageHandler) {
      return eGAgent.asyncStorageHandler.getItem(tempClientId)
    }

    return new Promise(resolve => resolve(tempClientId))
  },
  setClientId: (clientId) => {
    if (eGAgent.asyncStorageHandler) {
      return eGAgent.asyncStorageHandler.setItem(tempClientId, clientId)
    }

    tempClientId = clientId
    return Promise.resolve()
  },
  /* eslint-enable @typescript-eslint/no-use-before-define */
  proxyHack: true,
}

export interface UseReactNativeOptions {
  errors?: TrackGlobalErrorsOptions | boolean
  editor?: OpenInEditorOptions | boolean
  overlay?: boolean
  asyncStorage?: AsyncStorageOptions | boolean
  networking?: NetworkingOptions | boolean
  storybook?: boolean
  devTools?: boolean
}

export interface eGReactNative {
  useReactNative: (
    options?: UseReactNativeOptions
  ) => eG<eGReactNative> & eGReactNative
  overlay: (App: React.ReactNode) => void
  storybookSwitcher: (App: React.ReactNode) => (Root: React.ReactNode) => React.ReactNode
  asyncStorageHandler?: any
  setAsyncStorageHandler?: (asyncStorage: any) => eG<eGReactNative> & eGReactNative
}

const eGAgent: eG<eGReactNative> & eGReactNative = createClient(DEFAULTS)

function getPluginOptions<T>(options?: T | boolean): T {
  return typeof options === "object" ? options : null
}

eGAgent.useReactNative = (options: UseReactNativeOptions = {}) => {
  if (options.errors !== false) {
    eGAgent.use(trackGlobalErrors(getPluginOptions(options.errors)))
  }

  if (options.editor !== false) {
    eGAgent.use(openInEditor(getPluginOptions(options.editor)))
  }

  if (options.overlay !== false) {
    eGAgent.use(overlay())
  }

  if (options.asyncStorage !== false) {
    eGAgent.use(asyncStorage(getPluginOptions(options.asyncStorage)))
  }

  if (options.networking !== false) {
    eGAgent.use(networking(getPluginOptions(options.networking)))
  }

  if (options.storybook !== false) {
    eGAgent.use(storybook())
  }

  if (options.devTools !== false) {
    eGAgent.use(devTools())
  }

  return eGAgent
}

eGAgent.setAsyncStorageHandler = asyncStorage => {
  eGAgent.asyncStorageHandler = asyncStorage

  return eGAgent
}

export { asyncStorage, trackGlobalErrors, openInEditor, overlay, networking, storybook, devTools }

export default eGAgent