'use strict'

const semver = require('semver')

const httpShared = require('../http-shared')
const shimmer = require('../shimmer')
const logger = require('../../logger')
const appendMsg = 'Https:'

exports.start = function (https, agent, version, enabled) {
  if (!enabled) return https;
  logger.debug(appendMsg, 'wrapping https.Server.prototype.emit function');

  try {
    shimmer.wrap(https && https.Server && https.Server.prototype, 'emit', httpShared.instrumentRequest(agent, 'https'));

    // From Node.js v9.0.0 and onwards, https requests no longer just call the
    // http.request function. So to correctly instrument outgoing HTTPS requests
    // in all supported Node.js versions, we'll only only instrument the
    // https.request function if the Node version is v9.0.0 or above.
    //
    // This change was introduced in:
    // https://github.com/nodejs/node/commit/5118f3146643dc55e7e7bd3082d1de4d0e7d5426
    if (semver.gte(version, '9.0.0')) {
      logger.debug(appendMsg, 'wrapping https.request function')
      shimmer.wrap(https, 'request', httpShared.traceOutgoingRequest(agent, 'https', 'request'))

      logger.debug('wrapping https.get function')
      shimmer.wrap(https, 'get', httpShared.traceOutgoingRequest(agent, 'https', 'get'))
    }

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }
  return https;
}

exports.stop = function (https, version) {
  shimmer.unwrap(https && https.Server && https.Server.prototype, 'emit')

  if (semver.gte(version, '9.0.0')) {
    shimmer.unwrap(https, 'request');
    shimmer.unwrap(https, 'get');
  }
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}
