$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

$StoreServiceFeatureType = "5af80ac4-488f-43bb-a162-3cfef90ea5ff"
function GetFeatureClass($FeatureName)
{
    $featureClass = $controller.FeatureClasses | where { $_.Name -eq $FeatureName }
    if (-not $featureClass)
    {
        Write-Host "Unknown feature class: $FeatureName"
        return $null
    }

    return $featureClass
}

function GetMemberNames([string]$AppPath)
{
    $members =  Get-Item -Path "$AppPath"

    foreach($entry in $members.GetEnumerator())
    {
        $entry.Hostname
    }
}


function GetFeatureInstances([string]$featureName)
{
    $featureClass = $controller.FeatureClasses | where { $_.Name -eq $featureName }
    if (-not $featureClass)
    {
        Write-Host "Unknown feature class: $featureName"
        return $null
    }

    # Find "installed" instances of the class.
    $controller.FeatureInstances | where { $_.ClassType -eq $featureClass.Type }
}

function GetClusterManagementInstance()
{
    # Checking for installed ClusterManagement Class
    $clusterManagementFeature = $null
    $featureClass = GetFeatureClass $FeatureName
    if ($featureClass)
    {
        $clusterManagementFeature = GetFeatureInstances $FeatureName
    }
    $clusterManagementFeature
}

function RemoveDeliveryServicesDrive([string]$Name=("DS"))
{
    Remove-PSDrive -Name $Name -ErrorAction SilentlyContinue
}

function GetAppPath([string]$AppPath, [string]$Name=("DS"))
{
    RemoveDeliveryServicesDrive -Name $Name

    $root = $Name+':\'

    # Sink the output of the New-PSDrive to the unused $drive variabe
    # Needs to be global scope otherwise module does not see the drive!
    $drive = New-PSDrive -DSName "$AppPath" -Name $Name -Root $root -PSProvider 'DeliveryServices' -Scope Global

    # Return the new drive root
    return $root
}


function Get-DSClusterMembersName()
{
    if (!(GetClusterManagementInstance))
    {
        Write-Error "A cluster has not been created or joined, no members"
    }

    $drive = GetAppPath -AppPath $FeatureName

    $memberspath = "${drive}DeliveryServices\clusterInformation\cluster\members"

    $members =  Get-Item -Path $memberspath

    $memberNames = New-Object PSObject
    $memberNames | Add-Member -MemberType NoteProperty -Name "Hostnames" -Value @(GetMemberNames -AppPath $memberspath)

    Write-Output $memberNames
}

function GetParentInstance($FeatureInstance)
{
    # Get the parent
    if ($FeatureInstance.ParentInstance.Guid -eq [System.Guid]::Empty)
    {
        return $FeatureInstance
    }

    # Expect single parent instance
    $parentInstance = $controller.FeatureInstances | where { $_.Id -eq $FeatureInstance.ParentInstance.Guid }

    GetParentInstance -FeatureInstance $parentInstance
}

function GetParentInstances($FeatureInstances)
{
    foreach($instance in @($FeatureInstances))
    {
        GetParentInstance -FeatureInstance $instance
    }
}


function Get-DSStoreFeatureInstances()
{
    $instances = (Get-DSFrameworkController).FeatureInstances

    $featureInstances = $instances | where { $_.ClassType -eq $StoreServiceFeatureType }
    if (-not $featureInstances)
    {
        return $null
    }

    GetParentInstances -FeatureInstances $featureInstances
}

$dsInstallProp = Get-ItemProperty -Path HKLM:\SOFTWARE\Citrix\DeliveryServicesManagement -Name InstallDir
$dsInstallDir = $dsInstallProp.InstallDir

$snapinModule = Get-PSSnapin -Registered -Name "Citrix.DeliveryServices.Framework.Commands" -ErrorAction SilentlyContinue
$snapinFile = $snapinModule.ModuleName
Import-Module -name $snapinFile

$snapinModule = Get-PSSnapin -Registered -Name "Citrix.DeliveryServices.ConfigurationProvider" -ErrorAction SilentlyContinue
$snapinFile = $snapinModule.ModuleName
Import-Module -name $snapinFile

$snapinModule = Get-PSSnapin -Registered -Name "Citrix.DeliveryServices.Web.Commands" -ErrorAction SilentlyContinue
$snapinFile = $snapinModule.ModuleName
Import-Module -name $snapinFile

$controller = Get-DSFrameworkController

Write-Host "Server Group starts here"
$clusterAddress = Get-DSFrameworkProperty -Key HostBaseUrl
write-host("Servergroup Url=" + $clusterAddress)
$featureName = 'ClusterManagement'
$clusterManagementFeature = GetFeatureInstances $featureName
if ($clusterManagementFeature)
{
	$memberNames = (Get-DSClusterMembersName).Hostnames
}
else
{
	$memberNames = @()
}
$icount = $memberNames.Count
write-host ("Number of Servers=" + $icount)
foreach($name in $memberNames)
{
	write-host ("Server Name=" + $name)
}

Write-Host "Server Group ends here"
write-Host "Stores starts here"
$stores=Get-DSStoreFeatureInstances
if($stores -ne $null)
{
	foreach($store in $stores)
	{
		Write-Host("Name=" + $store.ServiceFriendlyName)
		Write-Host("Base Url=" + $store.ServiceBaseUrl)
		Write-Host("Site Id=" + $store.SiteId)
		Write-Host("Virtual Directory=" + $store.VirtualPath)
		Write-Host("ConfigLocation=" + $store.ConfigLocation)
	}
}
write-Host "Stores ends here"

