'use strict'

const eos = require('end-of-stream')

const shimmer = require('../shimmer');
const parser = require('../../parser');
const logger = require('../../logger');
const symbols = require('../../symbols');
const httpShared = require('../http-shared');
const { parseUrl } = require('../../parser');
const appConstant = require('../../app-constants');

const appendMsg = 'Http2:'
const eGGUID = appConstant.GUID_NAME;

exports.start = function (http2, agent, version, enabled) {
  if (!enabled) return http2;
  const ins = agent._instrumentation;

  try {
    logger.debug(appendMsg, 'shimming http2.createServer function');
    shimmer.wrap(http2, 'createServer', wrapCreateServer);
    shimmer.wrap(http2, 'createSecureServer', wrapCreateServer);

    logger.debug(appendMsg, 'shimming http2.connect function');
    shimmer.wrap(http2, 'connect', wrapConnect);

    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'Instrumentation error', e);
  }
  return http2;

  // The `createServer` function will unpatch itself after patching
  // the first server prototype it patches.
  function wrapCreateServer(original) {
    return function wrappedCreateServer(options, handler) {
      var server = original.apply(this, arguments)
      shimmer.wrap(server.constructor.prototype, 'emit', wrapEmit);
      logger.debug(appendMsg, 'intercepted request event call to http2.Server.prototype.emit')
      wrappedCreateServer[symbols.unwrap]()
      return server
    }
  }

  function wrapEmit(original) {
    let patched = false;
    return function wrappedEmit(event, stream, headers) {
      if (event === 'stream') {
        if (!patched) {
          patched = true;
          const proto = stream.constructor.prototype;
          shimmer.wrap(proto, 'pushStream', wrapPushStream);
          shimmer.wrap(proto, 'respondWithFile', wrapRespondWith);
          shimmer.wrap(proto, 'respondWithFD', wrapRespondWith);
          shimmer.wrap(proto, 'respond', wrapHeaders);
        }

        const req = {};
        req.url = headers[':path'];
        logger.debug(appendMsg, 'New incoming http2 request call url:', req.url);

        if (httpShared.isUrlExcluded(agent, req.url)) {
          logger.debug(appendMsg, 'ignoring blacklisted request to url:', req.url);
          ignoreTheRequest();
        } else {
          req.method = headers[':method'];
          req.headers = Object.assign({}, headers);
          req.connection = {};
          const option = httpShared.getRequestIdentifier(req, agent);
          option.method = req.method;
          const trans = agent.startTransaction(req.url, 'http2:inbound', option);

          if (trans) {
            ins.bindEmitter(stream)

            eos(stream, function () {
              logger.debug(appendMsg, "call ended with status code:", trans.requestIdentifier.statusCode, 'id:', trans.id);
              trans.userContext = parser.getUserContextFromReq(req);
              trans.end();
            });
          } else {
            ignoreTheRequest();
          }
        }
      }

      return original.apply(this, arguments)
    }

    function ignoreTheRequest() {
      // don't leak previous transaction
      agent._instrumentation.supersedeWithEmptyRunContext();
      // for testing perpose, we are calling the fn. don't remove this.
      if (agent._externalTransport) agent._externalTransport();
    }
  }

  function updateHeaders(headers) {
    const trans = agent.currTransaction();
    if (trans) {
      trans.requestIdentifier.statusCode = headers[':status'] || 200;
    }
  }

  function wrapHeaders(original) {
    return function (headers) {
      updateHeaders(headers)
      return original.apply(this, arguments)
    }
  }

  function wrapRespondWith(original) {
    return function (_, headers) {
      updateHeaders(headers)
      return original.apply(this, arguments)
    }
  }

  function wrapPushStream(original) {
    return function wrappedPushStream(...args) {
      // Note: Break the run context so that the wrapped `stream.respond` et al
      // for this pushStream do not overwrite outer transaction state.
      const callback = args.pop();
      args.push(agent._instrumentation.bindFunctionToEmptyRunContext(callback));
      return original.apply(this, args);
    }
  }

  function wrapConnect(orig) {
    return function (host) {
      const ret = orig.apply(this, arguments)
      shimmer.wrap(ret, 'request', orig => wrapRequest(orig, host))
      return ret
    }
  }

  function wrapRequest(orig, host) {
    return function (headers) {
      const urlObj = parseUrl(headers[':path']);
      const uri = host + urlObj.pathname;
      const method = headers[':method'] || 'GET';
      const span = agent.startSpan(method + ' ' + uri, 'http:outbound', 'Http2');
      const id = span && span.transaction.id;
      logger.debug(appendMsg, 'intercepted call to http2.request, id:', id);
      headers[eGGUID] = span.getNeweGGUID();
      const options = {};
      options.uri = uri;
      options.method = method;
      options.nodeOrder = span.nodeOrder;
      span.options = options;

      const req = orig.apply(this, arguments);
      if (!span) return req;

      ins.bindEmitter(req);
      req.on('response', (resHeaders) => {
        span.options.statusCode = resHeaders[':status'] || null;
      });

      req.on('error', function (err) {
        if (err) {
          logger.debug(appendMsg, 'error captured');
          span.captureError(err);
        }

        span.end();
      });

      req.on('end', () => {
        logger.debug(appendMsg, 'http outbound request event is completed', {
          id: id,
          url: span.options.uri
        })
        span.end();
      });


      return req
    }
  }
}

exports.stop = function (http2, version) {
  shimmer.unwrap(http2, 'createServer');
  shimmer.unwrap(http2, 'createSecureServer');
  shimmer.unwrap(http2, 'connect');
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}