'use strict'

const url = require('url');
const semver = require('semver');

const shimmer = require('../../shimmer');
const logger = require('../../../logger');
const symbols = require('../../../symbols');

let isWrapped = false;
const appendMsg = `RabbitMQ-amqplib-connect:`;

exports.start = function (amqplib, agent, version, enabled) {
  if (!enabled) return amqplib;
  if (!semver.satisfies(version, '>=0.5')) {
    logger.info(appendMsg, `version ${version} is not supported`);
    return amqplib;
  }

  try {
    logger.debug(appendMsg, 'wrapping amqplib.connect');
    shimmer.wrap(amqplib, 'connect', wrapConnect);
    logger.info(appendMsg, 'Wrapped successfully..!, Version', version);
  } catch (e) {
    logger.error(appendMsg, 'wrapChannel', e);
  }

  isWrapped = true;
  return amqplib;

  function wrapConnect(originalFn) {
    return function wrappedConnect(urlParam) {
      let result = null;
      let urlParsed = null;
      const cb = arguments[2];
      urlParam = urlParam || 'amqp://localhost';

      try {
        if (typeof urlParam === 'object') {
          urlParsed = urlParam;
        } else {
          if (typeof urlParam !== 'string') urlParam = 'amqp://localhost';
          urlParsed = url.parse(urlParam);
        }
        urlParsed.protocol = urlParsed.protocol.replace(':', '');
        urlParsed.port = urlParsed.port || ((urlParsed.protocol === 'amqp') ? 5672 : 5671);
        arguments[2] = wrapConnectCallback(cb);
      } catch (e) {
        logger.error(appendMsg, 'wrapConnect', e);
      }

      try {
        result = originalFn.apply(this, arguments);
      } catch (err) {
        logger.debug(appendMsg, 'error captured', err.message);
        if (span) span.captureError(err);
      }
      return result;

      function wrapConnectCallback(cb) {
        return function wrappedConnectCallback(err, conn) {
          if (err) agent.captureError(err);
          if (conn) {
            Object.defineProperty(conn, symbols.rabbitMQConnectionDetails, {
              value: urlParsed,
              enumerable: false,
            });
          }
          return cb.apply(this, arguments);
        }
      }
    }
  }
}

exports.stop = function (amqplib, version) {
  if (!isWrapped) return;
  try {
    shimmer.unwrap(amqplib, 'connect');
  } catch (e) {
    logger.error(appendMsg, 'stop', e);
  }
  isWrapped = false;
  logger.info(appendMsg, 'unwrapped successfully..!, Version', version);
}