<#
    Retrieve Citrix Studio logs
#>
[CmdletBinding()]
Param
(
    [string]$ddc = 'localhost',
    [Parameter(Mandatory=$true, ParameterSetName = "TimeSpan")]
    [datetime]$start ,
    [Parameter(Mandatory=$false, ParameterSetName = "TimeSpan")]
    [datetime]$end = [datetime]::Now ,
    [Parameter(Mandatory=$true, ParameterSetName = "Last")]
    [string]$last,
    [int]$maxRecordCount = 5000
)

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}
 
if( ! [string]::IsNullOrEmpty( $last ) )
{

    [long]$multiplier = 0
    switch( $last[-1] )
    {
        "s" { $multiplier = 1 }
        "m" { $multiplier = 60 }
        "h" { $multiplier = 3600 }
        "d" { $multiplier = 86400 }
        "w" { $multiplier = 86400 * 7 }
        "y" { $multiplier = 86400 * 365 }
        default { Throw "Unknown multiplier `"$($last[-1])`"" }
    }
    if( $last.Length -le 1 )
    {
        $start = $end.AddHours( -$multiplier )
    }
    else
    {
        $start = $end.AddSeconds( - ( ( $last.Substring( 0 ,$last.Length - 1 ) -as [long] ) * $multiplier ) )
    }
}
elseif( ! $PSBoundParameters[ 'start' ] )
{
    $start = (Get-Date).AddDays( -7 )
}
Add-PSSnapin -Name 'Citrix.ConfigurationLogging.Admin.*'
if( ! ( Get-Command -Name 'Get-LogHighLevelOperation' -ErrorAction SilentlyContinue ) )
{
    Throw "Unable to find the Citrix Get-LogHighLevelOperation cmdlet required"
} 
[hashtable]$queryparams = @{
    'AdminAddress' = $ddc
    'SortBy' = '-StartTime'
    'MaxRecordCount' = $maxRecordCount
    'ReturnTotalRecordCount' = $true
}
$RecordCount = $null 
[array]$results = @( Get-LogHighLevelOperation -Filter { StartTime -ge $start -and EndTime -le $end }  @queryparams -ErrorAction SilentlyContinue -ErrorVariable RecordCount | ForEach-Object -Process `
{
    $result = [pscustomobject][ordered]@{
        'Started' = $_.StartTime
        'Duration (s)' = [math]::Round( (New-TimeSpan -Start $_.StartTime -End $_.EndTime).TotalSeconds , 2 )
        'User' = $_.User
        'From' = $_.AdminMachineIP
        'Operation' = $_.text
        'Source' = $_.Source
        'Type' = $_.OperationType
        'Targets' = $_.TargetTypes -join ','
        'Successful' = $_.IsSuccessful
} 
    $result
})
if( $recordCount -and $recordCount.Count )
{
    if( $recordCount[0] -match 'Returned (\d+) of (\d+) items' )
    {
        if( [int]$matches[1] -lt [int]$matches[2] )
        {
            Write-Host "Only retrieved $($matches[1]) of a total of $($matches[2]) items, use -maxRecordCount to return more"
        }
        ## else we got all the records
    }
    else
    {
        Write-Error $recordCount[0]
    }
}
if( ! $results -or ! $results.Count )
{
    Write-Host "No log entries found between $(Get-Date $start -Format G) and $(Get-Date $end -Format G)"
}
else
{
    foreach($result in $results)
    {
	$result
	Write-Host "======================================="
    }
 
}