<#
We are using restart-computer cmdlet which is part of powershell version 5.1. 
This cmdlet can work only when the Security policy 
"User Rights Assignment->Force Shutdown from a remote system"
is enabled with the adminstrators equivalent or required user account
#>

[CmdletBinding()]
Param(
	[Parameter(Mandatory=$True)]
	[int]$days = 30,
	[Parameter(Mandatory=$True)]
	[string]$dgGroups = "none",
	[switch] $shutdown
)

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

$deliveryGroups = $dgGroups -split "," 
$MaxRecordCount = 250

Add-PSSnapin Citrix.Broker.*

foreach($deliveryGroupName in $deliveryGroups)
{
    $allVMs = @()
    $logonDate = (get-date).AddDays(-$days)
    do 
    {
        if($deliveryGroupName -like "*none*")
        {
            # "Checking for machines that have not been used over $days days"
            $VMs = Get-BrokerMachine -MaxRecordCount $MaxRecordCount | where-object {$_.LastConnectionTime -le $logonDate} 
        }
        else {
            # "Checking for machines that have not been used over $days days in the delivery group $deliveryGroupName"
            $VMs = Get-BrokerMachine -DesktopGroupName $deliveryGroupName -MaxRecordCount $MaxRecordCount | where-object {$_.LastConnectionTime -le $logonDate} 
            
        }
        if($VMs.count -gt 0)
        {
            $allVMs += $VMs
        }
    } while ($VMs.count -ge $MaxRecordCount) 
    if($allVMs)
    {
        Write-Host
        Write-Host "Delivery Group;Machine;Last_Connection_Attempt;Shutdown_Action_Status"
        Write-host "----------------------------------------------------------------"
    
        foreach($item in $allVMs)
        {
            $computer = $item.machinename.split('\').item(1)
            $LastConnectionAttempt = $item.LastConnectionTime
            $shutdownAttemptStatus = "Failed"
            if($item.PowerState -eq 'On' -or $item.PowerState -eq 'Unmanaged')
            {
                if($shutdown)
                {
                    try
                    {
                        & $($env:SystemRoot) + "\System32\Shutdown.exe /s /m \\"+$($computer) + " /f" 
                        $shutdownAttemptStatus = "Success"
                    }
                    catch
                    {
                        $shutdownAttemptStatus = "Failed"	
                    }
                    Write-Host "$($item.DesktopGroupName);$computer;$LastConnectionAttempt;$shutdownAttemptStatus"
                }    
                else {
                    Write-Host "$($item.DesktopGroupName);$computer;$LastConnectionAttempt;No shutdown requested"
                }
            }
            else {
                Write-Host "The powerstate of the machine '$($computer)' in the delivery group '$($item.DesktopGroupName)' is not either 'On' or 'Unmanaged'"
            }
        }   
    }
    else {
	"Checking for machines that have not been used over $days days in the delivery group $deliveryGroupName"
    "All machines in the selected delivery group are accessed by users recently or machines in the delivery group are powered off already"
    }
}    

Remove-pssnapin Citrix.Broker.*




                        