$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

<#
.SYNOPSIS
       Check if a Windows Update is installed on a system
.DESCRIPTION
       The script checks if a Windows Update is installed, using Get-WMIObject commandlet and WMI information under 'Win32_QuickFixEngineering'. This provider is also used by Get-HotFix cmdlet.
       If no records found in Win32_QuickFixEngineering, then it tries to find records in Win32_ReliabilityRecords containing HotFixID.    
.EXAMPLE
       Get-WmiObject -query "select * from win32_quickfixengineering WHERE HotFixID = 'kb2992611'" -Computername server
       gwmi -cl Win32_ReliabilityRecords -Computername Server | where { $_.message -match "kb2992611"}
.INPUTS
       HotFixID
.OUTPUTS
       Update installed state, shows some Update Details if Update is found on system
.NOTES
       Requirements 
    - PowerShell 2.0 or greater required
       - Limitations: 
             - Win32_QuickFixEngineering:                  
                    - returns only the updates supplied by Component Based Servicing (CBS) - See MSDN article
                    - prefered source > faster, but does not contain always all details about updates, that may be present in Win32_ReliabilityRecords
             - Win32_ReliabilityRecords:
                    - To access information from this WMI class, the group policy Configure Reliability WMI Providers must be enabled (disabled on server OS by default). See See MSDN article
                    - can contain records about failed installations and uninstallations
#>

$ErrorActionPreference = "Stop"
$UpdateDetails = $null
$HotFixID = $args[0]
$CmpName = $env:computername
$bFound = $False

#using Win32_QuickFixEngineering

try
{      
       Write-Host "Looking for Update in Win32_QuickFixEngineering..."
    $MyUpdate = Get-WmiObject -Query "select * from win32_quickfixengineering WHERE HotFixID = '$($HotFixID)'" -Computername $CmpName
    If ($MyUpdate -ne $null)
    {
        Write-Host ">> Update found in Win32_QuickFixEngineering."
        $bFound = $True
        $MyUpdateType = $MyUpdate | Get-Member | Select -ExpandProperty TypeName -Unique  
        $UpdateDetails = $MyUpdate | ForEach {
            [PSCustomObject] @{
                          Source              =       $MyUpdateType
                          UpdateID     =      $_.HotFixID
                          SourceName   =      $_.SourceName
                          Description  =      $_.Description
                          Details             =       $_.FixComments
                          InstalledBy  =      $_.InstalledBy
                          #InstalledOn =      $_.InstalledOn
                                      }
        }    
    }
    Else
    {
             Write-Host ">> Update not found in Win32_QuickFixEngineering."
       }
}
catch
{      
       Write-Host "WARN: Could not get Updatelist from Win32_QuickFixEngineering. Check if system is online and you have appropriate permissions!"
       Exit 1
}
finally
{
    If ($Error[0])
    {
             Write-Host "Exception Details for QFE query:"
             Write-Host "Exception: $($Error[0].Exception.Message)"
             Write-Host "ExceptionType: $($Error[0].Exception.GetType().Fullname)"
             Exit 1
       }
}
If (!$bFound)
{
    try
    {
             Write-Host "Looking for Update in Win32_ReliabilityRecords..."            
             $MyUpdate = Get-WmiObject -Class Win32_ReliabilityRecords -Computername $CmpName | Where { $_.message -match "$($HotFixID)"} #| select -last 1
        If ($MyUpdate -ne $null)
        {
            Write-Host ">> Update found in Win32_ReliabilityRecords."
                    $bFound = $True  
            $MyUpdateType = $MyUpdate | Get-Member | Select -ExpandProperty TypeName -Unique
                    $UpdateDetails = $MyUpdate | ForEach {
                        [PSCustomObject] @{
                                 Source              =       $MyUpdateType
                                 UpdateID     =       $_.ProductName
                                 SourceName   =       $_.SourceName
                                 Description  =       $_.ProductName
                                 Details             =       $_.Message
                                 InstalledBy  =      $_.user
                                 InstalledOn  =       $_.ConvertToDateTime($_.TimeGenerated)
                                          }
                    }
        }
        Else
        {
                    Write-Host ">> Update not found in Win32_ReliabilityRecords."
             }
    }
    catch
    {
             Write-Host "WARN: Could not get Updatelist from Win32_ReliabilityRecords. Check if system is online, [Win32_ReliabilityRecords] provider is loaded and you have appropriate permissions!!"
       }
    finally
    {
        If ($Error[0])
        {                 
                    Write-Host "Exception Details for ReliabilityRecords query:"
                    Write-Host "Exception: $($Error[0].Exception.Message)"
                    Write-Host "ExceptionType: $($Error[0].Exception.GetType().Fullname)"
             }
    }
}
If ($bFound)
{
       foreach ($UpdateDetail in $UpdateDetails)
    {
             Write-Host "Update Details:"
             Write-Host "=================================================="
             $UpdateDetail
             Write-Host ""
             If ($UpdateDetail.Source.Equals("System.Management.ManagementObject#root\cimv2\Win32_ReliabilityRecords"))
        {
                    #Win32_ReliabilityRecords can also contain unsuccessfull installations of updates.
                    Write-Host "UpdateDetails for [ $HotFixID ] found. Check UpdateDetails if installed successfully!"
                    Write-Host ""
                    Write-Host ""
             }
        else
        {
                    Write-Host "Update [ $HotFixID ] is installed."
             }
       }
}
else
{
       Write-Host ""
       If ($Error[0])
    {
             Write-Host "Could not query all providers! See Output for more details!"
       }
    else
    {
             Write-Host "Update [$HotFixID] not found."
       }
} 
