$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

<#
.DESCRIPTION
	This script is used to get DAG - Database Availability Group information
.NOTES
	File Name      : XchgDAGDetails.ps1
	Created        : 06 Feburary 2015
#>

$Global:ErrorActionPreference="SilentlyContinue"
$dags = @(Get-DatabaseAvailabilityGroup -Status)
[int]$replqueuewarning = 8

foreach ($dag in $dags)
{
	$dbcopyReport = @()		#Database copy health report
	$memberReport = @()		#DAG member server health report
	$databaseSummary = @()		#Database health summary report

	# DAG members found
	$dagmembers = @($dag | Select-Object -ExpandProperty Servers | Sort-Object Name)

	# DAG database found
	$dagdatabases = @(Get-MailboxDatabase -Status | Where-Object {$_.MasterServerOrAvailabilityGroup -eq $dag.Name} | Sort-Object Name)

	foreach ($database in $dagdatabases)
	{
		#Custom object for Database
		$objectHash = @{
		"Database" = $database.Identity
		"Mounted on" = "Unknown"
		"Preference" = $null
		"Total Copies" = $null
		"Healthy Copies" = $null
		"Unhealthy Copies" = $null
		"Healthy Queues" = $null
		"Unhealthy Queues" = $null
		"Lagged Queues" = $null
		"Healthy Indexes" = $null
		"Unhealthy Indexes" = $null
		}

		$databaseObj = New-Object PSObject -Property $objectHash
		$dbcopystatus = @($database | Get-MailboxDatabaseCopyStatus)
		foreach ($dbcopy in $dbcopystatus)
		{
			$objectHash = @{
			"Database Copy" = $dbcopy.Identity
			"Database Name" = $dbcopy.DatabaseName
			"Mailbox Server" = $null
			"Activation Preference" = $null
			"Status" = $null
			"Copy Queue" = $null
			"Replay Queue" = $null
			"Replay Lagged" = $null
			"Truncation Lagged" = $null
			"Content Index" = $null
			}

			$dbcopyObj = New-Object PSObject -Property $objectHash
			$mailboxserver = $dbcopy.MailboxServer
			$pref = ($database | Select-Object -ExpandProperty ActivationPreference | Where-Object {$_.Key -eq $mailboxserver}).Value
			$copystatus = $dbcopy.Status
			$copyqueuelength = $dbcopy.CopyQueueLength
			$replayqueuelength = $dbcopy.ReplayQueueLength
			$contentindexstate = $dbcopy.ContentIndexState

			#Checking whether this is a replay lagged copy
			$replaylagcopies = @($database | Select -ExpandProperty ReplayLagTimes | Where-Object {$_.Value -gt 0})
			if ($($replaylagcopies.count) -gt 0)
			{
				$replaylag = $false
				foreach ($replaylagcopy in $replaylagcopies)
				{
					if ($replaylagcopy.Key -eq $mailboxserver)
					{
						$replaylag = $true
					}
				}

			}
			else
			{
				$replaylag = $false
			}
			#end replay lagged copy

			#Checking for truncation lagged copies
			$truncationlagcopies = @($database | Select -ExpandProperty TruncationLagTimes | Where-Object {$_.Value -gt 0})
			if ($($truncationlagcopies.count) -gt 0)
			{
				$truncatelag = $false
				foreach ($truncationlagcopy in $truncationlagcopies)
				{
					if ($truncationlagcopy.Key -eq $mailboxserver)
					{
						$truncatelag = $true
					}
				}
			}
			else
			{
				$truncatelag = $false
			}

			$dbcopyObj | Add-Member NoteProperty -Name "Mailbox Server" -Value $mailboxserver -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Activation Preference" -Value $pref -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Status" -Value $copystatus -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Copy Queue" -Value $copyqueuelength -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Replay Queue" -Value $replayqueuelength -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Replay Lagged" -Value $replaylag -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Truncation Lagged" -Value $truncatelag -Force
			$dbcopyObj | Add-Member NoteProperty -Name "Content Index" -Value $contentindexstate -Force

			$dbcopyReport += $dbcopyObj
		}

		$copies = @($dbcopyReport | Where-Object { ($_."Database Name" -eq $database) })
		$mountedOn = ($copies | Where-Object { ($_.Status -eq "Mounted") })."Mailbox Server"
		if ($mountedOn)
		{
			$databaseObj | Add-Member NoteProperty -Name "Mounted on" -Value $mountedOn -Force
		}

		$activationPref = ($copies | Where-Object { ($_.Status -eq "Mounted") })."Activation Preference"
		$databaseObj | Add-Member NoteProperty -Name "Preference" -Value $activationPref -Force

		$totalcopies = $copies.count
		$databaseObj | Add-Member NoteProperty -Name "Total Copies" -Value $totalcopies -Force

		$healthycopies = @($copies | Where-Object { (($_.Status -eq "Mounted") -or ($_.Status -eq "Healthy")) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Healthy Copies" -Value $healthycopies -Force

		$unhealthycopies = @($copies | Where-Object { (($_.Status -eq "Failed") -or ($_.Status -eq "ServiceDown") -or ($_.Status -eq "Dismounted") -or ($_.Status -eq "Dismounting") -or ($_.Status -eq "DisconnectedAndHealthy") -or ($_.Status -eq "DisconnectedAndResynchronizing") -or ($_.Status -eq "FailedAndSuspended")) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Unhealthy Copies" -Value $unhealthycopies -Force

		$healthyqueues  = @($copies | Where-Object { (($_."Copy Queue" -lt $replqueuewarning) -and (($_."Replay Queue" -lt $replqueuewarning)) -and ($_."Replay Lagged" -eq $false)) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Healthy Queues" -Value $healthyqueues -Force

		$unhealthyqueues = @($copies | Where-Object { (($_."Copy Queue" -ge $replqueuewarning) -or (($_."Replay Queue" -ge $replqueuewarning) -and ($_."Replay Lagged" -eq $false))) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Unhealthy Queues" -Value $unhealthyqueues -Force

		$laggedqueues = @($copies | Where-Object { ($_."Replay Lagged" -eq $true) -or ($_."Truncation Lagged" -eq $true) }).Count
		$databaseObj | Add-Member NoteProperty -Name "Lagged Queues" -Value $laggedqueues -Force

		$healthyindexes = @($copies | Where-Object { ($_."Content Index" -eq "Healthy") }).Count
		$databaseObj | Add-Member NoteProperty -Name "Healthy Indexes" -Value $healthyindexes -Force

		$unhealthyindexes = @($copies | Where-Object { ($_."Content Index" -ne "Healthy") }).Count
		$databaseObj | Add-Member NoteProperty -Name "Unhealthy Indexes" -Value $unhealthyindexes -Force

		$databaseSummary += $databaseObj
	}

	if ($dagmembers -ne $null)
	{
		foreach ($dagmember in $dagmembers)
		{
			$memberObj = New-Object PSObject
			$memberObj | Add-Member NoteProperty -Name "Server" -Value $($dagmember.Name)

			# Checking replication health for DAG member
			$replicationhealth = $dagmember | Invoke-Command {Test-ReplicationHealth}
			foreach ($healthitem in $replicationhealth)
			{
				$memberObj | Add-Member NoteProperty -Name $($healthitem.Check) -Value $($healthitem.Result)
			}
			$memberReport += $memberObj
		}
	}

	# Database Availability Group Helath Summary

	if ($databaseSummary -ne $null)
	{
		foreach ($line in $databaseSummary)
		{
			$dbName = $line."Database"
			$dbMount = $line."Mounted on"
			$dbPref = $line."Preference"
			$totCopies = $line."Total Copies"
			$healthyCopies = $line."Healthy Copies"
			$unhealthyCopies = $line."Unhealthy Copies"
			$healthQueues = $line."Healthy Queues"
			$unhealthyQueues = $line."Unhealthy Queues"
			$laggedQueues = $line."Lagged Queues"
			$healthyInd = $line."Healthy Indexes"
			$unhealthyUnd = $line."Unhealthy Indexes"
			Write-Host ("DAG Health Summary", $dbName, $dbMount, $dbPref, $totCopies, $healthyCopies, $unhealthyCopies, $healthQueues, $unhealthyQueues, $laggedQueues, $healthyInd, $unhealthyUnd) -Separator ";"
		}
	}

	# Database Availability Group Helath Details

	if ($dbcopyReport -ne $null)
	{
		foreach ($line in $dbcopyReport)
		{
			$dbName = $line."Database Name"
			$dbCopy = $line."Database Copy"
			$mbSrvr = $line."Mailbox Server"
			$actPref = $line."Activation Preference"
			$mbStatus = $line."Status"
			$copyQueue = $line."Copy Queue"
			$replayQueue = $line."Replay Queue"
			$replayLag = $line."Replay Lagged"
			$trunLag = $line."Truncation Lagged"
			$contIndex = $line."Content Index"
			Write-Host ("DAG Health Details", $dbName, $dbCopy, $mbSrvr, $actPref, $mbStatus, $copyQueue, $replayQueue, $replayLag, $trunLag, $contIndex) -Separator ";"
		}
	}

	# Database Availability Group Member Health

	if ($memberReport -ne $null)
	{
		foreach ($line in $memberReport)
		{
			$serverName = $line."Server"
			$cluService = $line."Cluster Service"
			$repService = $line."Replay Service"
			$actManager = $line."Active Manager"
			$rpcListener = $line."Tasks RPC Listener"
			$tcpListener = $line."TCP Listener"
			$dagMemberUp = $line."DAG Members Up"
			$cluNetwork = $line."Cluster Network"
			$qurmGroup = $line."Quorum Group"
			$fileQurm = $line."File Share Quorum"
			$dbSuspend = $line."DB Copy Suspended"
			$dbInitialize = $line."DB Initializing"
			$dbDisconnect = $line."DB Disconnected"
			$dbCopyUp = $line."DB Log Copy Keeping Up"
			$dbReplayUp = $line."DB Log Replay Keeping Up"
			Write-Host ("DAG Member Health", $serverName, $cluService, $repService, $actManager, $rpcListener, $tcpListener, $dagMemberUp, $cluNetwork, $qurmGroup, $fileQurm, $dbSuspend, $dbInitialize, $dbDisconnect, $dbCopyUp, $dbReplayUp) -Separator ";"
		}
	}
}