#Requires -Version 3.0
#Requires -Module dnsserver
#This File is in Unicode format.  Do not edit in an ASCII editor.

#region help text

<#
.SYNOPSIS
	Provides DNS server information in the customized format.
.DESCRIPTION
	To run the script from a workstation, RSAT is required.
	Remote Server Administration Tools for Windows 7 with Service Pack 1 (SP1)
		http://www.microsoft.com/en-us/download/details.aspx?id=7887
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520

.PARAMETER ComputerName
	Specifies a computer to use to run the script against.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	Default is localhost.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  
#>

#endregion

#region script parameters
[CmdletBinding(SupportsShouldProcess = $False, ConfirmImpact = "None") ]

Param(
	
	[parameter(Mandatory=$False)] 
	[string]$ComputerName="LocalHost"
	)
#endregion

#region initialize variables
$SaveEAPreference = $ErrorActionPreference
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False
$scriptStartIdentifier = "*** Start - DNS config data ***"
$scriptEndIdentifier = "*** End - DNS config data ***"
$delimiter = "@@@"
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region script setup function
Function TestComputerName
{
	Param([string]$Cname)
	If(![String]::IsNullOrEmpty($CName)) 
	{
		If(Test-Connection -ComputerName $CName -quiet)
		{
			#Write-Verbose "$(Get-Date): Server $CName is online."
		}
		Else
		{
			#Write-Verbose "$(Get-Date): Computer $CName is offline"
			$ErrorActionPreference = $SaveEAPreference
			Write-Error "`n`n`t`tComputer $CName is offline.`nScript cannot continue.`n`n"
			Exit
		}
	}

	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
		#Write-Verbose "$(Get-Date): Computer name has been renamed from localhost to $CName"
	}

	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip).AddressList.IPAddressToString
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
			#Write-Verbose "$(Get-Date): Computer name has been renamed from $($ip) to $CName"
		}
		Else
		{
			Write-Warning "Unable to resolve $CName to a hostname"
		}
	}
	Else
	{
		#computer is online but for some reason $ComputerName cannot be converted to a System.Net.IpAddress
	}

	$Results = Get-DNSServer -ComputerName $CName -EA 0 3>$Null
	If($Null -ne $Results)
	{
		#the computer is a dns server
		$Script:DNSServerData = $Results
		Return $CName
	}
	ElseIf($Null -eq $Results)
	{
		#the computer is not a dns server
		#Write-Verbose "$(Get-Date): Computer $CName is not a DNS Server"
		$ErrorActionPreference = $SaveEAPreference
		Write-Error "`n`n`t`tComputer $CName is not a DNS Server.`n`n`t`tRerun the script using -ComputerName with a valid DNS server name.`n`n`t`tScript cannot continue.`n`n"
		Exit
	}
	Return $CName
}

Function ProcessScriptStart
{
	$ComputerName = TestComputerName $ComputerName
}
#endregion

#region Process Forward Lookup Zone Name servers
Function ProcessForwardLookupZoneNameServer
{
	#Write-Verbose "$(Get-Date): Processing Forward Lookup Zones"
	$DNSZones = $Script:DNSServerData.ServerZone | Where-Object {$_.IsReverseLookupZone -eq $False -and ($_.ZoneType -eq "Primary" -and $_.ZoneName -ne "TrustAnchors" -or $_.ZoneType -eq "Stub" -or $_.ZoneType -eq "Secondary")}
	
	ForEach($DNSZone in $DNSZones)
	{
		OutputLookupZone "Forward" $DNSZone
	}
}

Function OutputLookupZone
{
	Param([string] $zType, [object] $DNSZone)
	
	#Name Servers tab
	$NameServers = Get-DnsServerResourceRecord -zonename $DNSZone.ZoneName -rrtype ns -node -ComputerName $ComputerName -EA 0 
	$serverIpaddres =" "
	If($? -and $Null -ne $NameServers)
	{   
		ForEach($NS in $NameServers)
		{
			try
			{
				$ipAddress = ([System.Net.Dns]::gethostentry($NS.RecordData.NameServer)).AddressList.IPAddressToString
			}
			catch 
			{
				$ipAddress = "Unknown"
			}
			
			If($ipAddress -is [array])
			{
				$cnt = -1
				
				ForEach($ipAddrs in $ipAddress)
				{
					$cnt++
					If($cnt -eq 0)
					{
						$serverIpaddres +=$ipAddrs
					}
					Else
					{
						$serverIpaddres += "  " + $ipAddrs
					}
				}
			}
			Else
			{
				$serverIpaddres += $ipAddress
			}
			Write-Host  "server name : $($env:ComputerName) $($delimiter) Forward Lookup Zone Name : $($DNSZone.ZoneName)  $($delimiter) Server FQDN: $($NS.RecordData.NameServer) $($delimiter) IP Address : $($serverIpaddres) "
			$serverIpaddres = ""
		}	
	}
	Else
	{
		Write-Host "Name Servers data could not be retrieved"
	}
}
#endregion

#region script core
#Script begins
Write-Host $scriptStartIdentifier 
ProcessScriptStart
try
{
	ProcessForwardLookupZoneNameServer
}
catch 
{
	$global:isErrorOccured = $True
	Write-Verbose "catch block Executed : ProcessForwardLookupZoneNameServer "
}
If(-Not $isErrorOccured)
{
	Write-Host $scriptEndIdentifier
}
#endregion
