$Global:ErrorActionPreference="SilentlyContinue"
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

if($args.Count -eq 0)
{
	"Syntax: CitrixCloudMonitor.ps1 <CustomerId> <Path_of_secureclientfile> [Optional]<eG_Install_Dir> [Optional]<ProxySettings>"
	return
}

function GetBearerToken
{
	param (
	[Parameter(Mandatory=$true)]
	[string] $clientId,
	[Parameter(Mandatory=$true)]
	[string] $clientSecret
	)
  
	$postHeaders = @{"Content-Type"="application/json"}
	$body = @{
	"ClientId"=$clientId;
	"ClientSecret"=$clientSecret
	}
	$trustUrl = "https://trust.citrixworkspacesapi.net/root/tokens/clients"

	$response = Invoke-RestMethod -Uri $trustUrl -Method POST -Body (ConvertTo-Json $body) -Headers $postHeaders
	$bearerToken = $response.token

	return $bearerToken;
}

Add-PSSnapin Citrix*
$ProxyHost = 'none'
$ProxyUser = 'none'
$ProxyPwd = 'none'

if($args.Length -eq 2)
{
	$CustomerId = $args[0]
	$SecureClientFile = $args[1]
}
elseif($args.Length -eq 4)
{
	$CustomerId = $args[0]
	$SecureClientFile = $args[1]
	$eGInstallDir = $args[2]
	# The format is https://ip:port#user#password
	$ProxyArr = $args[3].ToString().Split('#') 
	if($ProxyArr.Length -eq 3)
	{
		$ProxyHost = $ProxyArr[0]
		$ProxyUser = $ProxyArr[1]
		$ProxyPwd = $ProxyArr[2]
		$OEPath = $eGInstallDir + "\lib\OE.exe"
		$ProxyPwd = & $OEPath $ProxyPwd
	}
}

if($ProxyHost -ne 'none')
{
	[system.net.webrequest]::defaultwebproxy = new-object system.net.webproxy($ProxyHost)
	if($ProxyUser -ne 'none' -and $ProxyPwd -ne 'none')
	{
		$proxyCred = New-Object -TypeName System.Management.Automation.PSCredential -argumentlist $ProxyUser, $(convertto-securestring $ProxyPwd -asplaintext -force)
		[system.net.webrequest]::defaultwebproxy.credentials = $proxyCred   #[System.Net.CredentialCache]::DefaultNetworkCredentials
		[system.net.webrequest]::defaultwebproxy.BypassProxyOnLocal = $true
	}
}

Set-XDCredentials -CustomerId $CustomerId -SecureClientFile $SecureClientFile -ProfileType CloudApi -StoreAs "default"
Get-XDCredentials -ProfileName "default" | out-null

<#
Function Get-BrokerMachinesPaged
{
	param (
	$DesktopGroupUid,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "MachineName")
	$desktops = @()
	$desktops += Get-BrokerMachine -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($desktops.count / $total * 100, 1)
		#Write-Host  "Getting Broker Machines. Retrieved $($desktops.count) of $total machines completed in $remainingPerc%"
		$last = $desktops[-1].MachineName
		$desktops += Get-BrokerMachine -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { MachineName -gt $last } @argumentList
	}
	return $desktops
}

Function Get-BrokerApplicationsPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($applications.count / $total * 100, 1)
		#Write-Host  "Getting Broker Applications. Retrieved $($applications.count) of $total applications completed in $remainingPerc%"
		$last = $applications[-1].Uid
		$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { Uid -gt $last } @argumentList
	}
	return $applications
}

Function Get-BrokerSessionsPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$sessions = @()
	$sessions += Get-BrokerSession -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($sessions.count / $total * 100, 1)
		#Write-Host  "Getting Broker Sessions. Retrieved $($sessions.count) of $total sessions completed in $remainingPerc%"
		$last = $sessions[-1].Uid
		$sessions += Get-BrokerSession -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { Uid -gt $last } @argumentList
	}
	return $sessions
}
#>

Function Get-BrokerMachinesPaged
{
	param (
	$DesktopGroupUid,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$desktops = @()
	$LastUid = 0;
	do
	{
		$NewDesktops = @(Get-BrokerMachine -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewDesktops.Count -gt 0)
		{
			$desktops += $NewDesktops
			$LastUid = $desktops[-1].Uid
		}
	} while ($NewDesktops.Count -ge $MaxRecordCount)
	return $desktops
}

Function Get-BrokerApplicationsPaged
{
	param (
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$LastUid = 0;
	do
	{
		$NewApplications = @(Get-BrokerApplication -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewApplications.Count -gt 0)
		{
			$applications += $NewApplications
			$LastUid = $applications[-1].Uid
		}
	} while ($NewApplications.Count -ge $MaxRecordCount)
	return $applications
}

Function Get-BrokerSessionsPaged
{
	param (
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$sessions = @()
	$LastUid = 0;
	do
	{
		$NewSessions = @(Get-BrokerSession -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewSessions.Count -gt 0)
		{
			$sessions += $NewSessions
			$LastUid = $sessions[-1].Uid
		}
	} while ($NewSessions.Count -ge $MaxRecordCount)
	return $sessions
}

$ZoneControllersHash = @{}
$ConfiguredZones = Get-ConfigZone -EA 0
foreach($Zone in $ConfiguredZones)
{
	$ZoneControllersHash.Add($Zone.Name,$Zone.ControllerSids)
}

Write-Host "Site"
Write-Host "----"
$Site = Get-BrokerSite
if ($Site -ne $null)
{
	$LicenseEdition = $Site.LicenseEdition
	if ($LicenseEdition -eq $null -or $LicenseEdition.length -eq 0)
	{
		$LicenseEdition = "-"
	}

	$LicenseModel = $Site.LicenseModel
	if ($LicenseModel -eq $null -or $LicenseModel.length -eq 0)
	{
		$LicenseModel = "-"
	}

	$LicensingGraceHoursLeft = $Site.LicensingGraceHoursLeft
	if($LicensingGraceHoursLeft -eq $null -or $LicensingGraceHoursLeft.length -eq 0)
	{
		$LicensingGraceHoursLeft = "-"
	}

	$IsSecondaryBroker = $Site.IsSecondaryBroker
	if ($IsSecondaryBroker -eq $null -or $IsSecondaryBroker.length -eq 0)
	{
		$IsSecondaryBroker = "-"
	}

	$LocalHostCacheEnabled = $Site.LocalHostCacheEnabled
	if ($LocalHostCacheEnabled -eq $null -or $LocalHostCacheEnabled.length -eq 0)
	{
		$LocalHostCacheEnabled = "-"
	}

	$ReuseMachinesWithoutShutdownInOutageAllowed = $Site.ReuseMachinesWithoutShutdownInOutageAllowed
	if ($ReuseMachinesWithoutShutdownInOutageAllowed -eq $null -or $ReuseMachinesWithoutShutdownInOutageAllowed.length -eq 0)
	{
		$ReuseMachinesWithoutShutdownInOutageAllowed = "-"
	}

	$TotalUniqueLicenseUsers = -5
	if ($Site.TotalUniqueLicenseUsers -ne $null)
	{
		$TotalUniqueLicenseUsers = $Site.TotalUniqueLicenseUsers
	}

	$PeakConcurrentLicenseUsers = -5
	if ($Site.PeakConcurrentLicenseUsers -ne $null)
	{
		$PeakConcurrentLicenseUsers = $Site.PeakConcurrentLicenseUsers
	}

	Write-Host ($Site.Name,$Site.LicenseServerName,$Site.LicenseServerPort,$Site.LicensedSessionsActive,
	$LicenseEdition,$LicenseModel,$Site.LicensingBurnInDate,$Site.DnsResolutionEnabled,$Site.SecureIcaRequired,
	$Site.TrustRequestsSentToTheXmlServicePort,
	$Site.ConnectionLeasingEnabled,$Site.LicensingGracePeriodActive,$Site.LicensingOutOfBoxGracePeriodActive,$LicensingGraceHoursLeft,
	$IsSecondaryBroker,$LocalHostCacheEnabled,$ReuseMachinesWithoutShutdownInOutageAllowed,$TotalUniqueLicenseUsers,$PeakConcurrentLicenseUsers) -Separator ";"
}

Write-Host "Connectors"
Write-Host "----------"

$EdgeServers = Get-ConfigEdgeServer
if ($EdgeServers -ne $null)
{
	foreach ($EdgeServer in $EdgeServers)
	{
		Write-Host ($EdgeServer.MachineAddress,$EdgeServer.ZoneName) -Separator ";"
	}
}

Write-Host "Hypervisor Details"
Write-Host "------------------"
$HypConnPath = "XDHyp:\Connections\"
$HypConnections = Get-ChildItem $HypConnPath
if ($HypConnections -ne $null)
{
	foreach ($HypConn in $HypConnections)
	{
		$ConnectionType = $HypConn.ConnectionType
		$HypervisorConnectionName = $HypConn.HypervisorConnectionName
		$HypervisorAddress = $HypConn.HypervisorAddress
		if ($HypervisorAddress -eq $null -or $HypervisorAddress.length -eq 0)
		{
			$HypervisorAddress = "-"
		}
		else
		{
			$HypvConn = ""
			$count = 0
			foreach ($key in $HypervisorAddress)
			{
				if ($count -eq 0)
				{
					$HypvConn = $key
				}
				else
				{
					$HypvConn = $HypvConn + "#" + $key
				}
				$count = $count + 1
			}
			if ($HypvConn -eq $null -or $HypvConn.length -eq 0)
			{
				$HypervisorAddress = "-"
			}
			else
			{
				$HypervisorAddress = $HypvConn
			}
		}
		$MaintenanceMode = $HypConn.MaintenanceMode
		$Persistent = $HypConn.Persistent
		$SupportsLocalStorageCaching = $HypConn.SupportsLocalStorageCaching
		$SupportsPvsVMs = $HypConn.SupportsPvsVMs
		$UserName = $HypConn.UserName
		$Scopes = $HypConn.Scopes
		if ($Scopes -eq $null -or $Scopes.length -eq 0)
		{
			$Scopes = "-"
		}
		$HypZone = $HypConn.ZoneName
		if($HypZone -eq $null -or $HypZone.length -eq 0)
		{
			$HypZone = "-"
		}
		# Hypervisor State
		# Unavailable - The broker is unable to contact the hypervisor.
		# InMaintenanceMode - The hosting server is in maintenance mode.
		# On - The broker is in contact with the hypervisor.
		$HyperConns = Get-BrokerHypervisorConnection -HypHypervisorConnectionUid $HypConn.HypervisorConnectionUid
		if($HyperConns -ne $null)
		{
			$State = $HyperConns.State
			$PreferredController = $HyperConns.PreferredController
			$IsReady = $HyperConns.IsReady
			$MachineCount = $HyperConns.MachineCount
			Write-Host ($HypervisorConnectionName,$State,$ConnectionType,$HypervisorAddress,$PreferredController,$MaintenanceMode,
			$Persistent,$SupportsLocalStorageCaching,$SupportsPvsVMs,$UserName,$Scopes,$HypZone,$IsReady,$MachineCount) -Separator ";"
		}
	}
}

Write-Host "Controllers"
Write-Host "-----------"
$Controllers = Get-BrokerController
if($Controllers -ne $null)
{
	foreach($Controller in $Controllers)
	{
		$DNSName = $Controller.DNSName
		$ControllerSid = $Controller.SID
		if ($DNSName -eq $null -or $DNSName.length -eq 0)
		{
			$DNSName = "-"
		}
		$ControllerZoneName = "-"
		if($ZoneControllersHash.Count -gt 0)
		{
			$ZoneKeys = $ZoneControllersHash.KEYS.GetEnumerator()
			foreach($ZoneKey in $ZoneKeys)
			{
				$ZoneValues = $ZoneControllersHash[$ZoneKey]
				if($ZoneValues.Contains($ControllerSid) -eq $true)
				{
					$ControllerZoneName = $ZoneKey
				}
			}
		}
		$LicensingGraceState = $Controller.LicensingGraceState
		if ($LicensingGraceState -eq $null -or $LicensingGraceState.length -eq 0)
		{
			$LicensingGraceState = "-"
		}
		$LicensingServerState = $Controller.LicensingServerState
		if ($LicensingServerState -eq $null -or $LicensingServerState.length -eq 0)
		{
			$LicensingServerState = "-"
		}
		Write-Host($DNSName,$Controller.MachineName,$Controller.State,$Controller.DesktopsRegistered,
		$Controller.ControllerVersion,$Controller.OSType,$Controller.OSVersion,$Controller.LastActivityTime,
		$Controller.LastStartTime,$ControllerZoneName,$LicensingGraceState,$LicensingServerState) -Separator ";"
	}
}

Write-Host "Controller Services"
Write-Host "-------------------"
if($Controllers -ne $null)
{
	foreach($Controller in $Controllers)
	{
		$DNSName = $Controller.DNSName
		if ($DNSName -eq $null -or $DNSName.length -eq 0)
		{
			continue
		}
		#$DNSName = $DNSName.Substring($DNSName.indexOf('.'))
		$State = $Controller.State
		if ($State -eq $null -or $State.length -eq 0)
		{
			continue
		}
		if ($State -eq "Off" -or $State -eq "Failed")
		{
			continue
		}
		$BrokerStatus = Get-BrokerServiceStatus -AdminAddress $DNSName
		if ($BrokerStatus -eq $null -or $BrokerStatus.length -eq 0)
		{
			$BrokerStatus = "-"
		}
		$ConfigStatus = Get-ConfigServiceStatus -AdminAddress $DNSName
		if ($ConfigStatus -eq $null -or $ConfigStatus.length -eq 0)
		{
			$ConfigStatus = "-"
		}
		Write-Host($DNSName,$BrokerStatus,$ConfigStatus) -Separator ";"
	}
}

Write-Host "Machine Catalogs"
Write-Host "----------------"
$Catalogs = Get-BrokerCatalog -MaxRecordCount 999
if($Catalogs -ne $null)
{
	foreach($Catalog in $Catalogs)
	{
		$Description = "-"
		$PvsAddress = $Catalog.PvsAddress 
		if ($PvsAddress -eq $null -or $PvsAddress.length -eq 0)
		{
			$PvsAddress = "-"
		}
		$PvsDomain = $Catalog.PvsDomain
		if ($PvsDomain -eq $null -or $PvsDomain.length -eq 0)
		{
			$PvsDomain = "-"
		}
		$Description = $Catalog.Description
		if ($Description -eq $null -or $Description.trim().length -eq 0)
		{
			$Description = "-"
		}
		$Scopes = $Catalog.Scopes
		if ($Scopes -eq $null -or $Scopes.length -eq 0)
		{
			$Scopes = "All"
		}
		$ZoneName = $Catalog.ZoneName
		if ($ZoneName -eq $null -or $ZoneName.length -eq 0)
		{
			$ZoneName = "-"
		}
		Write-Host ($Catalog.Name,$Catalog.AllocationType,$Catalog.AssignedCount,$Catalog.AvailableAssignedCount,$Catalog.AvailableCount,
		$Catalog.AvailableUnassignedCount,$Catalog.UnassignedCount,$Catalog.UsedCount,$Catalog.MachinesArePhysical,$PvsAddress,
		$PvsDomain,$Catalog.PersistUserChanges,$Catalog.ProvisioningType,$Catalog.SessionSupport,$Description,$Scopes,$ZoneName) -Separator ";"
	}
}

#BrowserName and Name Map Hashtbale
$AppNameHash = @{}
$AppAdminNameHash = @{}
#$PublishedAppInstances = Get-BrokerApplication -MaxRecordCount 2147483647
$PublishedAppInstances = Get-BrokerApplicationsPaged
if ($PublishedAppInstances -ne $null)
{
	foreach($PublishedAppInstance in $PublishedAppInstances)
	{
		$BrowserName = $PublishedAppInstance.BrowserName
		if ($BrowserName -ne $null -and $BrowserName.length -gt 0)
		{
			$Name = $PublishedAppInstance.Name
			if ($Name -eq $null -or $Name.length -eq 0)
			{
				$Name = $BrowserName
			}
			$AppAdminNameHash.Add($Name,$PublishedAppInstance)
			$Name = $Name + "(" + $BrowserName + ")"
			$AppNameHash.Add($BrowserName,$Name)
		}
	}
}

Write-Host "Broker Machines"
Write-Host "---------------"

#$BrokerMachines = Get-BrokerMachine -MaxRecordCount 2147483647
$DesktopGroups = Get-BrokerDesktopGroup -MaxRecordCount 999
if ($DesktopGroups  -ne $null)
{
	foreach ($DesktopGroup in $DesktopGroups)
	{
		$BrokerMachines = Get-BrokerMachinesPaged -DesktopGroupUid $DesktopGroup.Uid
		if($BrokerMachines -ne $null)
		{
			foreach($BrokerMachine in $BrokerMachines)
			{
				$HostedMachineName = $BrokerMachine.HostedMachineName
				if ($HostedMachineName -eq $null -or $HostedMachineName.length -eq 0)
				{
					$HostedMachineName = "-"
				}
				$HypervisorConnectionName = $BrokerMachine.HypervisorConnectionName
				if ($HypervisorConnectionName -eq $null -or $HypervisorConnectionName.length -eq 0)
				{
					$HypervisorConnectionName = "-"
				}
				$PublishedApplications = $BrokerMachine.PublishedApplications
				if ($PublishedApplications -eq $null -or $PublishedApplications.length -eq 0)
				{
					$PublishedApplications = "-"
				}
				else
				{
					$PublishApps = ""
					$count = 0
					foreach ($key in $PublishedApplications)
					{
						if ($count -eq 0)
						{
							$PublishApps = $AppNameHash.Get_Item($key)
						}
						else
						{
							$PublishApps = $PublishApps + "#" + $AppNameHash.Get_Item($key)
						}
						$count = $count + 1
					}
					if ($PublishApps -eq $null -or $PublishApps.length -eq 0)
					{
						$PublishedApplications = "-"
					}
					else
					{
						$PublishedApplications = $PublishApps
					}
				}
				$SessionUserName = $BrokerMachine.SessionUserName
				if ($SessionUserName -eq $null -or $SessionUserName.length -eq 0)
				{
					$SessionUserName = "-"
				}
				$IPAddress = $BrokerMachine.IPAddress
				if ($IPAddress -eq $null -or $IPAddress.length -eq 0)
				{
					$IPAddress = "-"
				}
				$OSType = $BrokerMachine.OSType
				if ($OSType -eq $null -or $OSType.length -eq 0)
				{
					$OSType = "-"
				}
				$OSVersion = $BrokerMachine.OSVersion
				if ($OSVersion -eq $null -or $OSVersion.length -eq 0)
				{
					$OSVersion = "-"
				}
				$ControllerDNSName = $BrokerMachine.ControllerDNSName
				if ($ControllerDNSName -eq $null -or $ControllerDNSName.length -eq 0)
				{
					$ControllerDNSName = "-"
				}
				$HostingServerName = $BrokerMachine.HostingServerName
				if ($HostingServerName -eq $null -or $HostingServerName.length -eq 0)
				{
					$HostingServerName = "-"
				}
				$AgentVersion = $BrokerMachine.AgentVersion
				if ($AgentVersion -eq $null -or $AgentVersion.length -eq 0)
				{
					$AgentVersion = "-"
				}
				$LastConnectionFailure = $BrokerMachine.LastConnectionFailure
				if ($LastConnectionFailure -eq $null)
				{
					$LastConnectionFailure = "-"
				}
				$LastConnectionTime = $BrokerMachine.LastConnectionTime
				if ($LastConnectionTime -eq $null)
				{
					$LastConnectionTime = "-"
				}
				$LastDeregistrationReason = $BrokerMachine.LastDeregistrationReason
				if ($LastDeregistrationReason -eq $null -or $LastDeregistrationReason.length -eq 0)
				{
					$LastDeregistrationReason = "-"
				}
				$LastDeregistrationTime = $BrokerMachine.LastDeregistrationTime
				if ($LastDeregistrationTime -eq $null)
				{
					$LastDeregistrationTime = "-"
				}
				$LastErrorReason = $BrokerMachine.LastErrorReason
				if ($LastErrorReason -eq $null -or $LastErrorReason.length -eq 0)
				{
					$LastErrorReason = "-"
				}
				$LastErrorTime = $BrokerMachine.LastErrorTime
				if ($LastErrorTime -eq $null)
				{
					$LastErrorTime = "-"
				}
				$ImageOutOfDate = $BrokerMachine.ImageOutOfDate
				if ($ImageOutOfDate -eq $null)
				{
					$ImageOutOfDate = "-"
				}
				$PublishedName = $BrokerMachine.PublishedName
				if ($PublishedName -eq $null -or $PublishedName.length -eq 0)
				{
					$PublishedName = "-"
				}
				$DesktopConditions = $BrokerMachine.DesktopConditions
				if ($DesktopConditions -eq $null -or $DesktopConditions.length -eq 0)
				{
					$DesktopConditions = "-"
				}
				$FaultState = $BrokerMachine.FaultState
				if ($FaultState -eq $null -or $FaultState.length -eq 0)
				{
					$FaultState = "-"
				}
				$LoadIndex = $BrokerMachine.LoadIndex
				if ($LoadIndex -eq $null -or $LoadIndex.length -eq 0)
				{
					$LoadIndex = "-"
				}
				$LoadIndexes = $BrokerMachine.LoadIndexes
				if ($LoadIndexes -eq $null -or $LoadIndexes.length -eq 0)
				{
					$LoadIndexes = "-"
				}
				else
				{
					$LoadIdxes = ""
					$count = 0
					foreach ($key in $LoadIndexes)
					{
						if ($count -eq 0)
						{
							$LoadIdxes = $key
						}
						else
						{
							$LoadIdxes = $LoadIdxes + "#" + $key
						}
						$count = $count + 1
					}
					if ($LoadIdxes -eq $null -or $LoadIdxes.length -eq 0)
					{
						$LoadIndexes = "-"
					}
					else
					{
						$LoadIndexes = $LoadIdxes
					}
				}

				$Tags = $BrokerMachine.Tags
				$BrokerMachineTags = "-"
				if ($Tags -ne $null -and $Tags.length -gt 0)
				{
					$TagCount = 0
					foreach ($Tag in $Tags)
					{
						if ($TagCount -eq 0)
						{
							$BrokerMachineTags = $Tag
						}
						else
						{
							$BrokerMachineTags = $BrokerMachineTags + "#" + $Tag
						}
						$TagCount = $TagCount + 1
					}
				}

				Write-Host($BrokerMachine.CatalogName,$BrokerMachine.DesktopGroupName,$BrokerMachine.AllocationType,$BrokerMachine.MachineName,$BrokerMachine.PowerState,
				$BrokerMachine.RegistrationState,$BrokerMachine.SummaryState,$BrokerMachine.DNSName,$ControllerDNSName,$HostedMachineName,$HypervisorConnectionName,
				$HostingServerName,$BrokerMachine.InMaintenanceMode,$BrokerMachine.IsAssigned,$BrokerMachine.IsPhysical,
				$BrokerMachine.MachineInternalState,$OSType,$OSVersion,$BrokerMachine.PersistUserChanges,$BrokerMachine.ProvisioningType,
				$PublishedApplications,$BrokerMachine.SessionCount,$BrokerMachine.SessionsEstablished,$BrokerMachine.SessionsPending,
				$BrokerMachine.SessionSupport,$SessionUserName,$IPAddress,$AgentVersion,$LastConnectionFailure,$LastConnectionTime,
				$LastDeregistrationReason,$LastDeregistrationTime,$LastErrorReason,$LastErrorTime,$ImageOutOfDate,$PublishedName,
				$BrokerMachine.PowerActionPending,$BrokerMachine.WillShutdownAfterUse,$FaultState,$LoadIndex,$LoadIndexes,$BrokerMachine.VMToolsState,$BrokerMachine.WindowsConnectionSetting,$BrokerMachineTags,$DesktopConditions) -Separator ";"
			}
		}
	}
}


Write-Host "Delivery Groups"
Write-Host "---------------"
#$DesktopGroups = Get-BrokerDesktopGroup -MaxRecordCount 2147483647
if($DesktopGroups -ne $null)
{
	foreach($DesktopGroup in $DesktopGroups)
	{
		$Description = $DesktopGroup.Description
		if ($Description -eq $null -or $Description.trim().length -eq 0)
		{
			$Description = "-"
		}
		$PublishedName = $DesktopGroup.PublishedName
		if ($PublishedName -eq $null -or $PublishedName.length -eq 0)
		{
			$PublishedName = "-"
		}

		$Tags = $DesktopGroup.Tags
		$DesktopGroupTags = "-"
		if ($Tags -ne $null -and $Tags.length -gt 0)
		{
			$TagCount = 0
			foreach ($Tag in $Tags)
			{
				if ($TagCount -eq 0)
				{
					$DesktopGroupTags = $Tag
				}
				else
				{
					$DesktopGroupTags = $DesktopGroupTags + "#" + $Tag
				}
				$TagCount = $TagCount + 1
			}
		}

		Write-Host ($DesktopGroup.Name,$Description,$DesktopGroup.Enabled,$DesktopGroup.InMaintenanceMode,$DesktopGroup.TotalDesktops,
		$DesktopGroup.DesktopsAvailable,$DesktopGroup.DesktopsDisconnected,$DesktopGroup.DesktopsInUse,
		$DesktopGroup.DesktopsNeverRegistered,$DesktopGroup.DesktopsUnregistered,$DesktopGroup.DesktopsPreparing,
		$DesktopGroup.DesktopKind,$DesktopGroup.SecureIcaRequired,$DesktopGroup.SessionSupport,
		$DesktopGroup.ShutdownDesktopsAfterUse,$PublishedName,$DesktopGroup.AutomaticPowerOnForAssigned,
		$DesktopGroup.AutomaticPowerOnForAssignedDuringPeak,$DesktopGroup.DeliveryType,
		$DesktopGroup.OffPeakBufferSizePercent,$DesktopGroup.PeakBufferSizePercent,
		$DesktopGroup.Sessions,$DesktopGroup.TotalApplications,$DesktopGroup.TurnOnAddedMachine,$DesktopGroupTags) -Separator ";"
	}
}

Write-Host "Sessions"
Write-Host "--------"
# SessionState
# Valid values are Other, PreparingSession, Connected, Active, Disconnected, Reconnecting, NonBrokeredSession, and Unknown.

#$BrokerSessions = Get-BrokerSession -MaxRecordCount 2147483647
$BrokerSessions = Get-BrokerSessionsPaged
if ($BrokerSessions -ne $null)
{
	foreach($BrokerSession in $BrokerSessions)
	{
		$IPAddress = $BrokerSession.IPAddress
		if ($IPAddress -eq $null -or $IPAddress.length -eq 0)
		{
			$IPAddress = "-"
		}
		$OSType = $BrokerSession.OSType
		if ($OSType -eq $null -or $OSType.length -eq 0)
		{
			$OSType = "-"
		}
		$HostedMachineName = $BrokerSession.HostedMachineName
		if ($HostedMachineName -eq $null -or $HostedMachineName.length -eq 0)
		{
			$HostedMachineName = "-"
		}
		$HostingServerName = $BrokerSession.HostingServerName
		if ($HostingServerName -eq $null -or $HostingServerName.length -eq 0)
		{
			$HostingServerName = "-"
		}
		$HypervisorConnectionName = $BrokerSession.HypervisorConnectionName
		if ($HypervisorConnectionName -eq $null -or $HypervisorConnectionName.length -eq 0)
		{
			$HypervisorConnectionName = "-"
		}
		$UserName = $BrokerSession.UserName
		if ($UserName -eq $null -or $UserName.length -eq 0)
		{
			$UserName = $BrokerSession.$BrokeringUserName
			if ($UserName -eq $null -or $UserName.length -eq 0)
			{
				$UserName = "-"
			}
		}
		$ClientAddress = $BrokerSession.ClientAddress
		if ($ClientAddress -eq $null -or $ClientAddress.length -eq 0)
		{
			$ClientAddress = "-"
		}
		$ClientName = $BrokerSession.ClientName
		if ($ClientName -eq $null -or $ClientName.length -eq 0)
		{
			$ClientName = "-"
		}
		$ClientVersion = $BrokerSession.ClientVersion
		if ($ClientVersion -eq $null -or $ClientVersion.length -eq 0)
		{
			$ClientVersion = "-"
		}
		$ClientPlatform = $BrokerSession.ClientPlatform
		if ($ClientPlatform -eq $null -or $ClientPlatform.length -eq 0)
		{
			$ClientPlatform = "-"
		}
		$SessionType = $BrokerSession.SessionType
		if ($SessionType -eq $null -or $SessionType.length -eq 0)
		{
			$SessionType = "-"
		}
		$BrokeringTime = $BrokerSession.BrokeringTime
		if ($BrokeringTime -eq $null)
		{
			$BrokeringTime = "-"
		}
		$ConnectedViaHostName = $BrokerSession.ConnectedViaHostName
		if ($ConnectedViaHostName -eq $null -or $ConnectedViaHostName.length -eq 0)
		{
			$ConnectedViaHostName = "-"
		}
		$ConnectedViaIP = $BrokerSession.ConnectedViaIP
		if ($ConnectedViaIP -eq $null -or $ConnectedViaIP.length -eq 0)
		{
			$ConnectedViaIP = "-"
		}
		$LaunchedViaHostName = $BrokerSession.LaunchedViaHostName
		if ($LaunchedViaHostName -eq $null -or $LaunchedViaHostName.length -eq 0)
		{
			$LaunchedViaHostName = "-"
		}
		$LaunchedViaIP = $BrokerSession.LaunchedViaIP
		if ($LaunchedViaIP -eq $null -or $LaunchedViaIP.length -eq 0)
		{
			$LaunchedViaIP = "-"
		}
		$Protocol = $BrokerSession.Protocol
		if ($Protocol -eq $null -or $Protocol.length -eq 0)
		{
			$Protocol = "-"
		}
		$ControllerDNSName = $BrokerSession.ControllerDNSName
		if ($ControllerDNSName -eq $null -or $ControllerDNSName.length -eq 0)
		{
			$ControllerDNSName = "-"
		}
		$LicenseId = $BrokerSession.LicenseId
		if ($LicenseId -eq $null -or $LicenseId -eq 0)
		{
			$LicenseId = "-"
		}
		$BrokeringDuration = $BrokerSession.BrokeringDuration
		if ($BrokeringDuration -eq $null)
		{
			$BrokeringDuration = "-"
		}
		$EstablishmentDuration = $BrokerSession.EstablishmentDuration
		if ($EstablishmentDuration -eq $null)
		{
			$EstablishmentDuration = "-"
		}
		$EstablishmentTime = $BrokerSession.EstablishmentTime
		if ($EstablishmentTime -eq $null)
		{
			$EstablishmentTime = "-"
		}
		$SessionStateChangeTime = $BrokerSession.SessionStateChangeTime
		if ($SessionStateChangeTime -eq $null)
		{
			$SessionStateChangeTime = "-"
		}
		$StartTime = $BrokerSession.StartTime
		if ($StartTime -eq $null)
		{
			$StartTime = "-"
		}
		$DeviceId = $BrokerSession.DeviceId
		if ($DeviceId -eq $null)
		{
			$DeviceId = "-"
		}
		$HardwareId = $BrokerSession.HardwareId
		if ($HardwareId -eq $null)
		{
			$HardwareId = "-"
		}
		$AgentVersion = $BrokerSession.AgentVersion
		if ($AgentVersion -eq $null -or $AgentVersion.length -eq 0)
		{
			$AgentVersion = "-"
		}

		$ApplicationsInUse = $BrokerSession.ApplicationsInUse
		if ($ApplicationsInUse -eq $null -or $ApplicationsInUse.length -eq 0)
		{
			$ApplicationsInUse = "-"
		}
		else
		{
			$AppsInUse = ""
			$count = 0
			foreach ($key in $ApplicationsInUse)
			{
				if ($count -eq 0)
				{
					$AppsInUse = $key
				}
				else
				{
					$AppsInUse = $AppsInUse + "#" + $key
				}
				$count = $count + 1
			}
			if ($AppsInUse -eq $null -or $AppsInUse.length -eq 0)
			{
				$ApplicationsInUse = "-"
			}
			else
			{
				$ApplicationsInUse = $AppsInUse
			}
		}
		$AppState = $BrokerSession.AppState
		if ($AppState -eq $null -or $AppState.length -eq 0)
		{
			$AppState = "-"
		}
		$LogoffInProgress = $BrokerSession.LogoffInProgress
		if ($LogoffInProgress -eq $null -or $LogoffInProgress.length -eq 0)
		{
			$LogoffInProgress = "-"
		}
		$LogonInProgress = $BrokerSession.LogonInProgress
		if ($LogonInProgress -eq $null -or $LogonInProgress.length -eq 0)
		{
			$LogonInProgress = "-"
		}
		$SessionKey = $BrokerSession.SessionKey
		if ($SessionKey -eq $null -or $SessionKey.length -eq 0)
		{
			$SessionKey = "-"
		}
		$IsAnonymousUser = $Broker.IsAnonymousUser
		if ($IsAnonymousUser -eq $null -or $IsAnonymousUser.length -eq 0)
		{
			$IsAnonymousUser = "-"
		}

		Write-Host ($BrokerSession.Uid,$BrokerSession.MachineName,$BrokerSession.DNSName,$IPAddress,$OSType,
		$BrokerSession.CatalogName,$BrokerSession.DesktopGroupName,$HostedMachineName,$HostingServerName,
		$HypervisorConnectionName,$UserName,$ClientAddress,$ClientName,$ClientVersion,$ConnectedViaHostName,$ConnectedViaIP,
		$LaunchedViaHostName,$LaunchedViaIP,$Protocol,$BrokerSession.SessionState,$BrokeringDuration,$EstablishmentDuration,$EstablishmentTime,
		$SessionStateChangeTime,$StartTime,$BrokeringTime,$LicenseId,$BrokerSession.Hidden,$BrokerSession.AutonomouslyBrokered,
		$DeviceId,$HardwareId,$AgentVersion,$BrokerSession.SessionSupport,$ControllerDNSName,$ApplicationsInUse,$AppState,
		$LogoffInProgress,$LogonInProgress,$SessionKey,$ClientPlatform,$SessionType,$IsAnonymousUser) -Separator ";"
	}
}

Write-Host "Application Instances"
Write-Host "---------------------"
if ($BrokerSessions -ne $null)
{
	foreach($Session in $BrokerSessions)
	{
		if($Session)
		{
			$ControllerDNSName = $Session.ControllerDNSName
			$UserName = $Session.UserName
			if ($UserName -eq $null -or $UserName.length -eq 0)
			{
				$UserName = "-"
			}
			$ClientName = $Session.ClientName
			if ($ClientName -eq $null -or $ClientName.length -eq 0)
			{
				$ClientName = "-"
			}
			$ClientAddress = $Session.ClientAddress
			if ($ClientAddress -eq $null -or $ClientAddress -eq 0)
			{
				$ClientAddress = "-"
			}
			$StartTime = $Session.StartTime
			if ($StartTime -eq $null)
			{
				$StartTime = "-"
			}
			$BrokeringTime = $Session.BrokeringTime
			if ($BrokeringTime -eq $null)
			{
				$BrokeringTime = "-"
			}
			$HostedMachineName = $Session.HostedMachineName
			if ($HostedMachineName -eq $null -or $HostedMachineName.length -eq 0)
			{
				$HostedMachineName = "-"
			}
			$MachineName = $Session.MachineName
			if ($MachineName -eq $null -or $MachineName.length -eq 0)
			{
				$MachineName = "-"
			}
			$DesktopGroupName = $Session.DesktopGroupName
			if ($DesktopGroupName -eq $null -or $DesktopGroupName.length -eq 0)
			{
				$DesktopGroupName = "-"
			}
			$CatalogName = $Session.CatalogName
			if ($CatalogName -eq $null -or $CatalogName.length -eq 0)
			{
				$CatalogName = "-"
			}
			$SessionState = $Session.SessionState
			if ($SessionState -eq $null -or $SessionState.length -eq 0)
			{
				$SessionState = "-"
			}
			$SessionSupport = $Session.SessionSupport
			if ($SessionSupport -eq $null -or $SessionSupport.length -eq 0)
			{
				$SessionSupport = "-"
			}
			$Protocol = $Session.Protocol
			if ($Protocol -eq $null -or $Protocol.length -eq 0)
			{
				$Protocol = "-"
			}

			#$AppInstances = Get-BrokerApplicationInstance -SessionUid $Session.Uid -EA 0
			$AppInstances = $Session.ApplicationsInUse
			if ($AppInstances -ne $null)
			{	
				foreach($AppInstance in $AppInstances)
				{
					#$Application = Get-BrokerApplication -Uid $AppInstance.ApplicationUid
					$Application = $AppAdminNameHash.Get_Item($AppInstance)
					if ($Application -ne $null)
					{
						$Tags = $Application.Tags
						$ApplicationTags = "-"
						if ($Tags -ne $null -and $Tags.length -gt 0)
						{
							$TagCount = 0
							foreach ($Tag in $Tags)
							{
								if ($TagCount -eq 0)
								{
									$ApplicationTags = $Tag
								}
								else
								{
									$ApplicationTags = $ApplicationTags + "#" + $Tag
								}
								$TagCount = $TagCount + 1
							}
						}
						#Write-Host($ControllerDNSName,$Application.Uid,$Application.Name,$AppInstance.Instances,$MachineName,$HostedMachineName,
						#	   $DesktopGroupName,$CatalogName,$AppInstance.SessionUid,$UserName,$ClientName,$ClientAddress,
						#	   $Protocol,$SessionState,$StartTime,$BrokeringTime,$SessionSupport,$ApplicationTags) -Separator ";"

						$Instances = 1
						Write-Host($ControllerDNSName,$Application.Uid,$Application.Name,$Instances,$MachineName,$HostedMachineName,
							   $DesktopGroupName,$CatalogName,$Session.Uid,$UserName,$ClientName,$ClientAddress,
							   $Protocol,$SessionState,$StartTime,$BrokeringTime,$SessionSupport,$ApplicationTags) -Separator ";"
					}
				}
			}
		}
	}
}

Write-Host "Published Applications"
Write-Host "----------------------"
#$PublishedAppInstances = Get-BrokerApplication -MaxRecordCount 2147483647
if ($PublishedAppInstances -ne $null)
{
	foreach($PublishedAppInstance in $PublishedAppInstances)
	{
		$Description = $PublishedAppInstance.Description
		if ($Description -eq $null -or $Description.trim().length -eq 0)
		{
			$Description = "-"
		}
		$DisplayName = "-"
		$WorkingDirectory = $PublishedAppInstance.WorkingDirectory
		if ($WorkingDirectory -eq $null -or $WorkingDirectory.length -eq 0)
		{
			$WorkingDirectory = "-"
		}

		$CommandLineExecutable = $PublishedAppInstance.CommandLineExecutable
		$CommandLineArgs = $PublishedAppInstance.CommandLineArguments
		$AppVCommandLineArgs = "-"
		if ($CommandLineExecutable -ne $null -and ($CommandLineExecutable -ieq "CtxAppVLauncher.exe"))
		{
			$MetadataMap = $PublishedAppInstance.MetadataMap
			if ($MetadataMap -ne $null)
			{
				$AppID = $MetadataMap.get_Item("Citrix_DesktopStudio_AppID")
				$PkgID = $MetadataMap.get_Item("Citrix_DesktopStudio_PackageId")
				if ($AppID -ne $null -and $AppID.length -gt 0 -and $PkgID -ne $null -and $PkgID.length -gt 0)
				{
					$Idx = $PkgID.LastIndexOf("_")
					if ($Idx -gt 0)
					{
						$PkgID = $PkgID.Substring(0,$Idx)
					}
					$AppVCommandLineArgs = "CtxAppVLauncher.exe" + "#" + $AppID + "#" + $PkgID
				}
			}
		}
		if ($CommandLineArgs)
		{
			$CommandLineExecutable = $CommandLineExecutable + " " + $CommandLineArgs
		}
		$Tags = $PublishedAppInstance.Tags
		$PublishedAppInstanceTags = "-"
		if ($Tags -ne $null -and $Tags.length -gt 0)
		{
			$TagCount = 0
			foreach ($Tag in $Tags)
			{
				if ($TagCount -eq 0)
				{
					$PublishedAppInstanceTags = $Tag
				}
				else
				{
					$PublishedAppInstanceTags = $PublishedAppInstanceTags + "#" + $Tag
				}
				$TagCount = $TagCount + 1
			}
		}

		$NameAndBrowserName = $PublishedAppInstance.Name + "(" + $PublishedAppInstance.BrowserName + ")"
		Write-Host($PublishedAppInstance.Uid,$NameAndBrowserName,$DisplayName,$Description,$PublishedAppInstance.ApplicationType,
		$PublishedAppInstance.CpuPriorityLevel,$PublishedAppInstance.Enabled,$PublishedAppInstance.Visible,
		$CommandLineExecutable,$WorkingDirectory,$AppVCommandLineArgs,$PublishedAppInstanceTags) -Separator ";"
	}
}

Write-Host "Zones"
Write-Host "------"
foreach($Zone in $ConfiguredZones)
{
	Write-Host($Zone.Name,$Zone.IsPrimary) -Separator ";"
}


Write-Host "Broker Tag Usage"
Write-Host "----------------"
$BrokerTagUsage = Get-BrokerTagUsage
if ($BrokerTagUsage -ne $null)
{
	foreach ($TagUsage in $BrokerTagUsage)
	{
		Write-Host ($TagUsage.TagUUID,$TagUsage.TagName,$TagUsage.TaggedDesktopGroups,$TagUsage.TaggedMachines,$TagUsage.TaggedApplicationGroups,
		$TagUsage.TaggedApplications,$TagUsage.TotalTaggedObjects,$TagUsage.TotalTagRestrictedObjects,$TagUsage.TagRestrictedApplicationGroups,
		$TagUsage.TagRestrictedEntitlementPolicyRules,$TagUsage.TagRestrictedRebootSchedules,$TagUsage.UnknownTaggedObjects,$TagUsage.UnknownTagRestrictedObjects) -Separator ";"
	}
}

Write-Host "Date Time Format"
Write-Host "----------------"
$ShortDatePattern = (Get-Culture).datetimeformat.ShortDatePattern
$LongTimePattern = (Get-Culture).datetimeformat.LongTimePattern
Write-Host "$ShortDatePattern $LongTimePattern"

Remove-PSSnapin citrix*


