# This Script will give all details of Citrix Cloud Discovery

$Global:ErrorActionPreference="SilentlyContinue"
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

if($args.Count -eq 0)
{
	"Syntax: CitrixCloudDiscovery.ps1 <CustomerId> <Path_of_secureclientfile> [Optional]<eG_Install_Dir> [Optional]<ProxySettings>"
	return
}

function GetBearerToken
{
	param (
	[Parameter(Mandatory=$true)]
	[string] $clientId,
	[Parameter(Mandatory=$true)]
	[string] $clientSecret
	)
  
	$postHeaders = @{"Content-Type"="application/json"}
	$body = @{
	"ClientId"=$clientId;
	"ClientSecret"=$clientSecret
	}
	$trustUrl = "https://trust.citrixworkspacesapi.net/root/tokens/clients"

	$response = Invoke-RestMethod -Uri $trustUrl -Method POST -Body (ConvertTo-Json $body) -Headers $postHeaders
	$bearerToken = $response.token

	return $bearerToken;
}

Add-PSSnapin Citrix*
$ProxyHost = 'none'
$ProxyUser = 'none'
$ProxyPwd = 'none'

if($args.Length -eq 2)
{
	$CustomerId = $args[0]
	$SecureClientFile = $args[1]
}
elseif($args.Length -eq 4)
{
	$CustomerId = $args[0]
	$SecureClientFile = $args[1]
	$eGInstallDir = $args[2]
	# The format is https://ip:port#user#password
	$ProxyArr = $args[3].ToString().Split('#') 
	if($ProxyArr.Length -eq 3)
	{
		$ProxyHost = $ProxyArr[0]
		$ProxyUser = $ProxyArr[1]
		$ProxyPwd = $ProxyArr[2]
		$OEPath = $eGInstallDir + "\lib\OE.exe"
		$ProxyPwd = & $OEPath $ProxyPwd
	}
}

if($ProxyHost -ne 'none')
{
	[system.net.webrequest]::defaultwebproxy = new-object system.net.webproxy($ProxyHost)
	if($ProxyUser -ne 'none' -and $ProxyPwd -ne 'none')
	{
		$proxyCred = New-Object -TypeName System.Management.Automation.PSCredential -argumentlist $ProxyUser, $(convertto-securestring $ProxyPwd -asplaintext -force)
		[system.net.webrequest]::defaultwebproxy.credentials = $proxyCred   #[System.Net.CredentialCache]::DefaultNetworkCredentials
		[system.net.webrequest]::defaultwebproxy.BypassProxyOnLocal = $true
	}
}

Set-XDCredentials -CustomerId $CustomerId -SecureClientFile $SecureClientFile -ProfileType CloudApi -StoreAs "default"
Get-XDCredentials -ProfileName "default" | out-null

<#
Function Get-BrokerMachinesPaged
{
	param (
	$MachineName,
	$AgentVersion,
	$AllocationType,
	$ApplicationInUse,
	$AssignedClientName,
	$AssignedIPAddress,
	$AssociatedUserFullName,
	$AssociatedUserName,
	$AssociatedUserSID,
	$AssociatedUserUPN,
	$BrowserName,
	$CatalogName,
	$CatalogUid,
	$CatalogUUID,
	$ColorDepth,
	$ControllerDNSName,
	$DeliveryType,
	$Description,
	$DesktopCondition,
	$DesktopGroupName,
	$DesktopGroupUid,
	$DesktopGroupUUID,
	$DesktopKind,
	$DesktopUid,
	$DNSName,
	$FaultState,
	$FunctionalLevel,
	$HostedMachineId,
	$HostedMachineName,
	$HostingServerName,
	$HypervisorConnectionName,
	$HypervisorConnectionUid,
	$HypHypervisorConnectionUid,
	$IconUid,
	$ImageOutOfDate,
	$InMaintenanceMode,
	$IPAddress,
	$IsAssigned,
	$IsPhysical,
	$LastConnectionFailure,
	$LastConnectionTime,
	$LastConnectionUser,
	$LastDeregistrationReason,
	$LastDeregistrationTime,
	$LastErrorReason,
	$LastErrorTime,
	$LastHostingUpdateTime,
	$LastPvdErrorReason,
	$LastPvdErrorTime,
	$LoadIndex,
	$MachineInternalState,
	$Metadata,
	$OSType,
	$OSVersion,
	$PersistUserChanges,
	$PowerActionPending,
	$PowerState,
	$ProvisioningType,
	$PublishedApplication,
	$PublishedName,
	$PvdEstimatedCompletionTime,
	$PvdPercentDone,
	$PvdStage,
	$PvdUpdateStartTime,
	$RegistrationState,
	$ScheduledReboot,
	$SecureIcaRequired,
	$SessionAutonomouslyBrokered,
	$SessionClientAddress,
	$SessionClientName,
	$SessionClientVersion,
	$SessionConnectedViaHostName,
	$SessionConnectedViaIP,
	$SessionCount,
	$SessionDeviceId,
	$SessionHardwareId,
	$SessionHidden,
	$SessionKey,
	$SessionLaunchedViaHostName,
	$SessionLaunchedViaIP,
	$SessionProtocol,
	$SessionSecureIcaActive,
	$SessionsEstablished,
	$SessionSmartAccessTag,
	$SessionsPending,
	$SessionStartTime,
	$SessionState,
	$SessionStateChangeTime,
	$SessionSupport,
	$SessionType,
	$SessionUid,
	$SessionUserName,
	$SessionUserSID,
	$SID,
	$SummaryState,
	$SupportedPowerActions,
	$Tag,
	$UUID,
	$VMToolsState,
	$WillShutdownAfterUse,
	$WindowsConnectionSetting,
	$AssignedUserSID,
	$AdminAddress,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "MachineName")
	$desktops = @()
	$desktops += Get-BrokerMachine -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt $MaxRecordCount)
	{
	$remainingPerc= [math]::round($desktops.count / $total * 100, 1)
	#Write-Host  "Getting Broker Machines. Retrieved $($desktops.count) of $total machines completed in $remainingPerc%"
	$last = $desktops[-1].MachineName
	$desktops += Get-BrokerMachine -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { MachineName -gt $last } @argumentList
	}

	return $desktops
}
#>

Function Get-BrokerMachinesPaged
{
	param (
	$MachineName,
	$AgentVersion,
	$AllocationType,
	$ApplicationInUse,
	$AssignedClientName,
	$AssignedIPAddress,
	$AssociatedUserFullName,
	$AssociatedUserName,
	$AssociatedUserSID,
	$AssociatedUserUPN,
	$BrowserName,
	$CatalogName,
	$CatalogUid,
	$CatalogUUID,
	$ColorDepth,
	$ControllerDNSName,
	$DeliveryType,
	$Description,
	$DesktopCondition,
	$DesktopGroupName,
	$DesktopGroupUid,
	$DesktopGroupUUID,
	$DesktopKind,
	$DesktopUid,
	$DNSName,
	$FaultState,
	$FunctionalLevel,
	$HostedMachineId,
	$HostedMachineName,
	$HostingServerName,
	$HypervisorConnectionName,
	$HypervisorConnectionUid,
	$HypHypervisorConnectionUid,
	$IconUid,
	$ImageOutOfDate,
	$InMaintenanceMode,
	$IPAddress,
	$IsAssigned,
	$IsPhysical,
	$LastConnectionFailure,
	$LastConnectionTime,
	$LastConnectionUser,
	$LastDeregistrationReason,
	$LastDeregistrationTime,
	$LastErrorReason,
	$LastErrorTime,
	$LastHostingUpdateTime,
	$LastPvdErrorReason,
	$LastPvdErrorTime,
	$LoadIndex,
	$MachineInternalState,
	$Metadata,
	$OSType,
	$OSVersion,
	$PersistUserChanges,
	$PowerActionPending,
	$PowerState,
	$ProvisioningType,
	$PublishedApplication,
	$PublishedName,
	$PvdEstimatedCompletionTime,
	$PvdPercentDone,
	$PvdStage,
	$PvdUpdateStartTime,
	$RegistrationState,
	$ScheduledReboot,
	$SecureIcaRequired,
	$SessionAutonomouslyBrokered,
	$SessionClientAddress,
	$SessionClientName,
	$SessionClientVersion,
	$SessionConnectedViaHostName,
	$SessionConnectedViaIP,
	$SessionCount,
	$SessionDeviceId,
	$SessionHardwareId,
	$SessionHidden,
	$SessionKey,
	$SessionLaunchedViaHostName,
	$SessionLaunchedViaIP,
	$SessionProtocol,
	$SessionSecureIcaActive,
	$SessionsEstablished,
	$SessionSmartAccessTag,
	$SessionsPending,
	$SessionStartTime,
	$SessionState,
	$SessionStateChangeTime,
	$SessionSupport,
	$SessionType,
	$SessionUid,
	$SessionUserName,
	$SessionUserSID,
	$SID,
	$SummaryState,
	$SupportedPowerActions,
	$Tag,
	$UUID,
	$VMToolsState,
	$WillShutdownAfterUse,
	$WindowsConnectionSetting,
	$AssignedUserSID,
	$AdminAddress,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$desktops = @()
	$LastUid = 0;
	do
	{
		$NewDesktops = @(Get-BrokerMachine -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewDesktops.Count -gt 0)
		{
			$desktops += $NewDesktops
			$LastUid = $desktops[-1].Uid
		}
	} while ($NewDesktops.Count -ge $MaxRecordCount)
	return $desktops
}

Write-Host "[Connectors]"
$EdgeServers = Get-ConfigEdgeServer
if ($EdgeServers -ne $null)
{
	foreach ($EdgeServer in $EdgeServers)
	{
		$ips = @(([net.dns]::GetHostEntry($EdgeServer.MachineAddress)).AddressList)
		$tips = ""
		$count = 0
		if($? -and $ips)
		{
			foreach($ip in $ips)
			{
				if($ip.AddressFamily -eq "InterNetwork")
				{
					if($count -eq 0)
					{
						$tips = $ip.IPAddressToString
					}
					else
					{
						$tips = $tips + "," + $ip.IPAddressToString
					}
					$count = $count + 1
				}	
			}
			if ($tips.length -eq 0)
			{
				$tips = "-"
			}
			Write-Host $EdgeServer.MachineAddress"=" $tips
		}
		else
		{
			Write-Host $EdgeServer.MachineAddress"= -"
		}
		$ips = @{}
		$count = 0
		$tips = ""
	}
}
$ips = @{}
$count = 0
$tips = ""

Write-Host "[Controllers]"
$Controllers = Get-BrokerController
if($Controllers -ne $null)
{
	foreach($Controller in $Controllers)
	{
		Write-Host $Controller.MachineName "=" $Controller.DNSName
	}
}

Write-Host "[Hypervisor_Details]"
$BrokerHypConns = Get-BrokerHypervisorConnection -State On
$HypConnsToBroker = @{}
if($BrokerHypConns -ne $null)
{
	foreach($BrokerHypConn in $BrokerHypConns)
	{
		$Value = $BrokerHypConn.HypHypervisorConnectionUid
		$Key = $BrokerHypConn.Uid
		$HypConnsToBroker[$Key] = $Value
	}
}

$HypConns = @{}
$HypConnPath = "XDHyp:\Connections\"
$HypConnections = Get-ChildItem $HypConnPath
if ($HypConnections -ne $null)
{
	foreach ($HypConn in $HypConnections)
	{
		$Key1 = $HypConn.HypervisorConnectionUid
		$Value1 = $HypConn.ConnectionType.toString() + ";" + $HypConn.HypervisorAddress[0].toString()
		$HypConns[$Key1] = $Value1
	}
}
if($BrokerToHypIDs -ne $null)
{
	$Keys = $BrokerToHypIDs.Keys
	foreach($key2 in $keys)
	{
		$Values = $BrokerToHypIDs[$key2]
		if($Values.Count -ge 1)
		{
			foreach($Value in $Values)
			{
				$hypuobj = Get-BrokerHypervisorConnection -Uid $Value
				if($hypuobj)
				{
					$hypuid = $hypuobj.HypHypervisorConnectionUid
					$output = $HypConns[$hypuid]
					$outputStr = $key2.toString() + "=" + $output
					Write-Host $outputStr
				}
			}
		}
	}
}

Write-Host "[Hypervisor_Type]"
#Support XenServer only
$HypConnPath = "XDHyp:\Connections\"
$HypConnections = Get-ChildItem $HypConnPath
if ($HypConnections -ne $null)
{
	foreach ($HypConn in $HypConnections)
	{
		$ConnUid = $HypConn.HypervisorConnectionUid
		$Value1 = $HypConn.ConnectionType.toString() + ";" + $HypConn.HypervisorAddress[0].toString()
		if ($HypConn.ConnectionType.toString() -ieq "XenServer")
		{
			#$machines = Get-BrokerMachine -SessionSupport 'MultiSession' -HypHypervisorConnectionUid $ConnUid
			$machines = Get-BrokerMachinesPaged -SessionSupport 'MultiSession' -HypHypervisorConnectionUid $ConnUid
			if ($machines -ne $null)
			{
				Write-Host $Value1"=Server"
			}
			else
			{
				Write-Host $Value1"=VDI"
			}
		}
		elseif ($HypConn.ConnectionType.toString() -ieq "vcenter")
		{
			Write-Host $Value1"=vCenter"
		}
	}
}

Write-Host "[ActiveDirectory_server]"
$d =  (gwmi win32_computersystem).domain
$ips = @(([net.dns]::GetHostEntry($d)).AddressList)
if($? -and $ips)
{
	foreach($ip in $ips)
	{
		if($ip.AddressFamily -eq "InterNetwork")
		{
			if($count -eq 0)
			{
				$tips = $ip.IPAddressToString
			}
			else
			{
				$tips = $tips +","+ $ip.IPAddressToString
			}
			$count = $count + 1
		}
	}
	Write-Host $d "=" $tips
}

$ips = ""
Write-Host "[XenApp]"
#$XenApps = Get-BrokerMachine -SessionSupport MultiSession -MaxRecordCount 2147483647
$BrokerDesktopGroup = Get-BrokerDesktopGroup -SessionSupport MultiSession -MaxRecordCount 999
if ($BrokerDesktopGroup  -ne $null)
{
	foreach ($DesktopGroup in $BrokerDesktopGroup)
	{
		$XenApps = Get-BrokerMachinesPaged -SessionSupport MultiSession -DesktopGroupUid $DesktopGroup.Uid
		if($XenApps -ne $null)
		{
			foreach($XenApp in $XenApps)
			{
				$ips = $XenApp.IPAddress
				if($? -and $ips)
				{
					Write-Host $XenApp.ControllerDNSName  "^^"  $XenApp.DNSName "^^" $XenApp.DesktopGroupName "="   $ips
				}
			}
		}
	}
}

Write-Host "[PVS]"
$PVSIPAddresses = @()
$Catalogs = Get-BrokerCatalog -MaxRecordCount 999
if($Catalogs -ne $null)
{
	foreach($Catalog in $Catalogs)
	{
		if ($Catalog.PvsAddress.length -gt 0)
		{
			$PvsAddress = $Catalog.PvsAddress
			if($PVSIPAddresses -Contains $PvsAddress)
			{
			}
			else
			{
				$PVSIPAddresses += $PvsAddress
			}
		}
	}
	Write-Host $PVSIPAddresses
}

Write-Host "[XD_Site]"
$Site = Get-BrokerSite
if ($Site -ne $null)
{

	$SiteName = $Site.Name
	$objIPProperties = [System.Net.NetworkInformation.IPGlobalProperties]::GetIPGlobalProperties()
	$fqdn = "{0}.{1}" -f $env:COMPUTERNAME, $objIPProperties.DomainName
	$ips = ""
	$ips = @(([net.dns]::GetHostEntry($env:COMPUTERNAME)).AddressList)
	$tips = ""
	$count = 0
	if($? -and $ips)
	{
		foreach($ip in $ips)
		{
			if($ip.AddressFamily -eq "InterNetwork")
			{
				if($count -eq 0)
				{
					$tips = $ip.IPAddressToString +":443"
				}
				else
				{
					$tips = $tips + "," + $ip.IPAddressToString+":443"
				}
				$count = $count + 1
			}
		}
		Write-Host $fqdn "^^" $SiteName "=" $tips
	}
}

Remove-PSSnapin citrix*


