#This File is in Unicode format.  Do not edit in an ASCII editor.
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region help text

<#
.SYNOPSIS
			
			To get the General Information of Active Directory
.DESCRIPTION

			This Script is used to Collect the General Information of Active Directory
			Measures are:
			1.Forest name,
			2.Domain name,
			3.NetBIOS name,
			4.Domain SID,
			5.Domain functional level,
			6.Forest functional level,
			7.Recycle bin status,
			8.Schema version,
			9.Tombstone lifetime(Days),
			10.Domains in this forest,
			11.Global catalog servers in the forest,
			12.Additional UPN suffixes

.INPUTS
			CONST_TEST_HDR and CONST_TEST_END
.OUTPUTS
			Collect the Measures of General Information in following format
            <Forest Name><Delimiter><Domain Name><Delimiter><NetBios Name><Delimiter>
            <Domain SID><Delimiter><Domain Functional Level><Delimiter><Forest Functional Level>
            <Delimiter><Recycle Bin Status><Delimiter><Schema Version><Delimiter><Tombstone lifetime>
            <Delimiter><Domains in this forest><Delimiter><Global Catalog servers in the forest><Delimiter>
            <Additional UPN suffixes>
#>

$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False

$global:CONST_TEST_HDR = $args[0]
$global:CONST_TEST_END = $args[1]


import-module ActiveDirectory

Clear-Host

# To get the General Information of Active Directory

Function getADConfigInfo
{
	
	$Forest = Get-ADForest -Current LocalComputer 
	$rootDomain = $Forest.RootDomain
	$allDomains=$Forest.Domains
	$ForestGC=$Forest.GlobalCatalogs
	$UPNsuffix=$Forest.UPNSuffixes
	$ffl = $Forest.ForestMode
	$Domain = Get-ADDomain | select DNSRoot,NetBIOSName,DomainSID,DomainMode
	$Recyclebin = "Feature not supported"
	if ( $ffl -like "Windows2008R2Forest" -or $ffl -like "Windows2012Forest" -or $ffl -like "Windows2012R2Forest" )

	{
		$ADRecBin=(Get-ADOptionalFeature -Server $rootDomain -Identity 766ddcd8-acd0-445e-f3b9-a7f9b6744f2a).EnabledScopes | Measure-Object
		if ( $ADRecBin.Count -ne 0 )
		{
			$Recyclebin="Enabled"
		}
		else
		{
			$Recyclebin="Disabled"
		}
	}
	$SchemaPartition = $Forest.PartitionsContainer.Replace("CN=Partitions","CN=Schema")
	$SchemaVersion = Get-ADObject -Server $rootDomain -Identity $SchemaPartition -Properties * | Select objectVersion
	$forestDN = $Forest.PartitionsContainer.Replace("CN=Partitions,CN=Configuration,","")
	$configPartition = $Forest.PartitionsContainer.Replace("CN=Partitions,","")
	$SchemaVersionStr = "unknown"
	switch ($SchemaVersion.objectVersion)
	{
		13 { $SchemaVersionStr = "13 - Windows 2000 Server" }
		30 { $SchemaVersionStr = "30 - Windows Server 2003"  }
		31 { $SchemaVersionStr = "31 - Windows Server 2003 R2" }
		44 { $SchemaVersionStr = "44- Windows Server 2008" }
		47 { $SchemaVersionStr = "47 - Windows Server 2008 R2" }
		51 { $SchemaVersionStr = "51 - Windows Server 8 Developers Preview" }
		52 { $SchemaVersionStr = "52 - Windows Server 8 Beta" }
		56 { $SchemaVersionStr = "56 - Windows Server 2012" }
		69 { $SchemaVersionStr = "69 - Windows Server 2012 R2" }
		72 { $SchemaVersionStr = "72 - Windows Server Technical Preview" }
		default { $SchemaVersionStr = "- unknown"}
	}

	$ForestMode = "Unknown Forest Functional Level"
	switch ($ffl)
	{

		Windows2000Forest { $ForestMode =  "Windows 2000" }
		Windows2003Forest { $ForestMode = "Windows Server 2003" }
		Windows2008Forest { $ForestMode =  "Windows Server 2008" }
		Windows2008R2Forest { $ForestMode =  "Windows Server 2008 R2" }
		Windows2012Forest { $ForestMode = "Windows Server 2012" }
		Windows2012R2Forest { $ForestMode =  "Windows Server 2012 R2" }
		default { $ForestMode =  "Unknown Forest Functional Level:" + $ffl }
	}

	$tombstoneLifetime = (Get-ADobject -Server $rootDomain -Identity "cn=Directory Service,cn=Windows NT,cn=Services,$configPartition" -Properties tombstoneLifetime).tombstoneLifetime
	$tombstoneDays = 60
	if ($tombstoneLifetime -ne $null)
	{
		$tombstoneDays = $tombstoneLifetime
	}

	if($Domain)
	{
		Write-Host "Forest Name: " $rootDomain
		Write-Host "Domain Name: " $Domain.DNSRoot
		Write-Host "NetBios Name: " $Domain.NetBIOSName
		Write-host "Domain SID: "  $Domain.DomainSID
		Write-Host "Domain Functional Level: " $Domain.DomainMode
		Write-host "Forest Functional Level: " $ForestMode
		Write-Host "Recycle Bin Status: " $Recyclebin
		Write-Host "Schema Version : " $SchemaVersionStr
		Write-Host "Tombstone lifetime: " $tombstoneDays
		Write-Host "Domains in this forest: " $allDomains
		Write-Host "Global Catalog servers in the forest: " $ForestGC | Sort
		Write-Host "Additional UPN suffixes: " $UPNsuffix | Sort
	}
	
}

#Main Region Starts
try
{
	Write-Host $global:CONST_TEST_HDR
	Write-Host "-------------------------"
   
    getADConfigInfo
}
catch [Exception]
{
	$global:isErrorOccured = $True
	Write-Verbose "Catch block executed and following Exception is occured : " $_.Exception.GetType().FullName, $_.Exception.Message
}

If($global:isErrorOccured -eq $False)
{
	Write-Host $global:CONST_TEST_END
}
#Main Region Ends

