# This Script will give all details of XenDesktop
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

$Global:ErrorActionPreference="SilentlyContinue"
Add-pssnapin "Citrix.Common.Commands"
Add-pssnapin "Citrix.Host.Admin.V1"
Add-pssnapin "Citrix.Broker.Admin.V1"
Add-pssnapin "Citrix.ADIdentity.Admin.V1"
Add-pssnapin "Citrix.Configuration.Admin.V1"
Add-pssnapin "Citrix.MachineCreation.Admin.V1"
Add-pssnapin "Citrix.MachineIdentity.Admin.V1"

$FQDN = [System.Net.Dns]::GetHostByName((hostname)).HostName
$SAMName = $ENV:USERDOMAIN + "\" + $ENV:COMPUTERNAME

Write-Host "Site"
Write-Host "----"
#DBUnconfigured = The broker does not have a database connection configured.
#DBRejectedConnection = The database rejected the logon from the Broker Service. This may be caused by bad credentials, or the database
#not being installed.
#InvalidDBConfigured = The database schema is missing (possibly just the stored procedures in it).
#DBNewerVersionThanService = The broker is too old to use the database. A newer version is required.
#DBOlderVersionThanService = The database is too old for the Broker Service. Upgrade the database.
#DBVersionChangeInProgress = A database schema upgrade is in progress.
#OK = The broker is connected to a database that is valid, and the service is running.
#PendingFailure = Connectivity between the Broker Service and the database has been lost. This may be a transitory network error,
#but may indicate a loss of connectivity that requires administrator intervention.
#Failed = Connectivity between the broker and the database has been lost for an extended period of time, or has failed du
#e to a configuration problem. The broker service cannot operate while its connection to the database is unavail
#able.
#Unknown = The Service's status cannot be determined.

$Site = Get-BrokerSite
$BrokerStatus = Get-BrokerServiceStatus
$HypStatus = Get-HypServiceStatus
$ADIdentityStatus = Get-AcctServiceStatus
$ConfigStatus = Get-ConfigServiceStatus
$MachineCreationStatus = Get-ProvServiceStatus
$MachineIdentityStatus = Get-PvsVmServiceStatus
if (($Site -ne $null) -and ($BrokerStatus -ne $null))
{
	if ($HypStatus -eq $null)
	{
		$HypStatus = "-"
	}
	if ($ADIdentityStatus -eq $null)
	{
		$ADIdentityStatus = "-"
	}
	if ($ConfigStatus -eq $null)
	{
		$ConfigStatus = "-"
	}
	if ($MachineCreationStatus -eq $null)
	{
		$MachineCreationStatus = "-"
	}
	if ($MachineIdentityStatus -eq $null)
	{
		$MachineIdentityStatus = "-"
	}
	Write-Host ($Site.Name,$Site.LicenseServerName,$Site.LicenseServerPort,$BrokerStatus,$HypStatus,$ADIdentityStatus,$ConfigStatus,$MachineCreationStatus,$MachineIdentityStatus) -Separator ";"
}


Write-Host "Hypervisor Details"
Write-Host "------------------"
# HyperConn.State
# Unavailable - The broker is unable to contact the hypervisor.
# InMaintenanceMode - The hosting server is in maintenance mode.
# On - The broker is in contact with the hypervisor.

$HyperConns = Get-BrokerHypervisorConnection -State On -PreferredController $SAMName
if($HyperConns -ne $null)
{
	foreach($HyperConn in $HyperConns)
	{
		$b = "XDHyp:\Connections\"
		$a = $b + $HyperConn.Name
		$HypervisorIP = "-"
		if ((Get-HypXenServerAddress -Literalpath $a).length -gt 0)
		{
			$HypervisorIP = Get-HypXenServerAddress -Literalpath $a
		}
		Write-Host ($HyperConn.Name,$HyperConn.State,$HypervisorIP) -Separator ";"
	}
}


Write-Host "Controllers"
Write-Host "-----------"
# Controller.State
# Valid values are: Failed, Off, On, and Active

$Controllers = Get-BrokerController
if($Controllers -ne $null)
{
	foreach($Controller in $Controllers)
	{
		Write-Host($Controller.DNSName,$Controller.State,$Controller.DesktopsRegistered,
		$Controller.LastActivityTime,$Controller.LastStartTime,$Controller.ActiveSiteServices) -Separator ";"
	}
}


Write-Host "Catalogs"
Write-Host "--------"
#AllocationType = Permanent or Random
#CatalogKind = Thin cloned and Single image,PowerManaged,Unmanaged and PVS

$Catalogs = Get-BrokerCatalog -MaxRecordCount 2147483647
if($Catalogs -ne $null)
{
	foreach($Catalog in $Catalogs)
	{
		$MachinesArePhysical = "-"
		$PvsAddress = "-"
		$PvsDomain = "-"
		if ($Catalog.MachinesArePhysical.length -gt 0)
		{
			$MachinesArePhysical = $Catalog.MachinesArePhysical
		}
		if ($Catalog.PvsAddress.length -gt 0)
		{
			$PvsAddress = $Catalog.PvsAddress
		}
		if ($Catalog.PvsDomain.length -gt 0)
		{
			$PvsDomain = $Catalog.PvsDomain
		}
		Write-Host ($Catalog.Name,$Catalog.AllocationType,$Catalog.AssignedCount,$Catalog.AvailableAssignedCount,
		$Catalog.AvailableCount,$Catalog.AvailableUnassignedCount,$Catalog.UnassignedCount,$Catalog.UsedCount,$Catalog.CatalogKind,
		$MachinesArePhysical,$PvsAddress,$PvsDomain ) -Separator ";"
	}
}


Write-Host "Broker Machines"
Write-Host "---------------"
Write-Host " "
 

Write-Host "Desktop Groups"
Write-Host "--------------"

$DesktopGroups = Get-BrokerDesktopGroup -MaxRecordCount 2147483647
if($DesktopGroups -ne $null)
{
	foreach($DesktopGroup in $DesktopGroups)
	{
		Write-Host ($DesktopGroup.Name,$DesktopGroup.Enabled,$DesktopGroup.InMaintenanceMode,$DesktopGroup.TotalDesktops,
		$DesktopGroup.DesktopsAvailable,$DesktopGroup.DesktopsDisconnected,$DesktopGroup.DesktopsInUse,
		$DesktopGroup.DesktopsNeverRegistered,$DesktopGroup.DesktopsUnregistered) -Separator ";"
	}
}


Write-Host "Virtual Desktops"
Write-Host "----------------"
# RegistrationState"
# Valid values are Registered, Unregistered, and AgentError.
# "SummaryState"
# Valid values are Off, Unregistered, Available, Disconnected, and InUse

$BrokerDesktops = Get-BrokerDesktop -MaxRecordCount 2147483647 -ControllerDNSName $FQDN
if ($BrokerDesktops -ne $null)
{
	foreach($BrokerDesktop in $BrokerDesktops)
	{	
		$IPAddress = "-"
		$OSType = "-"
		$HostedMachineName = "-"
		$HostingServerName = "-"
		$HypervisorConnectionName = "-"
		$SessionUserName = "-"
		$ControllerDNSName = "-"
		$DesktopConditions = "-"
		$LastConnectionFailure = "-"
		$LastConnectionTime = "-"
		$LastConnectionUser = "-"
		$AssociatedUserName = "-"
		$ImageOutOfDate = "-"
		if ( $BrokerDesktop.IPAddress.length -gt 0)
		{
			$IPAddress = $BrokerDesktop.IPAddress
		}
		if ($BrokerDesktop.OSType.length -gt 0)
		{
			$OSType = $BrokerDesktop.OSType
		}
		if ($BrokerDesktop.HostedMachineName.length -gt 0)
		{
			$HostedMachineName = $BrokerDesktop.HostedMachineName
		}
		if ($BrokerDesktop.HostingServerName.length -gt 0)
		{
			$HostingServerName = $BrokerDesktop.HostingServerName
		}
		if ($BrokerDesktop.HypervisorConnectionName.length -gt 0)
		{
			$HypervisorConnectionName = $BrokerDesktop.HypervisorConnectionName
		}
		if ($BrokerDesktop.SessionUserName.length -gt 0)
		{
			$SessionUserName = $BrokerDesktop.SessionUserName
		}
		if ($BrokerDesktop.ControllerDNSName.length -gt 0)
		{
			$ControllerDNSName = $BrokerDesktop.ControllerDNSName
		}
		if ($BrokerDesktop.DesktopConditions.length -gt 0)
		{
			$DesktopConditions = $BrokerDesktop.DesktopConditions
		}
		if ($BrokerDesktop.LastConnectionFailure -ne $null)
		{
			$LastConnectionFailure = $BrokerDesktop.LastConnectionFailure
		}
		if ($BrokerDesktop.LastConnectionTime -ne $null)
		{
			$LastConnectionTime = $BrokerDesktop.LastConnectionTime
		}
		if ($BrokerDesktop.LastConnectionUser -gt 0)
		{
			$LastConnectionUser = $BrokerDesktop.LastConnectionUser
		}
		if ($BrokerDesktop.AssociatedUserName -ne $null -and $BrokerDesktop.AssociatedUserName.length -gt 0)
		{
			$AssociatedUsers = $BrokerDesktop.AssociatedUserName
			$AssociatedUser = ""
			$count = 0
			foreach ($key in $AssociatedUsers)
			{
				if ($count -eq 0)
				{
					$AssociatedUser = $key
				}
				else
				{
					$AssociatedUser = $AssociatedUser + "," + $key
				}
				$count = $count + 1
			}
			if ($AssociatedUser -eq $null -or $AssociatedUser.length -eq 0)
			{
				$AssociatedUserName = "-"
			}
			else
			{
				$AssociatedUserName = $AssociatedUser
			}
		}
		if ($BrokerDesktop.ImageOutOfDate -ne $null)
		{
			$ImageOutOfDate = $BrokerDesktop.ImageOutOfDate
		}
		Write-Host ($BrokerDesktop.MachineName,$BrokerDesktop.DNSName,$IPAddress,$OSType,$BrokerDesktop.PowerState,
		$BrokerDesktop.RegistrationState,$BrokerDesktop.SummaryState,$BrokerDesktop.DesktopGroupName,$BrokerDesktop.DesktopKind,
		$HostedMachineName,$HostingServerName,$HypervisorConnectionName,$BrokerDesktop.InMaintenanceMode,$SessionUserName,
		$BrokerDesktop.CatalogName,$ControllerDNSName,$LastConnectionFailure,$ImageOutOfDate,$LastConnectionTime,$LastConnectionUser,$AssociatedUserName,$DesktopConditions) -Separator ";"
	}
}

Write-Host "Sessions"
Write-Host "--------"
# SessionState
# Valid values are Other, PreparingSession, Connected, Active, Disconnected, Reconnecting, NonBrokeredSession, and Unknown.

if ($BrokerDesktops -ne $null)
{
	foreach($BrokerDesktop in $BrokerDesktops)
	{
		if ($BrokerDesktop -ne $null)
		{
			$IPAddress = "-"
			$OSType = "-"
			
			if ( $BrokerDesktop.IPAddress.length -gt 0)
			{
				$IPAddress = $BrokerDesktop.IPAddress
			}
			if ($BrokerDesktop.OSType.length -gt 0)
			{
				$OSType = $BrokerDesktop.OSType
			}
			$BrokerSessions = Get-BrokerSession -MaxRecordCount 2147483647 -DesktopUid $BrokerDesktop.Uid
			if ($BrokerSessions -ne $null)
			{
				foreach($BrokerSession in $BrokerSessions)
				{
					$BrokeringUserName = "-"
					$BrokeringTime = "-"
					$ConnectedViaHostName = "-"
					$ClientAddress ="-";
					$ClientName = "-"
					$ClientVersion = "-"
					$Protocol = "-"
					$ControllerDNSName = "-"
					$BrokeringUserName = $BrokerSession.UserName
					if ($BrokeringUserName -eq $null -or $BrokeringUserName.length -eq 0)
					{
						$BrokeringUserName = $BrokerSession.BrokeringUserName
						if ($BrokeringUserName -eq $null -or $BrokeringUserName.length -eq 0)
						{
							$BrokeringUserName = "-"
						}
					}
					if ($BrokerSession.ClientAddress.length -gt 0)
					{
						$ClientAddress = $BrokerSession.ClientAddress
					}
					if ($BrokerSession.ClientName.length -gt 0)
					{
						$ClientName = $BrokerSession.ClientName
					}
					if ($BrokerSession.ClientVersion.length -gt 0)
					{
						$ClientVersion = $BrokerSession.ClientVersion
					}
					if ($BrokerSession.BrokeringTime -ne $null)
					{
						$BrokeringTime = $BrokerSession.BrokeringTime
					}
					if ($BrokerSession.ConnectedViaHostName -ne $null)
					{
						$ConnectedViaHostName = $BrokerSession.ConnectedViaHostName
					}
					if ($BrokerSession.Protocol.length -gt 0)
					{
						$Protocol = $BrokerSession.Protocol
					}
					if ($BrokerDesktop.ControllerDNSName.length -gt 0)
					{
						$ControllerDNSName = $BrokerDesktop.ControllerDNSName
					}
					if ($BrokerDesktop.LicenseId.length -gt 0)
					{
						$LicenseId = $BrokerDesktop.LicenseId
					}
					Write-Host ($BrokerDesktop.MachineName,$BrokerDesktop.DNSName,$IPAddress,$OSType,$BrokeringUserName,
					$ClientAddress,$ClientName,$ClientVersion,$ConnectedViaHostName,
					$Protocol,$BrokerSession.SessionState,$BrokerSession.StartTime,$BrokeringTime,$LicenseId,$ControllerDNSName) -Separator ";"
				}
			}
		}
	}
}


Write-Host "Application Instances"
Write-Host "---------------------"
$AppInstances = Get-BrokerApplicationInstance -MaxRecordCount 2147483647
if ($AppInstances -ne $null)
{
	foreach($AppInstance in $AppInstances)
	{
		$BrokeringTime = "-"
		$BrokerDesktop = Get-BrokerDesktop -SessionUid $AppInstance.SessionUid
		$session = Get-BrokerSession -Uid $AppInstance.SessionUid
		if ($session.BrokeringTime -ne $null)
		{
			$BrokeringTime = $session.BrokeringTime
		}

		$application = Get-BrokerApplication -Uid $AppInstance.ApplicationUid
		$folder = Get-BrokerApplicationFolder -Uid $application.FolderUid
		if (($session -ne $null) -and ($application -ne $null))
		{
			if($folder -ne $null)
			{
				$folderName = $folder.Name
				if ($folderName -ne $null -and $folerName.length -gt 0)
				{
					$name = $folderName + "\" + $application.DisplayName
				}
				else
				{
					$name = $application.DisplayName
				}
			}
			else
			{
				$name = $application.DisplayName
			}
			Write-Host($BrokerDesktop.MachineName,$name,$application.CommandLineExecutable,$session.BrokeringUserName,$session.ClientAddress,$BrokerSession.Protocol,$BrokerSession.SessionState,$BrokerSession.StartTime,$BrokeringTime,$BrokerDesktop.CatalogName,$BrokerDesktop.LicenseId) -Separator ";"
		}
	}
}


Write-Host "Unavailable Virtual Desktops"
Write-Host "----------------------------"
if($DesktopGroups -ne $null)
{
	foreach($DesktopGroup in $DesktopGroups)
	{
		$dirtyVMs = Get-BrokerDesktop -DesktopGroupUid $DesktopGroup.Uid -PowerActionPending $false -PowerState On -SummaryState Available -WillShutdownAfterUse $true -MaxRecordCount 2147483647
		if($dirtyVMs ne $null)
		{
			foreach($dirtyVM in $dirtyVMs)
			{
				$HostedMachineName = "-"
				$IPAddress = "-"
				$OSType = "-"
				$HostingServerName = "-"
				$HypervisorConnectionName = "-"
				$ControllerDNSName = "-"
				if ( $dirtyVM.IPAddress.length -gt 0)
				{
					$IPAddress = $dirtyVM.IPAddress
				}
				if ($dirtyVM.OSType.length -gt 0)
				{
					$OSType = $dirtyVM.OSType
				}
				if ($dirtyVM.HostedMachineName.length -gt 0)
				{
					$HostedMachineName = $dirtyVM.HostedMachineName
				}
				if ($dirtyVM.HostingServerName.length -gt 0)
				{
					$HostingServerName = $dirtyVM.HostingServerName
				}
				if ($dirtyVM.HypervisorConnectionName.length -gt 0)
				{
					$HypervisorConnectionName = $dirtyVM.HypervisorConnectionName
				}
				if ($dirtyVM.ControllerDNSName.length -gt 0)
				{
					$ControllerDNSName = $dirtyVM.ControllerDNSName
				}
				Write-Host($dirtyVM.MachineName,$dirtyVM.DNSName,$IPAddress,$OSType,$dirtyVM.DesktopGroupName,$dirtyVM.DesktopKind,$dirtyVM.CatalogName,$HostedMachineName,$HostingServerName,$HypervisorConnectionName,$ControllerDNSName) -Separator ";"
			}
		}
	}
}

Remove-pssnapin "Citrix.Common.Commands"
Remove-pssnapin "Citrix.Host.Admin.V1"
Remove-pssnapin "Citrix.Broker.Admin.V1"
Remove-pssnapin "Citrix.ADIdentity.Admin.V1"
Remove-pssnapin "Citrix.Configuration.Admin.V1"
Remove-pssnapin "Citrix.MachineCreation.Admin.V1"
Remove-pssnapin "Citrix.MachineIdentity.Admin.V1"


# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUYTYoMfrQEvJ/qS8lcq7XZZuz
# 2a6gghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFMWPxo4gBk2yjqtN
# R+2220sWgLqsMA0GCSqGSIb3DQEBAQUABIICAFZlFoo1wx7hiTjiLu/5ctD0ZMug
# QU7kaz2CBeEaCa6pyhrmJ4qhHDnaxzLP1gGGsvI+gXu3z8C4vUOmqunqdVrbUpyp
# 7rjZ07chDb8R+v4RLF+6ka1Gr47su5OWSO7XWtHz905l5Yinv/V6qtTxMdFkwku8
# vQRfkK94bzNLQ/oeKMc5PAjWLN/j4tmPY1W9B3ENU60EY7GujZ9B80oiEb2MahaV
# LkfPg+EAz4l+NBg7D7NDsG8XrMTG+aENTa/hxUTrrfn1iYUC+l1SmxRtqtscYr57
# pF/E+SB76R3/Et24+cNkd7PbSCvy7itzZdoTScyXhasuLBqEGSqlBixfrh2bOXWa
# XSz/EFecspeWNEm0oKeRQq0sWCCrWaacAJhEETSbebwPoPRZaLC5cldxVl73jCbG
# 6JVrxifip649B+9o1C9d2GGE4St/Xfkgdm2a2vXUvM1nMtC29vNx+DWun+iE1rVl
# 5PZ1S9/kqllbjIc7zkcz8uitbtpIavK5y3K0Btk4Yd4QawfNW8+PrJBt9ko4HWqF
# A98pACNzPY0/bZszucQBJr/5sZ+Bi++fPz8MIRtqUdQiGiEdS1GIZDiUPrLRku2U
# vFMjVZbYzwxMdHCLpKvHZ697XPPa8gf1VaSLdLEh3ZRJhCktGMA3BRGTKuhpgZmU
# 3tY7TxCwCgetE0MkoYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MTA5MDExMjM3MTZaMC8GCSqGSIb3DQEJBDEiBCBGuqaoPsrjcFELLuHzpQfhPvIv
# 7ho1cb51h0rHlUuGXDANBgkqhkiG9w0BAQEFAASCAQBkx21EnerwQ3AM08qCK9Ci
# Lk3MUzoRzq6MMZSZ6ut1cpo1cPhPKsmEtZmnCajdMJMlRDGlXrpzrODbyqgXP4OT
# N1379gQ+pvL0AiXx79EOWeQHGoa53g4PXWUt+FJeXHJsPkClvyE9zsSxLF/P7g4J
# rHIoUBwBHzhElgtAcRi9wWIaUtH85HDYHynBwoh5Cp3ub+KkAUc3lrOjK1lLBtx9
# 9NyKWCO8sog0LhYWJNSYmza0eKpyN9TMj608UfWdzX9Piye85RFsq1xaAFJfVH+u
# 5FNREAgCXspA64Xt1HBkbYoBjOcA4cvj7bhpG1feEtKMhwusQ/bs9+8s4rFAOcXp
# SIG # End signature block
