$Global:ErrorActionPreference="SilentlyContinue"
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

Add-PSSnapin "Citrix.Common.Commands"
Add-PSSnapin "Citrix.Host.Admin.V2"
Add-PSSnapin "Citrix.Broker.Admin.V2"
Add-PSSnapin "Citrix.ADIdentity.Admin.V2"
Add-PSSnapin "Citrix.Configuration.Admin.V2"
Add-PSSnapin "Citrix.MachineCreation.Admin.V2"
Add-PSSnapin "Citrix.Licensing.Admin.V1"
Add-PSSnapin "Citrix.DelegatedAdmin.Admin.V1"
Add-PSSnapin "Citrix.Monitor.Admin.V1"
Add-PSSnapin "Citrix.ConfigurationLogging.Admin.V1"
Add-PSSnapin "Citrix.AppLibrary.Admin.V1"

<#
Function Get-BrokerApplicationsPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($applications.count / $total * 100, 1)
		#Write-Host  "Getting Broker Applications. Retrieved $($applications.count) of $total applications completed in $remainingPerc%"
		$last = $applications[-1].Uid
		$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { Uid -gt $last } @argumentList
	}
	return $applications
}
#>

Function Get-BrokerApplicationsPaged
{
	param (
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$LastUid = 0;
	do
	{
		$NewApplications = @(Get-BrokerApplication -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewApplications.Count -gt 0)
		{
			$applications += $NewApplications
			$LastUid = $applications[-1].Uid
		}
	} while ($NewApplications.Count -ge $MaxRecordCount)
	return $applications
}

$ZoneControllersHash = @{}
$ConfiguredZones = Get-ConfigZone -EA 0
foreach($Zone in $ConfiguredZones)
{
	$ZoneControllersHash.Add($Zone.Name,$Zone.ControllerSids)
}
Write-Host "Sites"
Write-Host "-----"
$site = Get-BrokerSite
if($site -ne $null)
{
	if($site.BaseOU -eq $null)
	{
		$ADOU = "-"
	}
	else
	{
		$ADOU = $site.BaseOU
	}

	Write-Host ($site.Name,$ADOU,$site.DnsResolutionEnabled,$site.TrustRequestsSentToTheXmlServicePort,$site.SecureIcaRequired,$site.ConnectionLeasingEnabled) -Separator "#~#"
}

Write-Host "Administrators"
Write-Host "--------------"
$brokerAdmins = Get-AdminAdministrator
if($brokerAdmins -ne $null)
{
	foreach($brokerAdmin in $brokerAdmins)
	{
		$RightsArr = $brokerAdmin.Rights
		$RoleName = $RightsArr[0].RoleName
		$ScopeName = $RightsArr[0].ScopeName
		$RoleObject = Get-AdminRole $RoleName
		$PermissionsArr = $RoleObject.Permissions
		$ScopeObject = Get-AdminScope $ScopeName
		$count = $PermissionsArr.Count
		for($c=0;$c -le $count; $c++)
		{
			if($c -eq 0)
			{
				$permissions = $PermissionsArr[$c]
			}
			else
			{
				$permissions = $permissions + ", " + $PermissionsArr[$c]
			}
		}

		if ($RoleName -eq $null -or $RoleName.length -eq 0)
		{
			$RoleName = "-"
		}

		if ($permissions -eq $null -or $permissions.length -eq 0)
		{
			$permissions = "-"
		}

		if ($ScopeName -eq $null -or $ScopeName.length -eq 0)
		{
			$ScopeName = "-"
		}
		Write-Host ($brokerAdmin.Name,$brokerAdmin.Enabled,$ScopeName,$RoleName,$PermissionsArr.BuiltIn,
		$permissions,$ScopeObject.BuiltIn) -Separator "#~#"
	}
}

Write-Host "Controllers"
Write-Host "------------"
$controllers = Get-BrokerController
if($controllers -ne $null)
{
	foreach($controller in $controllers)
	{
		$servicesArr = $controller.ActiveSiteServices
		$count = $servicesArr.Count
		$ControllerSid = $controller.SID
		for($c=0;$c -le $count; $c++)
		{
			if($c -eq 0)
			{
				$services = $servicesArr[$c]
			}
			else
			{
				$services = $services + "," + $servicesArr[$c]
			}
		}
		$ControllerZoneName = "-"
		if($ZoneControllersHash.Count -gt 0)
		{
			$ZoneKeys = $ZoneControllersHash.KEYS.GetEnumerator()
			foreach($ZoneKey in $ZoneKeys)
			{
				$ZoneValues = $ZoneControllersHash[$ZoneKey]
				if($ZoneValues.Contains($ControllerSid) -eq $true)
				{
					$ControllerZoneName = $ZoneKey
				}
			}
		}
		Write-Host ($controller.DNSName, $controller.ControllerVersion, $controller.DesktopsRegistered, $controller.State, $services,
				$ControllerZoneName) -Separator "#~#"
	}
}

Write-Host "Hosts"
Write-Host "-----"
$hyphosts = get-BrokerHypervisorConnection
if($hyphosts -ne $null)
{
	foreach($hyphost in $hyphosts)
	{
		$a = "XDHyp:\Connections\"
		$b = $a + $hyphost.Name
		$CapabilityList = $hyphost.Capabilities

		if ($CapabilityList -eq $null -or $CapabilityList.length -eq 0)
		{
			$Capabilities = "-"
		}
		else
		{
			$Capabilities = ""
			foreach($capability in $CapabilityList)
			{
				if ($capability -ne $null -or $capability.length -ne 0)
				{
					$Capabilities = $Capabilities + "," +$capability
				}
			}
			if ($Capabilities.startsWith(","))
			{
				$Capabilities = $Capabilities.substring(1)
			}
		}
		$HypObject = Get-Item $b
		if($?  -and $HypObject)
		{
			$ConnectionType = $HypObject.ConnectionType
			$HypervisorAddress = $HypObject.HypervisorAddress
			$HypervisorConnectionName = $HypObject.HypervisorConnectionName
			$MaintenanceMode = $HypObject.MaintenanceMode
			$Metadata = $HypObject.MetadataMap


			if ($Metadata -eq $null -or $Metadata.length -eq 0)
			{
				$metadataList = "-"
			}
			else
			{
				$metadataList = ""
				foreach($key in $Metadata.KEYS)
				{
					$temp = $key + "=" + $Metadata.item($key)
					$metadataList = $metadataList + ", " + $temp
				}
				if ($metadataList.startsWith(","))
				{
					$metadataList = $metadataList.substring(1)
				}
			}
			$Persistent = $HypObject.Persistent
			$SupportsLocalStorageCaching = $HypObject.SupportsLocalStorageCaching
			$SupportsPvsVMs = $HypObject.SupportsPvsVMs
			$UserName = $HypObject.UserName
			$UsesCloudInfrastructure = $HypObject.UsesCloudInfrastructure
		}
		$HypHostName = $hyphost.Name
		$PreferredController = $hyphost.PreferredController
		$State = $hyphost.State

		Write-Host ($HypHostName, $ConnectionType, $HypervisorAddress, $State, $PreferredController,  $Capabilities,
		 $HypervisorConnectionName, $MaintenanceMode, $metadataList, $Persistent, $SupportsLocalStorageCaching,
		$SupportsPvsVMs, $UserName, $UsesCloudInfrastructure) -Separator "#~#"
	}
}

Write-Host
Write-Host "Licensing"
Write-Host "---------"
$LicensingGraceHoursLeft = $site.LicensingGraceHoursLeft
if($LicensingGraceHoursLeft -eq $null -or $LicensingGraceHoursLeft.length -eq 0)
{
	$LicensingGraceHoursLeft = "-"
}
Write-Host ($site.Name,$site.LicenseServerName, $site.LicenseServerPort, $site.LicenseEdition, $site.LicenseModel,
		$site.LicensingBurnIn, $site.LicensingGracePeriodActive ,$site.LicensingOutOfBoxGracePeriodActive, $LicensingGraceHoursLeft) -Separator "#~#"


Write-Host "XenDesktop Licenses"
Write-Host "--------------------"
$LicenseCertHash = $(Get-LicCertificate -AdminAddress $site.LicenseServerName).CertHash
$licenses = get-licinventory -adminaddress $site.LicenseServerName -CertHash $LicenseCertHash
if($licenses)
{
	foreach($license in $licenses)
	{
		$ProductName = $license.LocalizedLicenseProductName
		$Model = $license.LocalizedLicenseModel
		if(($Model -eq $null) -or ($Model.length -le 0))
		{
			$Model = "-"
		}
		$ExpirationDate = $license.LicenseExpirationDate
		if(($ExpirationDate -eq $null) -or ($ExpirationDate.length -le 0))
		{
			$ExpirationDate = "-"
		}

		$SubscriptionAdvantageDate = $license.LicenseSubscriptionAdvantageDate
		$LicenseType = $license.LocalizedLicenseType
		$Total = $license.LicensesAvailable

		Write-Host ($ProductName, $Model, $ExpirationDate, $SubscriptionAdvantageDate, $LicenseType, $Total) -Separator "#~#"
	}
}

Write-Host "Catalogs"
Write-Host "---------"
$catalogs = Get-BrokerCatalog -MaxRecordCount 2147483647
if($catalogs -ne $null)
{
	foreach($catalog in $catalogs)
	{
		$Name = $catalog.Name
		$Description = $catalog.Description
		$SessionSupport = $catalog.SessionSupport
		$IsPhysical = $catalog.MachinesArePhysical
		$IsRemotePC = $catalog.IsRemotePC


		$MachineType = "-"
		if($SessionSupport -eq 'SingleSession')
		{
			$MachineType = "Windows Desktop OS"
		}
		else
		{
			$MachineType = "Windows Server OS"
		}
		if($IsPhysical)
		{
			$MachineType = $MachineType + " (Physical)"
		}
		else
		{
			$MachineType = $MachineType + " (Virtual)"
		}

		$ProvisioningMethod = "-"
		if($catalog.ProvisioningType -eq "MCS")
		{
			$ProvisioningMethod = "Machine creation services"
		}
		else
		{
			$ProvisioningMethod = $catalog.ProvisioningType
		}

		$AllocationType = "-"
		if($catalog.AllocationType -eq "Static")
		{
			$AllocationType = "Permanent"
		}
		else
		{
			$AllocationType = $catalog.AllocationType
		}

		$Scopes = "-"
		if($catalog.Scopes)
		{
			$Scopes = $catalog.Scopes
		}
		else
		{
			$Scopes = "All"
		}

		$UserData = "-"
		if($catalog.PersistUserChanges -eq "OnLocal")
		{
			$UserData = "On local disk"
		}
		elseif($catalog.PersistUserChanges -eq "OnPvd")
		{
			$UserData = "On personal vDisk"
		}
		else
		{
			$UserData = "Discarded"
		}
		$PvsAddress = $catalog.PvsAddress
		$PvsDomain = $catalog.PvsDomain

		if ($Description -eq $null -or $Description.length -eq 0)
		{
			$Description = "-"
		}
		Write-Host ($Name, $Description,$MachineType, $ProvisioningMethod, $AllocationType, $Scopes, $UserData,$IsRemotePC,$PvsAddress,
				$PvsDomain, $catalog.ZoneName) -Separator "#~#"
	}
}

Write-Host "Delivery Groups configuration"
Write-Host "-----------------------------"
$DeliveryGroupUUIDHash = @{}
$desktopGroups = Get-BrokerDesktopGroup -MaxRecordCount 2147483647
if($desktopGroups -ne $null)
{
	foreach($group in $desktopGroups)
	{
		$DeliveryGroupUUID = $group.UUID
		$DeliveryType = $group.DeliveryType
		$DesktopKind = $group.DesktopKind
		$MaintenanceMode = $group.InMaintenanceMode
		$AutomaticPowerOnForAssigned = $group.AutomaticPowerOnForAssigned
		$AutomaticPowerOnForAssignedDuringPeak = $group.AutomaticPowerOnForAssignedDuringPeak
		$ShutdownDesktopsAfterUse = $group.ShutdownDesktopsAfterUse
		$PublishedName = $group.PublishedName
		$Name = $group.Name
		$Enabled = $group.Enabled
		$ColorDepth = $group.ColorDepth
		$TimeZone = $group.TimeZone
		$EnableSecureICA = $group.SecureIcaRequired
		$OffPeakBufferSizePercent = $group.OffPeakBufferSizePercent
		$OffPeakDisconnectAction = $group.OffPeakDisconnectAction
		$OffPeakDisconnectTimeout = $group.OffPeakDisconnectTimeout
		$OffPeakExtendedDisconnectAction = $group.OffPeakExtendedDisconnectAction
		$OffPeakExtendedDisconnectTimeout = $group.OffPeakExtendedDisconnectTimeout
		$OffPeakLogoffAction = $group.OffPeakLogOffAction
		$OffPeakLogoffTimeout = $group.OffPeakLogOffTimeout
		$PeakBufferSizePercent = $group.PeakBufferSizePercent
		$PeakDisconnectAction = $group.PeakDisconnectAction
		$PeakDisconnectTimeout = $group.PeakDisconnectTimeout
		$PeakExtendedDisconnectAction = $group.PeakExtendedDisconnectAction
		$PeakExtendedDisconnectTimeout = $group.PeakExtendedDisconnectTimeout
		$PeakLogoffAction = $group.PeakLogOffAction
		$PealLogoffTimeout = $group.PeakLogOffTimeout

		$DeliveryGroupUUIDHash.Add($DeliveryGroupUUID,$Name)

		$Scopes = "-"
		$DGscopes = $group.Scopes
		if($DGscopes)
		{
			foreach($scope in $DGscopes)
			{
				$Scopes = $Scopes+", "+$scope
			}
			if($Scopes.startsWith(","))
			{
				$Scopes = $Scopes.substring(1)
			}
		}
		else
		{
			$Scopes = "All"
		}

		$SecureIcaRequired = $group.SecureIcaRequired
		$SessionSupport = $group.SessionSupport
		$MachineType = "-"
		if($SessionSupport -eq 'SingleSession')
		{
			$MachineType = "Windows Desktop OS"
		}
		else
		{
			$MachineType = "Windows Server OS"
		}

		$AppDiskUIDs = $group.AppDisks
		if($AppDiskUIDs -eq $null -or $AppDiskUIDs.count -eq 0)
		{
			$AppDiskNames = "-"
		}
		else
		{
			$AppDiskNames = ""
			foreach($AppDiskUID in $AppDiskUIDs)
			{
				if($AppDiskUidMap.ContainsKey($AppDiskUID) -eq $True)
				{
					$AppDiskNames += $AppDiskUidMap[$AppDiskUID]
				}
			}
		}
		$TurnOnAddedMachine = $group.TurnOnAddedMachine

		Write-Host ($Name,$DeliveryType, $DesktopKind, $Enabled, $MaintenanceMode, $PublishedName,
		$ColorDepth, $TimeZone, $EnableSecureICA, $Scopes, $SecureIcaRequired, $MachineType, $AutomaticPowerOnForAssigned,
		$AutomaticPowerOnForAssignedDuringPeak, $ShutdownDesktopsAfterUse, $OffPeakBufferSizePercent, $OffPeakDisconnectAction,
		$OffPeakDisconnectTimeout, $OffPeakExtendedDisconnectAction, $OffPeakExtendedDisconnectTimeout, $OffPeakLogoffAction,
		$OffPeakLogoffTimeout, $PeakBufferSizePercent, $PeakDisconnectAction, $PeakDisconnectTimeout, $PeakExtendedDisconnectAction,
		$PeakExtendedDisconnectTimeout, $PeakLogoffAction, $PealLogoffTimeout, $TurnOnAddedMachine, $AppDiskNames) -Separator "#~#"
	}
}


Write-Host "Application"
Write-Host "------------"
#$PublishedAppInstances = Get-BrokerApplication -MaxRecordCount 2147483647
$PublishedAppInstances = Get-BrokerApplicationsPaged
if ($PublishedAppInstances -ne $null)
{
	foreach($PublishedAppInstance in $PublishedAppInstances)
	{
		
		$AdminFolderName = $PublishedAppInstance.AdminFolderName
		$ApplicationName = $PublishedAppInstance.ApplicationName
		$ApplicationType = $PublishedAppInstance.ApplicationType
		$BrowserName = $PublishedAppInstance.BrowserName
		$AssociatedDesktopGroupUUIDs = $PublishedAppInstance.AssociatedDesktopGroupUUIDs
		
		if($AssociatedDesktopGroupUUIDs -eq $null -or $AssociatedDesktopGroupUUIDs.count -eq 0)
		{
			$DeliveryGroups = "-"
		}
		else
		{
			$DeliveryGroups = ""
			foreach($DesktopGroupUUID in $AssociatedDesktopGroupUUIDs)
			{
				$DGName = $DeliveryGroupUUIDHash.Get_Item($DesktopGroupUUID)
				if ($Description -ne $null -and $Description.length -ne 0)
				{
					$DeliveryGroups = $DeliveryGroups + "," + $DGName
				}
			}
			if($DeliveryGroups.startsWith(","))
			{
				$DeliveryGroups = $DeliveryGroups.substring(1)
			}
		}

		$Description = $PublishedAppInstance.Description
		if ($Description -eq $null -or $Description.length -eq 0)
		{
			$Description = "-"
		}
		$DisplayName = "-"
		$WorkingDirectory = $PublishedAppInstance.WorkingDirectory
		if ($WorkingDirectory -eq $null -or $WorkingDirectory.length -eq 0)
		{
			$WorkingDirectory = "-"
		}

		$CommandLineExecutable = $PublishedAppInstance.CommandLineExecutable
		$CommandLineArgs = $PublishedAppInstance.CommandLineArguments
		$AppVCommandLineArgs = "-"
		if ($CommandLineExecutable -ne $null -and ($CommandLineExecutable -ieq "CtxAppVLauncher.exe"))
		{
			$MetadataMap = $PublishedAppInstance.MetadataMap
			if ($MetadataMap -ne $null)
			{
				$AppID = $MetadataMap.get_Item("Citrix_DesktopStudio_AppID")
				$PkgID = $MetadataMap.get_Item("Citrix_DesktopStudio_PackageId")
				if ($AppID -ne $null -and $AppID.length -gt 0 -and $PkgID -ne $null -and $PkgID.length -gt 0)
				{
					$Idx = $PkgID.LastIndexOf("_")
					if ($Idx -gt 0)
					{
						$PkgID = $PkgID.Substring(0,$Idx)
					}
					$AppVCommandLineArgs = "CtxAppVLauncher.exe" + "#" + $AppID + "#" + $PkgID
				}
			}
		}
		if ($CommandLineArgs)
		{
			$CommandLineExecutable = $CommandLineExecutable + " " + $CommandLineArgs
		}

		# $NameAndBrowserName = $BrowserName + " (" +$PublishedAppInstance.Name + ")"
		$ClientFolder = $PublishedAppInstance.ClientFolder
		$CpuPriorityLevel = $PublishedAppInstance.CpuPriorityLevel
		$IconFromClient = $PublishedAppInstance.IconFromClient
		$PublishedName = $PublishedAppInstance.PublishedName
		$SecureCmdLineArgumentsEnabled = $PublishedAppInstance.SecureCmdLineArgumentsEnabled
		$ShortcutAddedToDesktop = $PublishedAppInstance.ShortcutAddedToDesktop
		$ShortcutAddedToStartMenu = $PublishedAppInstance.ShortcutAddedToStartMenu
		$UUID = $PublishedAppInstance.UUID
		$UserFilterEnabled = $PublishedAppInstance.UserFilterEnabled
		$WaitForPrinterCreation = $PublishedAppInstance.WaitForPrinterCreation

		Write-Host ($PublishedAppInstance.Name, $Description, $PublishedName, $AdminFolderName, $ApplicationName, 
		$PublishedAppInstance.ApplicationType,$PublishedAppInstance.CpuPriorityLevel, $ClientFolder,
		$CommandLineExecutable, $AppVCommandLineArgs, $SecureCmdLineArgumentsEnabled, $IconFromClient,
		$ShortcutAddedToDesktop, $ShortcutAddedToStartMenu, $PublishedAppInstance.Enabled, $PublishedAppInstance.Uid,
		$UUID, $UserFilterEnabled, $WaitForPrinterCreation, $PublishedAppInstance.Visible, $WorkingDirectory, $DeliveryGroups) -Separator "#~#"
	}
}

$AppDiskUidMap = @{}
Write-Host "AppDisks"
Write-Host "-----------"
$AppDisks = Get-AppLibAppDisk -MaxRecordCount 2147483647
if($AppDisks -ne $null)
{
	foreach($AppDisk in $AppDisks)
	{
		$AppDiskUidMap[$AppDisk.AppDiskUid] = $AppDisk.AppDiskName
		$PackageString = ""
		$Packages = $AppDisk.Packages
		if($Packages -eq $null -or $Packages.Count -eq 0)
		{
			$PackageString = "-#-#-"
		}
		else
		{
			foreach($Package in $Packages)
			{
				$str1 = $Package.Name
				$str2 = $Package.Manufacturer
				$str3 = $Package.Version
				$PackageString += $str1+"#"+$str2 + "#"+$str3+"$=$"
			}
		}
		if($PackageString.endsWith("$=$"))
		{
			$PackageString = $PackageString.substring(0,$PackageString.length-3)			
		}
		Write-Host($AppDisk.AppDiskName,$AppDisk.State,$AppDisk.HypervisorConnectionName,$AppDisk.VirtualDiskId,$PackageString) -Separator ";"
	}
}

Write-Host "Zones"
Write-Host "------"
foreach($Zone in $ConfiguredZones)
{
	Write-Host($Zone.Name,$Zone.IsPrimary) -Separator ";"
}

Write-Host "#EG_SCRIPT_DONE#"

Remove-PSSnapin "Citrix.Common.Commands"
Remove-PSSnapin "Citrix.Host.Admin.V2"
Remove-PSSnapin "Citrix.Broker.Admin.V2"
Remove-PSSnapin "Citrix.ADIdentity.Admin.V2"
Remove-PSSnapin "Citrix.Configuration.Admin.V2"
Remove-PSSnapin "Citrix.MachineCreation.Admin.V2"
Remove-PSSnapin "Citrix.Licensing.Admin.V1"
Remove-PSSnapin "Citrix.DelegatedAdmin.Admin.V1"
Remove-PSSnapin "Citrix.Monitor.Admin.V1"
Remove-PSSnapin "Citrix.ConfigurationLogging.Admin.V1"
Remove-PSSnapin "Citrix.AppLibrary.Admin.V1"



# SIG # Begin signature block
# MIIldgYJKoZIhvcNAQcCoIIlZzCCJWMCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUTMp2FO1wgbJ4Wy7V6ZPaLxiZ
# 94Sggh6cMIIFMDCCBBigAwIBAgIQBAkYG1/Vu2Z1U0O1b5VQCDANBgkqhkiG9w0B
# AQsFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMTMxMDIyMTIwMDAwWhcNMjgxMDIyMTIwMDAwWjByMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQg
# Q29kZSBTaWduaW5nIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
# +NOzHH8OEa9ndwfTCzFJGc/Q+0WZsTrbRPV/5aid2zLXcep2nQUut4/6kkPApfmJ
# 1DcZ17aq8JyGpdglrA55KDp+6dFn08b7KSfH03sjlOSRI5aQd4L5oYQjZhJUM1B0
# sSgmuyRpwsJS8hRniolF1C2ho+mILCCVrhxKhwjfDPXiTWAYvqrEsq5wMWYzcT6s
# cKKrzn/pfMuSoeU7MRzP6vIK5Fe7SrXpdOYr/mzLfnQ5Ng2Q7+S1TqSp6moKq4Tz
# rGdOtcT3jNEgJSPrCGQ+UpbB8g8S9MWOD8Gi6CxR93O8vYWxYoNzQYIH5DiLanMg
# 0A9kczyen6Yzqf0Z3yWT0QIDAQABo4IBzTCCAckwEgYDVR0TAQH/BAgwBgEB/wIB
# ADAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMweQYIKwYBBQUH
# AQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQwYI
# KwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFz
# c3VyZWRJRFJvb3RDQS5jcnQwgYEGA1UdHwR6MHgwOqA4oDaGNGh0dHA6Ly9jcmw0
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwOqA4oDaG
# NGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RD
# QS5jcmwwTwYDVR0gBEgwRjA4BgpghkgBhv1sAAIEMCowKAYIKwYBBQUHAgEWHGh0
# dHBzOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwCgYIYIZIAYb9bAMwHQYDVR0OBBYE
# FFrEuXsqCqOl6nEDwGD5LfZldQ5YMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6en
# IZ3zbcgPMA0GCSqGSIb3DQEBCwUAA4IBAQA+7A1aJLPzItEVyCx8JSl2qB1dHC06
# GsTvMGHXfgtg/cM9D8Svi/3vKt8gVTew4fbRknUPUbRupY5a4l4kgU4QpO4/cY5j
# DhNLrddfRHnzNhQGivecRk5c/5CxGwcOkRX7uq+1UcKNJK4kxscnKqEpKBo6cSgC
# PC6Ro8AlEeKcFEehemhor5unXCBc2XGxDI+7qPjFEmifz0DLQESlE/DmZAwlCEIy
# sjaKJAL+L3J+HNdJRZboWR3p+nRka7LrZkPas7CM1ekN3fYBIM6ZMWM9CBoYs4Gb
# T8aTEAb8B4H6i9r5gkn3Ym6hU/oSlBiFLpKR6mhsRDKyZqHnGKSaZFHvMIIFsTCC
# BJmgAwIBAgIQASQK+x44C4oW8UtxnfTTwDANBgkqhkiG9w0BAQwFADBlMQswCQYD
# VQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGln
# aWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0Ew
# HhcNMjIwNjA5MDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEV
# MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29t
# MSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZ
# wuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4V
# pX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAd
# YyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3
# T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjU
# N6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNda
# SaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtm
# mnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyV
# w4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3
# AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYi
# Cd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmp
# sh3lGwIDAQABo4IBXjCCAVowDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7Nfj
# gtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNt
# yA8wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHkGCCsGAQUF
# BwEBBG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMG
# CCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRB
# c3N1cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwz
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwIAYDVR0g
# BBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBDAUAA4IBAQCa
# FgKlAe+B+w20WLJ4ragjGdlzN9pgnlHXy/gvQLmjH3xATjM+kDzniQF1hehiex1W
# 4HG63l7GN7x5XGIATfhJelFNBjLzxdIAKicg6okuFTngLD74dXwsgkFhNQ8j0O01
# ldKIlSlDy+CmWBB8U46fRckgNxTA7Rm6fnc50lSWx6YR3zQz9nVSQkscnY2W1ZVs
# RxIUJF8mQfoaRr3esOWRRwOsGAjLy9tmiX8rnGW/vjdOvi3znUrDzMxHXsiVla3R
# y7sqBiD5P3LqNutFcpJ6KXsUAzz7TdZIcXoQEYoIdM1sGwRc0oqVA3ZRUFPWLvdK
# RsOuECxxTLCHtic3RGBEMIIGMzCCBRugAwIBAgIQCJp0nrgtw+wn6mXq2/g1MTAN
# BgkqhkiG9w0BAQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQg
# SW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2Vy
# dCBTSEEyIEFzc3VyZWQgSUQgQ29kZSBTaWduaW5nIENBMB4XDTIxMDUyNTAwMDAw
# MFoXDTI0MDUyOTIzNTk1OVowcTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5ldyBK
# ZXJzZXkxDzANBgNVBAcTBklzZWxpbjEdMBsGA1UEChMUZUcgSW5ub3ZhdGlvbnMs
# IEluYy4xHTAbBgNVBAMTFGVHIElubm92YXRpb25zLCBJbmMuMIICIjANBgkqhkiG
# 9w0BAQEFAAOCAg8AMIICCgKCAgEAxGUJ0prjbfxcmgZGjd7JDxOs7sySHvaQ3wGR
# 2KbD3aZageSE+kG0tKkEsQ4na8bnNViY/zF4Pbo0ZkYtMzJIk0AZXDTMvXr/mEvm
# KxCbiJFTCpKkmZ4sa5BZfO7igIvcNSa0zII2a5jiQJFy85j+f9I4EgTo9OcdVeIN
# XhgQ2xZ6TjEK+pbhqmVXvr8DB26JclOaed0L5Vs7+CbGzZuKifgxL2i8d5FDzkhZ
# SZfdCtGtEvE5pLesXSWfpzUddvCNRisIPGB7mg+Rln2XvUZyyaRURtlx11pYfJ/K
# NYAkUCL23rnh38/maxtSvdeioYBj4xLyt7poQBG+b0uzCjgRADe3/k0NtiVwJIo0
# ZEsUyDANp7JIIjMePzsOMYzfMPyi8clAwfBYc2XhSuUcX3yKf4kpRFmClYfK5Lst
# GQRNbONjpsCBTEgZuwEfkdOU0rmaMywZFVLHEpmpYZePz7M3VhN+aV56kR/efo8e
# D81E5VDQqmJiGWKq/s9jhbkYH7g2uhL2MXb/uRshVDSOkFRvIID/l+yDJogDBpae
# 3x2ov5YvjY8Zo8RF0tXvyS0rfsDvANZw7mptzHn0Blmw5lug3F2CVJpNbzb1HmAH
# 6k0yFHmWYCoUANbBB0YR/k4JvgY2byttF7cQPJyl7UI9Cu4KTAg7ROmGkwdLhmXo
# PCBco0ECAwEAAaOCAcQwggHAMB8GA1UdIwQYMBaAFFrEuXsqCqOl6nEDwGD5LfZl
# dQ5YMB0GA1UdDgQWBBT5yESaz9gTSS8aBNa1r34MMA42SjAOBgNVHQ8BAf8EBAMC
# B4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdwYDVR0fBHAwbjA1oDOgMYYvaHR0cDov
# L2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5jcmwwNaAzoDGG
# L2h0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFzc3VyZWQtY3MtZzEuY3Js
# MEsGA1UdIAREMEIwNgYJYIZIAYb9bAMBMCkwJwYIKwYBBQUHAgEWG2h0dHA6Ly93
# d3cuZGlnaWNlcnQuY29tL0NQUzAIBgZngQwBBAEwgYQGCCsGAQUFBwEBBHgwdjAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tME4GCCsGAQUFBzAC
# hkJodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRTSEEyQXNzdXJl
# ZElEQ29kZVNpZ25pbmdDQS5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsF
# AAOCAQEAlAjncFPvLKywT/4DECG7bHbtiVl+uumfRj2YrFuChsgkv1PmQpgfypxC
# 3g+ErV9yRV8+XXSCwOaKJ3v6RLnfphLhEwYc2+0Qs/Nlib5NAxDGuIczAIeXOc5k
# RRpvFsQ2XSNtM7XL1tLDm6p/VG7BoUAyqRXsMPdWbTkN/9ndCmGSsqcxjG2ud8O6
# Vhte9J5LaHBVk3lIZAMtH6ACdo5QTrM49nbIU8QGuRYNXZKRLAUu6IgD6WJKMVfZ
# XWlyfD8dZ2r3ej6Q1uAO/Nbtd397T+BVQrDWMOG8+GeRiJwoevxbIWh0SenZOUrA
# q9vTJaSvFMSvctkJm/oxLUcUdEGS3zCCBq4wggSWoAMCAQICEAc2N7ckVHzYR6z9
# KGYqXlswDQYJKoZIhvcNAQELBQAwYjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERp
# Z2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMY
# RGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTIyMDMyMzAwMDAwMFoXDTM3MDMy
# MjIzNTk1OVowYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2IFRp
# bWVTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAMaG
# NQZJs8E9cklRVcclA8TykTepl1Gh1tKD0Z5Mom2gsMyD+Vr2EaFEFUJfpIjzaPp9
# 85yJC3+dH54PMx9QEwsmc5Zt+FeoAn39Q7SE2hHxc7Gz7iuAhIoiGN/r2j3EF3+r
# GSs+QtxnjupRPfDWVtTnKC3r07G1decfBmWNlCnT2exp39mQh0YAe9tEQYncfGpX
# evA3eZ9drMvohGS0UvJ2R/dhgxndX7RUCyFobjchu0CsX7LeSn3O9TkSZ+8OpWNs
# 5KbFHc02DVzV5huowWR0QKfAcsW6Th+xtVhNef7Xj3OTrCw54qVI1vCwMROpVymW
# Jy71h6aPTnYVVSZwmCZ/oBpHIEPjQ2OAe3VuJyWQmDo4EbP29p7mO1vsgd4iFNmC
# KseSv6De4z6ic/rnH1pslPJSlRErWHRAKKtzQ87fSqEcazjFKfPKqpZzQmiftkaz
# nTqj1QPgv/CiPMpC3BhIfxQ0z9JMq++bPf4OuGQq+nUoJEHtQr8FnGZJUlD0UfM2
# SU2LINIsVzV5K6jzRWC8I41Y99xh3pP+OcD5sjClTNfpmEpYPtMDiP6zj9NeS3YS
# UZPJjAw7W4oiqMEmCPkUEBIDfV8ju2TjY+Cm4T72wnSyPx4JduyrXUZ14mCjWAkB
# KAAOhFTuzuldyF4wEr1GnrXTdrnSDmuZDNIztM2xAgMBAAGjggFdMIIBWTASBgNV
# HRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBS6FtltTYUvcyl2mi91jGogj57IbzAf
# BgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNVHQ8BAf8EBAMCAYYw
# EwYDVR0lBAwwCgYIKwYBBQUHAwgwdwYIKwYBBQUHAQEEazBpMCQGCCsGAQUFBzAB
# hhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUHMAKGNWh0dHA6Ly9j
# YWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3J0MEMG
# A1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2Vy
# dFRydXN0ZWRSb290RzQuY3JsMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG
# /WwHATANBgkqhkiG9w0BAQsFAAOCAgEAfVmOwJO2b5ipRCIBfmbW2CFC4bAYLhBN
# E88wU86/GPvHUF3iSyn7cIoNqilp/GnBzx0H6T5gyNgL5Vxb122H+oQgJTQxZ822
# EpZvxFBMYh0MCIKoFr2pVs8Vc40BIiXOlWk/R3f7cnQU1/+rT4osequFzUNf7WC2
# qk+RZp4snuCKrOX9jLxkJodskr2dfNBwCnzvqLx1T7pa96kQsl3p/yhUifDVinF2
# ZdrM8HKjI/rAJ4JErpknG6skHibBt94q6/aesXmZgaNWhqsKRcnfxI2g55j7+6ad
# cq/Ex8HBanHZxhOACcS2n82HhyS7T6NJuXdmkfFynOlLAlKnN36TU6w7HQhJD5TN
# OXrd/yVjmScsPT9rp/Fmw0HNT7ZAmyEhQNC3EyTN3B14OuSereU0cZLXJmvkOHOr
# pgFPvT87eK1MrfvElXvtCl8zOYdBeHo46Zzh3SP9HSjTx/no8Zhf+yvYfvJGnXUs
# HicsJttvFXseGYs2uJPU5vIXmVnKcPA3v5gA3yAWTyf7YGcWoWa63VXAOimGsJig
# K+2VQbc61RWYMbRiCQ8KvYHZE/6/pNHzV9m8BPqC3jLfBInwAM1dwvnQI38AC+R2
# AibZ8GV2QqYphwlHK+Z/GqSFD/yYlvZVVCsfgPrA8g4r5db7qS9EFUrnEw4d2zc4
# GqEr9u3WfPwwggbGMIIErqADAgECAhAKekqInsmZQpAGYzhNhpedMA0GCSqGSIb3
# DQEBCwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7
# MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1l
# U3RhbXBpbmcgQ0EwHhcNMjIwMzI5MDAwMDAwWhcNMzMwMzE0MjM1OTU5WjBMMQsw
# CQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xJDAiBgNVBAMTG0Rp
# Z2lDZXJ0IFRpbWVzdGFtcCAyMDIyIC0gMjCCAiIwDQYJKoZIhvcNAQEBBQADggIP
# ADCCAgoCggIBALkqliOmXLxf1knwFYIY9DPuzFxs4+AlLtIx5DxArvurxON4XX5c
# Nur1JY1Do4HrOGP5PIhp3jzSMFENMQe6Rm7po0tI6IlBfw2y1vmE8Zg+C78KhBJx
# bKFiJgHTzsNs/aw7ftwqHKm9MMYW2Nq867Lxg9GfzQnFuUFqRUIjQVr4YNNlLD5+
# Xr2Wp/D8sfT0KM9CeR87x5MHaGjlRDRSXw9Q3tRZLER0wDJHGVvimC6P0Mo//8Zn
# zzyTlU6E6XYYmJkRFMUrDKAz200kheiClOEvA+5/hQLJhuHVGBS3BEXz4Di9or16
# cZjsFef9LuzSmwCKrB2NO4Bo/tBZmCbO4O2ufyguwp7gC0vICNEyu4P6IzzZ/9KM
# u/dDI9/nw1oFYn5wLOUrsj1j6siugSBrQ4nIfl+wGt0ZvZ90QQqvuY4J03ShL7BU
# dsGQT5TshmH/2xEvkgMwzjC3iw9dRLNDHSNQzZHXL537/M2xwafEDsTvQD4ZOgLU
# MalpoEn5deGb6GjkagyP6+SxIXuGZ1h+fx/oK+QUshbWgaHK2jCQa+5vdcCwNiay
# CDv/vb5/bBMY38ZtpHlJrYt/YYcFaPfUcONCleieu5tLsuK2QT3nr6caKMmtYbCg
# QRgZTu1Hm2GV7T4LYVrqPnqYklHNP8lE54CLKUJy93my3YTqJ+7+fXprAgMBAAGj
# ggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8E
# DDAKBggrBgEFBQcDCDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEw
# HwYDVR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYDVR0OBBYEFI1kt4kh
# /lZYRIRhp+pvHDaP3a8NMFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwzLmRp
# Z2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3Rh
# bXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEBBIGDMIGAMCQGCCsGAQUFBzABhhhodHRw
# Oi8vb2NzcC5kaWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0dHA6Ly9jYWNlcnRz
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1l
# U3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcNAQELBQADggIBAA0tI3Sm0fX46kuZPwHk
# 9gzkrxad2bOMl4IpnENvAS2rOLVwEb+EGYs/XeWGT76TOt4qOVo5TtiEWaW8G5iq
# 6Gzv0UhpGThbz4k5HXBw2U7fIyJs1d/2WcuhwupMdsqh3KErlribVakaa33R9QIJ
# T4LWpXOIxJiA3+5JlbezzMWn7g7h7x44ip/vEckxSli23zh8y/pc9+RTv24KfH7X
# 3pjVKWWJD6KcwGX0ASJlx+pedKZbNZJQfPQXpodkTz5GiRZjIGvL8nvQNeNKcEip
# tucdYL0EIhUlcAZyqUQ7aUcR0+7px6A+TxC5MDbk86ppCaiLfmSiZZQR+24y8fW7
# OK3NwJMR1TJ4Sks3KkzzXNy2hcC7cDBVeNaY/lRtf3GpSBp43UZ3Lht6wDOK+Eoo
# jBKoc88t+dMj8p4Z4A2UKKDr2xpRoJWCjihrpM6ddt6pc6pIallDrl/q+A8GQp3f
# BmiW/iqgdFtjZt5rLLh4qk1wbfAs8QcVfjW05rUMopml1xVrNQ6F1uAszOAMJLh8
# UgsemXzvyMjFjFhpr6s94c/MfRWuFL+Kcd/Kl7HYR+ocheBFThIcFClYzG/Tf8u+
# wQ5KbyCcrtlzMlkI5y2SoRoR/jKYpl0rl+CL05zMbbUNrkdjOEcXW28T2moQbh9J
# t0RbtAgKh1pZBHYRoad3AhMcMYIGRDCCBkACAQEwgYYwcjELMAkGA1UEBhMCVVMx
# FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNv
# bTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmlu
# ZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYBBAGCNwIB
# DDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFJCsrVOoVGKaG4dM3VtBvkxo
# C7CDMA0GCSqGSIb3DQEBAQUABIICAD0u0M6StWt8egI1sbyvmSmLt7CzqRDeBLSK
# KcaQ+SZos0VVuCBub5UJX/AxFY1/nKhxviWsBVaWw+U6PvRmo3B1344aW6SbI0mT
# OC+c38tRDvJS4jroBSwa+Ysuf0S/PABs97U5n7XNGXcxnEf3i6g2eqYW/H2HRMhq
# 4Cv1saUgeIsoIbNBC1XGH8BdZLudymuqf8nu8s1LUhOjQWdGi0ZBbKUl0D4jbO2w
# ZkEMi1+ntgY5X+AdoZwtE9nZ3m6SVXMETLvQQgtNsrRwqnTeLJXWHJv8Ohhc9hIx
# m5BXDSrh+n09KeYIkV0enrY1tg0CRoOyELYXGay0FdXo8Jla8RIx/lXlLSrRnhk1
# M+L7waQtsAqBSHQmMYgLE5kUaiTOPrx/od5ktlkDmvesQ3Lq2Pmk88AQsfgPQL4H
# DnzseyjNqNS0U5Ae7qupqoFdJxpEJqERY1fHGNnwRpSs4Pm8vJD6uLXXovGkqTSW
# rwy1nR5c4zcu2FHQqPTVX9R0/qq/1KIR7/GKHG+U7dAFpP+qAgvczVsVGL3ffxoJ
# dUa3w1wDmaei1fecfyAv3ymj7SOksdhJgCCiRXJwTMqPRBM2QLepiho+jmYufoTU
# QmcLMVwLBrglXe2lO4u4wN2r6GlEasRTGoNgAWULRMB/gGRd1dsYhoTLhenPt6rv
# bdI/z1pOoYIDIDCCAxwGCSqGSIb3DQEJBjGCAw0wggMJAgEBMHcwYzELMAkGA1UE
# BhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2Vy
# dCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQCnpK
# iJ7JmUKQBmM4TYaXnTANBglghkgBZQMEAgEFAKBpMBgGCSqGSIb3DQEJAzELBgkq
# hkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTIyMDcxMjEwMzkxN1owLwYJKoZIhvcN
# AQkEMSIEIGmJtqH9/yO9ajivFHotgxayYMaxYllcVjX+34fpwA1EMA0GCSqGSIb3
# DQEBAQUABIICAG103v+m7WvERee47jBSCUDznmEQGXEbMsFMzluKRrCBS1ohV2Js
# AX3FXyxRskPMNgUkvQN47GkoZbwbCFfN0giwaYmSF8n82YnV9SKOy1Fj3d+DqFza
# Tzf6TUCj9bSSXH+EwEVvpODIsRdH1SAa1sDhe9TW5O1Yc6j8v0x9jP4wbXixn/gg
# E1Fz9bqxsZ5hUr836r1ILDnenk8CTYPgx4BQ0sADxmO+I/uyMcwh9qYqgIGFiaBd
# vqivsZPShlaoGhOEUyq+/Fauuwu0zEOMOUbPaLt15IpsLyuRZ4HONBNj98DM10/p
# LcaxFiy1qw3/vvvW/2Ad9rLfQR2Oevr2k/OzVDw6hWHFjSeM5f0HPNR7pFreD7EQ
# gzq+TOPz4zSMU6nkpUOlomRKNr3uIz4wjKgsPqdXFQNvTJXP3h6tfOF83VaHhupe
# 5BYJUzz0vjI4Fn00jdklu5gPFx4+TcKrVo6iQsFIyeaBbdYvn11mpNGaub5Ut1tk
# OWIUE32BT7wefT3P7/oLiam7nVJhxJNF4TvQVWKiUH5EhzoCituKCQEnND47ClQK
# uSelnOCdD5JsCmB/gIo2DEkTGZ0EbCWP0U/BDIY1Ab7yxMdqzdJHHVi7t39KMyCY
# DcVDmtoJj7ZC6vdqvFATjJWs6KQWDPYhXtSL4Nse8bs5YqU5yfhj4ZOO
# SIG # End signature block
