$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

function Get-ViewAPIService
{
	param(
	[Parameter(Mandatory = $false)]
	$HvServer
	)
	if ($null -ne $hvServer)
	{
		if ($hvServer.GetType().name -ne 'ViewServerImpl')
		{
			$type = $hvServer.GetType().name
			Write-Error "Expected hvServer type is ViewServerImpl, but received: [$type]"
			return $null
		}
		elseif ($hvServer.IsConnected)
		{
			return $hvServer.ExtensionData
		}
	} elseif ($global:DefaultHVServers.Length -gt 0)
	{
		$hvServer = $global:DefaultHVServers[0]
		return $hvServer.ExtensionData
	}
	return $null
}

#Load VMware View Snapin to powershell v2 - Starts
[string]$eGInstallDir = $args[0]
$vmhelper = "VMware.HV.Helper.psm1"
$moduleStr = $eGInstallDir+"\"+$vmhelper

Set-Variable product_name "VMware View PowerCLI" -scope Private
Set-Variable view_snapin_name "VMware.View.Broker" -scope Private
$ViewSnapinLoaded = Get-PSSnapin | Where { $_.Name -eq $view_snapin_name }
if(!$ViewSnapinLoaded)
{
	# Install or Re-register View Cmdlets
	$installpath = (get-itemproperty "HKLM:\Software\VMware, Inc.\VMware VDM").ServerInstallPath
	$BuildVersion = (get-itemproperty "HKLM:\Software\VMware, Inc.\VMware VDM").BuildNumber
	$os=Get-WMIObject win32_operatingsystem | select OSArchitecture
	if ($os.OSArchitecture -like "64*")
	{
		$path="$env:windir\Microsoft.NET\Framework64\v2.0.50727\installUtil.exe"
		if(![System.IO.File]::Exists($path))
		{
			$path="$env:windir\Microsoft.NET\Framework64\v4.0.30319\installUtil.exe"
			if(![System.IO.File]::Exists($path))
			{
				Write-Host "Either .NET v2 or v4 frameworks are not installed"
				return
			}
		}
	}
	else
	{
		$path="$env:windir\Microsoft.NET\Framework\v2.0.50727\installUtil.exe"
		if(![System.IO.File]::Exists($path))
		{
			$path="$env:windir\Microsoft.NET\Framework\v4.0.30319\installUtil.exe"
			if(![System.IO.File]::Exists($path))
			{
				Write-Host "Either .NET v2 or v4 frameworks are not installed"
				return
			}
		}
	}
	set-alias installUtil $path
	$null = (installUtil ($installpath  + "\bin\PowershellServiceCmdlets.dll"))
}

Remove-Variable product_name -scope Private
Remove-Variable view_snapin_name -scope Private

#Load VMware View Snapin to powershell v2 - Ends

Add-pssnapin "VMware.View.Broker"
Import-Module -Name VMware.VimAutomation.HorizonView -EA SilentlyContinue
Import-Module -Name $moduleStr -Force -EA SilentlyContinue

$hzUser = $args[1]
$hzPass = $args[2]
$DirPath = $eGInstallDir + "\OE.exe"
& $DirPath $hzPass | Tee-Object -Variable pwd | Out-Null
$hzDomain = $args[3]
[string]$hzConnStr = $args[4]

$Global:ErrorActionPreference="SilentlyContinue"
Set-PowerCLIConfiguration -Scope User -ParticipateInCEIP $false -confirm:$false |out-null
Set-PowerCLIConfiguration -InvalidCertificateAction Ignore -confirm:$false |out-null

try
{
	# Establish connection to Connection Server
	$hvServer = Connect-HVServer -Server $hzConnStr -User $hzUser -Password $pwd -Domain $hzDomain
}
catch
{
	write-host "Not able to connect to the server $hzConnStr".
	Write-host $_.Exception.Message
	return
}

$Global:hvServices = $hvServer.ExtensionData
$services = Get-ViewAPIService -hvServer $hvServer
$vc_service_helper = New-Object VMware.Hv.VirtualCenterService
$vcList = $vc_service_helper.VirtualCenter_List($services)

cls

$hostname = $env:computername
$hostname = "DomainMonitor." + $hostname
$LDAPPath = 'LDAP://localhost:389/OU=Servers,DC=vdi,DC=vmware,DC=int'
$LDAPEntry = New-Object DirectoryServices.DirectoryEntry $LDAPPath
$Selector = New-Object DirectoryServices.DirectorySearcher
$Selector.SearchRoot = $LDAPEntry

$LDAPPath1 = 'LDAP://localhost:389/OU=Server Groups,DC=vdi,DC=vmware,DC=int'
$LDAPEntry1 = New-Object DirectoryServices.DirectoryEntry $LDAPPath1
$Selector1 = New-Object DirectoryServices.DirectorySearcher
$Selector1.SearchRoot = $LDAPEntry1

Write-Host "Connection Brokers"
Write-Host "-------------------"
$connBrokers = Get-ConnectionBroker | select broker_id,activation,lastBackup,ldapBackupFrequency,type
if($connBrokers -ne $null)
{
	foreach($connBroker in $connBrokers)
	{
        if($connBroker.type -eq "Connection Server")
        {
		    Write-Host ($connBroker.broker_id,$connBroker.activation,
		    $connBroker.lastBackup,$connBroker.ldapBackupFrequency ) -Separator "#"
        }
	}
}
else
{
	Write-Host "- #"  -nonewline;
}

<##
logic for findig the current connection server is master or replica and ignore the session monitoring data
#$vdm = Get-ItemProperty "HKLM:\SOFTWARE\VMware, Inc.\VMware VDM" 
#$isStandardVDM = $vdm.ServerInstanceType
##>
$isStandardVDM = $False
$ldapaddr = 'LDAP://localhost:389/DC=vdi,DC=vmware,DC=int'
$ldapopen = [adsi]$ldapaddr
$objectCategoryarr = $ldapopen.'objectCategory' -split ','
#$objectCategoryarr.count
if($objectCategoryarr.count -eq 4)
{
    $schemaguid = $objectCategoryarr[3]
    $schemaConfigurationaddr= 'LDAP://localhost:389/CN=Schema,CN=Configuration,'+$schemaguid
    #$schemaConfigurationaddr
    $schemaConfigurationOpen = [adsi]$schemaConfigurationaddr
    #$schemaConfigurationOpen.fSMORoleOwner
    $fsmoRoleOwnerAddr = $schemaConfigurationOpen.fSMORoleOwner -split ","
    if($fsmoRoleOwnerAddr.count -gt 2)
    {
        $masterarr= $fsmoRoleOwnerAddr[1].split('`$')
        if($masterarr.count -eq 2)
        {
            $localhost = "CN=$env:computername"
            #$localhost
            #$masterarr[0]
            $isStandardVDM = $localhost -eq $masterarr[0]
        }
    }
}

$isStandardVDM

Write-Host "Pools"
Write-Host "--------"
if($isStandardVDM -eq $True)
{
	$pools = get-hvpool
	foreach($pool in $pools)
	{
		$headroomcount = -5
		$maximumcount = -5
		$minimumcount = -5
		$vcServerName = -5

		$DesktopBase = $pool.Base
		$id = $DesktopBase.Name
		$displayName = $DesktopBase.DisplayName

		$DesktopSettings = $pool.DesktopSettings
		$enabled = $DesktopSettings.Enabled
		$deliveryModel = $pool.Type
		$desktopSource = $pool.Source
		if($deliveryModel -eq "AUTOMATED")
		{
			$DesktopData = $pool.AutomatedDesktopData
			$VmNamingSettings = $DesktopData.VmNamingSettings
			$PatternNamingSettings = $VmNamingSettings.PatternNamingSettings
			$maximumcount = $PatternNamingSettings.MaxNumberOfMachines
			$minimumcount = $PatternNamingSettings.MinNumberOfMachines
			$headroomcount = $PatternNamingSettings.NumberOfSpareMachines
			if($minimumcount -eq $null -or $minimumcount.length -eq 0) {$minimumcount = -5}
		}
		elseif($deliveryModel -eq "MANUAL")
		{
			$DesktopData = $pool.ManualDesktopData
		}

		if($DesktopData.VirtualCenter -ne $null)
		{
			$vcid = $DesktopData.VirtualCenter.Id
			foreach($vCenter in $vcList)
			{
				if($vCenter.Id.Id -eq $vcid)
				{
					$vcServerSpec = $vCenter.ServerSpec
					$vcServerName = $vcServerSpec.ServerName +":" +  $vcServerSpec.Port
					if($vcServerName -eq $null -or $vcServerName.length -eq 0)
					{
						$vcServerName = "-"
					}
					break
				}
			}
		}
		$poolType = $deliveryModel
		write-host ($id, $displayName,$enabled, $deliveryModel, $desktopSource,$vcServerName, $poolType, $headroomcount, $maximumcount, $minimumcount) -Separator "#"

		

		$poolentitlements = Get-PoolEntitlement -pool_id $id  | select displayName
		Write-Host "poolentitlement starts ^!^" -nonewline;
		if($poolentitlements -ne $null)
		{
			foreach($poolentitlement in $poolentitlements)
			{
				Write-Host ($poolentitlement.displayName) ("^!^")  -nonewline;
			}
		}
		else
		{
			Write-Host "- ^!^"  -nonewline;
		}
		Write-Host "poolentitlement stops"

		Write-Host "Desktops starts ^!^" -nonewline;
		if($desktopSource -eq "VIRTUAL_CENTER" -or $desktopSource -eq "VIEW_COMPOSER" -or $desktopSource -eq "INSTANT_CLONE_ENGINE")
		{
			$machines = Get-HVMachine -poolname $id
			if($machines -ne $null)
			{
				foreach($machine in $machines)
				{
					$machineBase = $machine.Base
					$Name = $machineBase.Name
					if($Name -eq $null -or $Name.length -eq 0)
					{
						$Name = "-"
					}
					$GuestFullName = $machineBase.OperatingSystem
					if($GuestFullName -eq $null -or $GuestFullName.length -eq 0)
					{
						$GuestFullName = "-"
					}
					$HostName = $machineBase.DnsName
					if($HostName -eq $null -or $HostName.length -eq 0)
					{
						$HostName = "-"
					}
					$IPAddress = $HostName
					$isInPool = $true
					$machine_id = $Name
					$ManagedMachineData = $machine.ManagedMachineData
					$VirtualCenterData = $ManagedMachineData.VirtualCenterData
					$Path = $VirtualCenterData.Path
					$ServerGroup = $id
					$BasicState = $machineBase.BasicState
					Write-Host ($id,$Name,$Path, $GuestFullName, $HostName, $IPAddress, $isInPool,$machine_id,$ServerGroup,$BasicState, "^!^") -Separator "#" -nonewline;
				}
			}
			else
			{
				Write-Host "- ^!^" -nonewline;
			}
		}
		elseif($desktopSource -eq "RDS")
		{
			$RdsDesktopData = $pool.RdsDesktopData
			if($RdsDesktopData -ne $null)
			{
				$FarmID = $RdsDesktopData.Farm
				$FarmInfo = $services.Farm.Farm_GetSummaryView($FarmID)
				if($FarmInfo -ne $null)
				{
					$farmData = $FarmInfo.Data
					$Path = "-"
					$GuestFullName = "-"
					$isInPool = $true
					$Enabled = $farmData.Enabled
					if($Enabled -eq $null -or $Enabled.length -eq 0)
					{
						$Enabled = 0	#0 means Enabled   1 means Disabled
					}
					$MaxConns = $farmData.MaximumNumberOfSessions
					if($MaxConns -eq $null -or $MaxConns.length -eq 0)
					{
						$MaxConns = 150
					}
					$farmHealth = $services.FarmHealth.FarmHealth_Get($FarmID)
					if($farmHealth -ne $null)
					{
						$name = $farmHealth.Name
						$RDSServersHealth = $farmHealth.RDSServerHealth
						foreach($RDSServer in $RDSServersHealth)
						{
							$HostName = "-"
							$IPAddress = "-"
							$machine_id = "-"
							$ServerGroup = "-"
							$BasicState = "-"

							$IPAddress = $RDSServer.Name
							$machine_id = $RDSServer.Name
							$HostName = $RDSServer.Name  
							$BasicState = $RDSServer.Status

							Write-Host ($id,$name,$Path, $GuestFullName, $HostName, $IPAddress, $isInPool,$machine_id,$ServerGroup,$BasicState,$Enabled,$MaxConns,"^!^") -Separator "#" -nonewline;
						}
					}
				}
			}
			else
			{
				Write-Host "- ^!^" -nonewline;
			}
		}
		Write-Host "Desktops stops"
		Write-Host "~#~"
	}
	

	Write-Host "PoolsUsage"
	Write-Host "----------"
	write-Host "localsessions starts ^!^" -nonewline;
	Write-Host "localsessions stops"
	Write-Host "remotesessions starts ^!^" -nonewline;
	$PodSessions = get-hvlocalsession
	if($podSessions -ne $null)
	{
		foreach($session in $podSessions)
		{
			$NamesData = $session.NamesData
			$mac_id = $NamesData.MachineOrRDSServerName
			$SessionData = $session.SessionData
			$userName = $NamesData.userName
			$SessUniqID = $session.id.id
			if($userName -eq $null -or $userName.length -eq 0)
			{
				$userName = "-"
			}
			$DNSName = $NamesData.MachineOrRDSServerDNS
			if($DNSName -eq $null -or $DNSName.length -eq 0)
			{
				$DNSName = "-"
			}
			$startTime = $SessionData.startTime
			if($startTime -eq $null -or $startTime.length -eq 0)
			{
				$startTime = "-"
			}
			$duration = $SessionData.LastSessionDurationMS
			if($duration -eq $null -or $duration.length -eq 0)
			{
				$duration = "-"
			}
			$state = $SessionData.SessionState
			if($state -eq $null -or $state.length -eq 0)
			{
				$state = "-"
			}
			$protocol = $SessionData.SessionProtocol
			if($protocol -eq $null -or $protocol.length -eq 0)
			{
				$protocol = "-"
			}
			$session_id = $SessionData.SessionType
			if($session_id -eq $null -or $session_id.length -eq 0)
			{
				$session_id = "-"
			}
			$pool_id = $NamesData.DesktopName
			if($pool_id -eq $null)
			{
				$pool_id = $NamesData.FarmName
			}
			Write-Host ($pool_id,$userName,$startTime, $DNSName, $duration, $mac_id, $state, $protocol, $session_id,$SessUniqID,"^!^") -Separator "#" -nonewline;
		}
	}
	else
	{
		Write-Host "- ^!^" -nonewline;
	}
	write-Host "remotesesions stops"
}
Write-Host 
Write-Host "Domain monitor"
Write-Host "---------------"
$domains = Get-Monitor -monitor DomainMonitor -monitor_id $env:computername
if($domains -ne $null)
{
	write-host($domains.domains)
}

Write-Host 



Write-Host "VC monitor"
Write-Host "-----------"
$vcs = Get-Monitor -monitor VCMonitor
if($vcs -ne $null)
{
	foreach($vc in $vcs)
	{
		Write-Host ($vc.URL,$vc.State) -Separator "^"
	}
}
Write-Host




Write-Host "DB monitor"
Write-Host "-------------"

$db = Get-monitor -monitor DBMonitor
Write-Host ($db.Server,$db.DBName,$db.State) -Separator "#"

Write-Host



Write-Host "License"
Write-Host "---------"

$license = Get-hvlicense
$license | fl

Write-Host "RDS Host Applications"
Write-Host "------------------"
if($isStandardVDM -eq $True)
{
	$AppForRDSServers = New-Object -TypeName System.Collections.Generic.List[string]
	$MissingAppForRDSServers = New-Object -TypeName System.Collections.Generic.List[string]
	$query_service_helper = New-Object VMware.Hv.QueryServiceService
	$defn = New-Object VMware.Hv.QueryDefinition
	$defn.queryEntityType = 'ApplicationInfo'
	$queryResults = $query_service_helper.QueryService_Query($services,$defn)
	$AppsInfo = $queryResults.results

	if($AppsInfo -ne $null)
	{
		foreach($AppInfo in $AppsInfo)
		{
			$MissingAppExists = $false
			$AppForRDSServers.Clear()
			$MissingAppForRDSServers.Clear()
			$AppData = $AppInfo.data
			$AppDisplayName = $AppData.Name
			$AppEnabled = $AppData.enabled
			$farmID = $AppInfo.executionData.farm
			$farmHealth = $services.FarmHealth.FarmHealth_Get($farmID)
			if($farmHealth -ne $null)
			{
				$fname = $farmhealth.name
				$RDSServersHealth = $farmHealth.RDSServerHealth
				foreach($RDSServer in $RDSServersHealth)
				{
					$IPAddress = $RDSServer.Name
					$AppForRDSServers.Add($IPAddress)
					$MissingApplications = $RDSServer.MissingApplications
					if($MissingApplications -ne $null)
					{
						foreach($missingApps in $MissingApplications)
						{
							
							if($missingApps.Name -eq $AppDisplayName)
							{
								$MissingAppExists = $true
								$MissingAppForRDSServers.add($IPAddress)
							}
						}
					} 
				}
				write-host ($AppDisplayName,$fname, $AppForRDSServers,$AppEnabled,$MissingAppExists,$MissingAppForRDSServers) -Separator "#"  
			}
		}
	}
}

write-host
write-host "RDS Farms"
Write-host "-----------"
if($isStandardVDM -eq $True)
{
	$farms = get-hvfarm -SuppressInfo $true
	if($farms -ne $null)
	{
		foreach($farm in $farms)
		{
			$farmId = $farm.ID
			$farmtype = $farm.Type
			$FarmInfo = $services.Farm.Farm_GetSummaryView($FarmID)
			if($FarmInfo -ne $null)
			{
				$farmData = $FarmInfo.Data
				$Path = "-"
				$GuestFullName = "-"
				$isInPool = $true
				$Enabled = $farmData.Enabled
				if($Enabled -eq $null -or $Enabled.length -eq 0)
				{
					$Enabled = 0	#0 means Enabled   1 means Disabled		
				}
				$MaxConns = $farmData.MaximumNumberOfSessions
				if($MaxConns -eq $null -or $MaxConns.length -eq 0)
				{
					$MaxConns = 150
				}
				$farmHealth = $services.FarmHealth.FarmHealth_Get($FarmID)
				if($farmHealth -ne $null)
				{
					$name = $farmHealth.Name
					$RDSServersHealth = $farmHealth.RDSServerHealth
					foreach($RDSServer in $RDSServersHealth)
					{
						$HostName = "-"
						$IPAddress = "-"
						$machine_id = "-"
						$ServerGroup = "-"
						$BasicState = "-"

						$IPAddress = $RDSServer.Name
						$machine_id = $RDSServer.Name
						$HostName = $RDSServer.Name  
						$BasicState = $RDSServer.Status
						Write-Host ($name,$Path, $GuestFullName, $HostName, $IPAddress, $isInPool,$machine_id,$ServerGroup,$BasicState,$Enabled,$MaxConns) -Separator "#"
					}
				}
			}
		}
	}
}
write-host
write-host "Date Format"
Write-host "-----------"
$f = (get-culture).datetimeformat.ShortDatePattern
$h = (get-culture).datetimeformat.LongTimePattern
Write-host "$f $h"

Disconnect-HVServer -confirm:$false


Remove-Pssnapin "VMware.View.Broker"

# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUiXE6hoXdL3+zDwBmGwXp9MgJ
# zWqgghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFLpSZzsP/ohmd3I2
# nn4JIuwnSTCIMA0GCSqGSIb3DQEBAQUABIICAFCBQ+uGKtPx1clmrmCK73QtG5TO
# Z9DDxwe2PlqPi+aeIXNPavkp5Wl0SQgnxl+b/YoT+4YR86j0OJ/izOtGr9EP5XUu
# +ZISWlTwExWwDTACQNTzdkhT3u6GIxzpSu1nBZ/MDmV1sYcEzS4xxRHetYls35bC
# OrHm+Nd5tQhrPSzrvRulxlE8lvkOGq/VJYHf2SyCoNYvdas9j9vcGHM6nIx2F2q2
# fFpExhzLnhi5+i0a0wnrkGDErAAmlEXijC+mI4Xd1PVlC8nHa//B4W/Q03uWRlMo
# 8qHu87+fH6Rv0kiClT5XvXrNmd74CEk5AripihWxgnvWxMURpAsdcgBG3wD3Nk++
# ffrZbKO9tL6hFCs77Q+i9hRBuGKfTN7CV9uohHgYS2bggBqqirNekb06FSSHL67l
# xSGsYg4OfbvI6fn6WDIu9AHwjdSEwC7TcOLv3bIAEAcUkJ6lwTBv3+DPVct7lwV9
# HUPk3anwYdom/bnnAXGFLdpF28OjAumjUExs5KyOgcAT8nnwdCWDgS3NJ6HhGoBv
# nzyPnJUyyWb41yC/xh9+FDxLDH1IiAzH3C0MxFSTXVmvRGwq3WMH4FP2Lwc7LV9+
# GXQJlnxu74c+Hd/KOa0UDEJNTTG4ulLmiAPqA0GYni6wq03Z6YzNOxe9xVEyybOZ
# q84A03KY7GWWEkcsoYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MjAyMDIxNjA4NTNaMC8GCSqGSIb3DQEJBDEiBCAueCRTEBk+h1ZldAJEYVux5U7D
# lDZpB2Kegc5cnvBewDANBgkqhkiG9w0BAQEFAASCAQAuaWU4t981j9f9znlSEq4F
# j3sfHh0A99sx2BIKnL+euMcd2hqjRfBxJrS8aedwRy29XsP1dql2nFb26PYb2OQO
# /GChS/5POIozsKcIY13B1gtf9m3ARfA5ct+oJRkk2T9GmWRp/I/GsJMEh8aIjoNV
# MkxbsGa1tnyAQpP6V8OmiDWd7m93j2xVngm15bypL7o6fK9+ot0TxxVKVQVqB8i+
# UAduArs0Rrh9FTg7XnGaHhI1dniZqr/KPXQsrjnX7Ax4ejxFKj4kizLEXlWp4zd/
# x72ogwkF3vLTlEzKiOdpAi2pmLwOiykcJelQUlRyuIk6e6ma6vBQNjVlljf0bSGd
# SIG # End signature block
