$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

Set-StrictMode -Version 2
$ErrorActionPreference = 'SilentlyContinue'
Function YesNo ([boolean] $condition)
{
    if ($condition -eq $True)
    {
	    return "Yes"
    }
    Else
    {
	    return "No"
    }
}
Function TranslatePasswordOptions($pwKey)
{
	Switch($pwKey)
    {
		"Always" 	{$pwValue = "At any time"}
		"ExpiredOnly" 	{$pwValue = "When expired"}
		"Never" 	{$pwValue = "Never"}
		Default		{$pwValue = $pwKey}
	}
	return $pwValue
}
Function TranslateMethod($methodKey)
{
	Switch($methodKey)
	{
		"ExplicitForms" 	{$methodValue = "User name and password"}
		"IntegratedWindows"	{$methodValue = "Domain pass-through"}
		"CitrixAGBasic"		{$methodValue = "Pass-through from NetScaler Gateway"}
		"HttpBasic"		{$methodValue = "HTTP Basic"}
		"Certificate"		{$methodValue = "Smart card"}
		Default			{$methodValue = "(Unknown authentication method)"}
	}
	return $methodValue
}
Function TranslateNSVersion($versionKey)
{
	Switch($versionKey)
	{
		"Version10_0_69_4" 	{$versionValue = "10.0 (Build 69.4) or later"}
		"Version9x"		{$versionValue = "9.x"}
		"Version5x"		{$versionValue = "5.x"}
		Default			{$versionValue = $versionKey}
	}
	return $versionValue
}

Function TranslateHTML5Deployment($HTML5Key)
{
	Switch($HTML5Key)
	{
		"Fallback" 	{$HTML5Value = "Use Receiver for HTML5 if local install fails"}
		"Always"	{$HTML5Value = "Always use Receiver for HTML5"}
		"Off"		{$HTML5Value = "Citrix Receiver installed locally"}
		Default		{$HTML5Value = $HTML5Key}
	}
	return $HTML5Value
}

Function TranslateLogonType($logonKey)
{
	Switch($logonKey)
	{
		"DomainAndRSA" 	{$logonValue = "Domain and security token"}
		"Domain"	{$logonValue = "Domain"}
		"RSA"		{$logonValue = "Security token"}
		"SMS"		{$logonValue = "SMS authentication"}
		"SmartCard"	{$logonValue = "Smart card"}
		"None"		{$logonValue = "None"}
		Default		{$logonValue = $logonKey}
	}
	return $logonValue
}

#Add-snapins
#Exclude these snaps to avoid clash with modules
$excludedSnapins = @("Citrix.DeliveryServices.ConfigurationProvider","Citrix.DeliveryServices.ClusteredCredentialWallet.Install","Citrix.DeliveryServices.Workflow.WCF.Install" )
$availableSnapins = Get-PSSnapin -Name "Citrix.DeliveryServices.*" -Registered | Select -ExpandProperty "Name"
$loadedSnapins = Get-PSSnapin -Name "Citrix.DeliveryServices.*" -ErrorAction SilentlyContinue | Select -ExpandProperty "Name"
foreach ($snapin in $availableSnapins)
{
	if (($excludedSnapins -notcontains $snapin) -and ($loadedSnapins -notcontains $snapin))
	{
		Add-PSSnapin -Name $snapin
	}
}


#Import-modules
$dsInstallProp = Get-ItemProperty -Path HKLM:\SOFTWARE\Citrix\DeliveryServicesManagement -Name InstallDir
$dsInstallDir = $dsInstallProp.InstallDir
$dsModules = Get-ChildItem -Path "$dsInstallDir\Cmdlets" | Where { $_.FullName.EndsWith('psm1') } | foreach { $_.FullName }
foreach ($dsModule in $dsModules)
{
    Import-Module $dsModule
}



$scriptpath = $MyInvocation.MyCommand.Path
$dirPath = Split-Path $scriptpath
$outputfile = $dirPath+'/sfconfig.txt'

$servername = $env:computername
"Server Info" | out-file $outputfile
"#------------" | out-file $outputfile -append
"ServerName: " + $servername | out-file $outputfile -append
$serverversion = (get-DSVersion).StoreFrontVersion
"ServerVersion: " + $serverversion  | out-file $outputfile -append
$SFVersion = $serverversion
# server group
"#-------------------------------" | out-file $outputfile -append
"Processing server group starts" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append
$baseURL = $(get-DSHostBaseUrl).hostBaseUrl
"Base URL: " + $baseURL | out-file $outputfile -append
$clusterMemberCount = $(get-DSClusterMembersCount).MembersCount
"Number of servers: " + $clusterMemberCount | out-file $outputfile -append
if (($clusterMemberCount -gt 1) -and (($SFVersion) -ge 2.5) )
{
	$lastSource = (Get-DSClusterConfigurationUpdateState).UpdateState.LastSourceServerName
	"Last Source: " + $lastSource | out-file $outputfile -append
}
$clusterMemberNames = $(get-DSClusterMembersName).HostNames

ForEach($member in $clusterMemberNames) 
{
	If ($clusterMemberCount -gt 1)
	{
		$lastSync = $(Get-DSXdServerGroupConfigurationUpdateState $member).LastEndTime.ToString()
		"Server Name: " + $member + "`t`tSynchronization Status: " + $lastSync | out-file $outputfile -append
	}
}
"#-------------------------------" | out-file $outputfile -append
"Processing server group end" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append

# authentication
"Processing authentication starts" | out-file $outputfile -append
"#---------------------------------" | out-file $outputfile -append
$auth = $(Get-DSAuthenticationServicesSummary)
$tokenURL = $auth.TokenIssuerUrl + "/validate"
If($auth.UseHttps -eq $True)
{
	$status = "StoreFront using HTTPS."
	$daysToExpire = (New-TimeSpan -End $auth.IISCertificate.NotAfter).Days
}
Else
{
	$status = "StoreFront using HTTP not HTTPS."
	$daysToExpire = ""
}
"Authentication Methods" | out-file $outputfile -append
"~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append

[int]$enabledMethods = 0
ForEach ($protocol in $auth.protocols)
{
	If($protocol.DisplayInConsole -eq $True)
	{
		$methodKey = $protocol.choice
		$method = TranslateMethod($methodKey)

		Switch($protocol.enabled)
		{
			$True	{$enabled = "Yes"; ++$enabledMethods}
			Default	{$enabled = "No"}
		}
		$method + ":`t"+ $enabled | out-file $outputfile -append
	}
}

"Authentication Service" | out-file $outputfile -append
"~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append

"Token validation service: " + $tokenURL | out-file $outputfile -append
"Number of enabled methods: " + $enabledMethods | out-file $outputfile -append
"Status: " +$status | out-file $outputfile -append


"Configure Trusted Domains" | out-file $outputfile -append
"~~~~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append

$domainInfo = Get-DSExplicitCommonDataModel ($auth.SiteID) $auth.VirtualPath
$defDomain = ($domaininfo.DefaultDomain).DefaultDomain
$changePW = TranslatePasswordOptions($domainInfo.AllowUserToChangePassword)
"Domain count: " + $domainInfo.Domains.Count | out-file $outputfile -append
$TrustedDomain = "-"
If($domainInfo.Domains.Count -gt 0)
{
	"Allow users to log on from: Trusted domains only" | out-file $outputfile -append
	"Default domain: " +$defDomain | out-file $outputfile -append
	if (($SFVersion) -ge 2.6)
	{
		$showDomains = YesNo($domainInfo.ShowDomainField)
		"ShowDomains: " + $showDomains | out-file $outputfile -append
	}
	else
	{
		"ShowDomains: -" | out-file $outputfile -append
	}
	ForEach($domain in $domainInfo.Domains)
	{
		$TrustedDomain = $TrustedDomain + "," + $domain
	}
	if($TrustedDomain.startsWith("-,"))
	{
		$TrustedDomain = $TrustedDomain.substring(2)
	}
}
Else
{
	"Allow users to logon from:  Any domain" | out-file $outputfile -append
	"Default domain: -" | out-file $outputfile -append
	"ShowDomains: -" | out-file $outputfile -append
}
"Trusted domains:" + $TrustedDomain | out-file $outputfile -append

"Allow users to change passwords: "+$changePW | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append
"Processing authentication end" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append

# stores
"Processing stores starts" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append
$accounts = @(((Get-DSGlobalAccounts).Accounts) | Sort Name)
foreach ($account in $accounts)
{
	$advertised = YesNo($account.Published)
	$store = Get-DSStoreServicesSummary | where {$_.FriendlyName -eq $account.Name}
	$friendlyName = $store.FriendlyName
	$URL = $store.Url
	if ($store.GatewayKeys.Count -gt 0)
	{
		$access = "Internal and external networks"
	}
	Else
	{
		$access = "Internal networks only"
	}

	if ($store.UseHttps -eq $True)
	{
		$status = "StoreFront using HTTPS"
	}
	Else
	{
		$status = "StoreFront using HTTP not HTTPS"
	}

	$locked = "-"
	$authenticated = "-"
	$filterTypes = "-"
	$filterKeywords = "-"
	$includeKeywords = "-"
	$excludeKeywords = "-"

	if (($SFVersion) -ge 2.5)
	{
		$locked = YesNo($store.IsLockedDown)
		$authenticated = YesNo(!$store.IsAnonymous)
		$filterTypes = Get-DSResourceFilterType $store.SiteID $store.VirtualPath
		$filterKeywords = Get-DSResourceFilterkeyword $store.SiteID $store.VirtualPath
		$includeKeywords = @($filterKeywords.Include)
		$excludeKeywords = @($filterKeywords.Exclude)
	}
	
	"General Details" | out-file $outputfile -append
	"~~~~~~~~~~~~~~~~" | out-file $outputfile -append
	"Store Name: " + $friendlyName | out-file $outputfile -append
	"Authenticated: " + $authenticated | out-file $outputfile -append
	"Advertised: " +$advertised | out-file $outputfile -append
	"Store URL: " + $URL | out-file $outputfile -append
	"Access: " +$access | out-file $outputfile -append
	"Status: " +$status | out-file $outputfile -append
	"Locked: " +$locked | out-file $outputfile -append

	" " | out-file $outputfile -append
	"Delivery Controllers" | out-file $outputfile -append
	"~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append
	$farmsets = @($store.Farmsets)
	foreach ($farmset in $farmsets)
	{
		$farms = @($farmset.Farms)
		foreach ($farm in $farms)
		{
			$farmName = $farm.FarmName 
			$farmType = $farm.FarmType
			$farmServers = $farm.Servers
			$transportType = $farm.TransportType
			$port = $farm.ServicePort
			$sslRelayPort = $farm.SSLRelayPort
			$loadBalance = YesNo($farm.LoadBalance)

			"Delivery Controller Name: " +$farmName | out-file $outputfile -append
			"Type: " +$farmType | out-file $outputfile -append
			"Servers: " + $farmServers | out-file $outputfile -append
			"XMLPort: " +$port | out-file $outputfile -append
		}
	}

	" " | out-file $outputfile -append
	"Integrate with Citrix Online" | out-file $outputfile -append
	"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~" | out-file $outputfile -append

	$GoToMeeting = YesNo($store.IsGoToMeetingEnabled)
	$GotoWebinar = YesNo($store.IsGoToWebinarEnabled)
	$GoToTraining = YesNo($store.IsGoToTrainingEnabled)
	"GoToMeeting: " +$GoToMeeting | out-file $outputfile -append
	"GoToWebinar: " +$GotoWebinar | out-file $outputfile -append
	"GoToTraining: " +$GoToTraining | out-file $outputfile -append

	# remote access
	$defaultGW = ""
	$fullVPN = $False
	$vpnService = Get-DSGlobalService -ServiceRef "VPN_$($store.ServiceRef)"
	$NSGatewayAppliances = "-"
	if($vpnService.Service)
	{
		$allGws = (Get-DSGlobalGateways).Gateways
		foreach($gw in $vpnService.Service.GatewayRefs)
		{
			$vpnGws = $allGws | Where-Object { $_.ID -eq $gw.RefId }
			$Script:fullVPN = $vpnService.Service.ServiceType -eq "VPN"
			if ($gw.Default) {$defaultGW = $vpnGws.Name}
			$NSGatewayAppliances = $NSGatewayAppliances + ", " + $($vpnGws.Name)
		}
	}
	else
	{
		$service = Get-DSGlobalService -ServiceRef $store.ServiceRef
		$allGws = (Get-DSGlobalGateways).Gateways
		foreach($gw in $service.Service.GatewayRefs)
		{
			$vpnGws = $allGws | Where-Object { $_.ID -eq $gw.RefId }
			$Script:fullVPN = $service.Service.ServiceType -eq "VPN"
			if ($gw.Default) {$defaultGW = $vpnGws.Name}
			$NSGatewayAppliances = $NSGatewayAppliances + ", " + $($vpnGws.Name)
		}
	}

	if($NSGatewayAppliances.startsWith("-,"))
	{
		$NSGatewayAppliances = $NSGatewayAppliances.substring(2)
	}

	"Netscaler Gateway Appliances: " + $NSGatewayAppliances | out-file $outputfile -append

	switch ($defaultGW)
	{
		""
		{
			"Default appliance: -" | out-file $outputfile -append
			"Remote access: None" | out-file $outputfile -append
		}
		default
		{
			"Default appliance: " +$defaultGW | out-file $outputfile -append
			if ($fullVPN)
			{
				"Remote access: Full VPN Tunnel" | out-file $outputfile -append
			}
			else
			{
				"Remote access: No VPN Tunnel"| out-file $outputfile -append
			}
		}
	}
	"XXXX----------------------------------------XXXX" | out-file $outputfile -append
}
"#------------------------------------------" | out-file $outputfile -append
"Processing stores end" | out-file $outputfile -append
"#------------------------------------------" | out-file $outputfile -append
# Receiver for Web
"Processing Receiver for Web sites starts" | out-file $outputfile -append
"#------------------------------------------" | out-file $outputfile -append
$receivers = @(Get-DSWebReceiversSummary)
foreach ($receiver in $receivers)
{
	$name = $receiver.FriendlyName
	$WebUrl  = $receiver.Url

	$ClassicExpr = "-"
	$authenticated = "-"
	$HTML5version = "-"
	$authMethods = "-"
	if (($SFVersion) -ge 2.5)
	{
		$authenticated = YesNo(!$receiver.IsAnonymousStore)
		$HTML5version = $receiver.HTML5Version
		$authMethods = @($receiver.AllowedAuthMethods)
	}

	if($receiver.ClassicEnabled)
	{
		$ClassicExpr = $receiver.ClassicEnabled
	}

	$storeURL = $receiver.StoreUrl
	$aStore = $storeURL -split "/"
	$store = $aStore[$aStore.Count-1]
	$deployment = TranslateHTML5Deployment($receiver.HTML5Configuration)
	$shortcuts = Get-DSAppShortcutsTrustedUrls $receiver.SiteId $receiver.VirtualPath

	"Name: " +$name | out-file $outputfile -append
	"Website URL: " +$WebUrl | out-file $outputfile -append
	"Classic Experience Enabled: "+ $ClassicExpr | out-file $outputfile -append
	"Store: " +$store | out-file $outputfile -append
	"Store URL: " + $storeURL | out-file $outputfile -append
	"Store Authenticated: "+$authenticated | out-file $outputfile -append
	"HTML5 Version: " +$HTML5version | out-file $outputfile -append

	# "------------------ " | out-file $outputfile -append
	# $name +"#~#"+ $WebUrl +"#~#"+ $store +"#~#"+ $storeURL +"#~#"+ $authenticated +"#~#"+ $HTML5version | out-file $outputfile -append
	# "------------------ " | out-file $outputfile -append

	if(($authMethods -ne $Null) -and ($authMethods.length -gt 0) -and ($authMethods -ne "-"))
	{
		$methods = ""
		foreach ($authMethod in $authMethods)
		{
			$method = TranslateMethod($authMethod)
			$methods = $methods +", "+$method
		}
		if($methods.startsWith(","))
		{
			$methods = $methods.substring(1)
		}
		"Authentication methods: " + $methods | out-file $outputfile -append
	}
	else
	{
		"Authentication methods: -" | out-file $outputfile -append
	}

	# $finalOutputLine = $finalOutputLine+ "#~#" + $methods
	# $finalOutputLine | out-file $outputfile -append

	if (($shortcuts -ne $Null))
	{
		$TrustedURLs = @($shortcuts.TrustedUrls)
		if($TrustedURLs)
		{
			"Shortcuts to Websites: " + $TrustedURLs | out-file $outputfile -append
		}else
		{
			"Shortcuts to Websites: -"| out-file $outputfile -append
		}

		if($deployment)
		{
			"Deployment: " +$deployment | out-file $outputfile -append
		}else
		{
			"Deployment: -" | out-file $outputfile -append
		}
		# $finalOutputLine = $finalOutputLine+ "#~#" + $TrustedURLs + "#~#" +$deployment
	}
	else
	{
		"Shortcuts equal to Null" | out-file $outputfile -append
		"Shortcuts to Websites: -" | out-file $outputfile -append
		"Deployment: -" | out-file $outputfile -append
	}
	"XXXX----------------------------------------XXXX" | out-file $outputfile -append
}
"#----------------------------------------" | out-file $outputfile -append
"Processing Receiver for Web sites ends" | out-file $outputfile -append
"#----------------------------------------" | out-file $outputfile -append

# NetScaler Gateways
"Processing NetScaler Gateways starts" | out-file $outputfile -append
"#--------------------------------------" | out-file $outputfile -append
$gateways = @((Get-DSGlobalGateways).Gateways)
foreach ($gateway in $gateways)
{
	$name = $gateway.Name
	$used = "Yes"
	$url = $gateway.Address
	$NSversion = TranslateNSVersion($gateway.AccessGatewayVersion)
	$callbackURL = $gateway.CallbackURL
	$deploymentMode = $gateway.DeploymentType
	$STAs = $gateway.SecureTicketAuthorityURLs
	if ($gateway.SessionReliability -eq $True) {$sessionReliability="Yes"} Else {$sessionReliability="No"}
	if ($gateway.RequestTicketTwoSTA -eq $True) {$request2STATickets="Yes"} Else {$request2STATickets="No"}
	"Display Name: "+$name | out-file $outputfile -append
	"Used by Services: " + $used | out-file $outputfile -append

	"URL: " +$url | out-file $outputfile -append
	"Version: " +$NSversion | out-file $outputfile -append

	if($NSversion -ne "5.x")
	{
		$logonType = TranslateLogonType($gateway.Logon)
		$smartCardFallback = TranslateLogonType($gateway.SmartCardFallback)
		$subnetIPaddress = $gateway.IPAddress
		"Subnet IP address: " + $subnetIPaddress | out-file $outputfile -append
		"Logon type: " + $logonType | out-file $outputfile -append
		"Smart card fallback:" + $smartCardFallback | out-file $outputfile -append
	}
	else
	{
		"Subnet IP address: -"| out-file $outputfile -append
		"Logon type: -" | out-file $outputfile -append
		"Smart card fallback: -" | out-file $outputfile -append
	}

	"Callback URL:" +$callbackURL | out-file $outputfile -append
	"Secure Ticket Authority URLs: " + $STAs | out-file $outputfile -append	
	"Enable session reliability : " + $sessionReliability | out-file $outputfile -append
	"Request tickets from two STAs, where available: " +$request2STATickets | out-file $outputfile -append
	"XXXX----------------------------------------XXXX" | out-file $outputfile -append
}
"#------------------------------------" | out-file $outputfile -append
"Processing NetScaler Gateways ends" | out-file $outputfile -append
"#------------------------------------" | out-file $outputfile -append

# Beacons
"Processing Beacons starts" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append
$internalBeacons = @((Get-DSGlobalBeacons "Internal").Beacons)
$externalBeacons = @((Get-DSGlobalBeacons "External").Beacons)

if ($internalBeacons.Count -gt 0)
{
	$intrBeacons = ""
	foreach ($beacon in $internalBeacons)
	{
		$beaconAddress = ($beacon).Address
		$intrBeacons = $intrBeacons+", "+$beaconAddress
	}
	if ($intrBeacons.startsWith(","))
	{
		$intrBeacons = $intrBeacons.substring(1)
	}
	"Internal: "+$intrBeacons | out-file $outputfile -append
}

if ($externalBeacons.Count -gt 0)
{
	$extBeacons = ""
	foreach ($beacon in $externalBeacons)
	{
		$beaconAddress = ($beacon).Address
		$extBeacons = $extBeacons+", "+$beaconAddress
	}
	if ($extBeacons.startsWith(","))
	{
		$extBeacons = $extBeacons.substring(1)
	}
	"External: " +$extBeacons | out-file $outputfile -append
}
"#-------------------------------" | out-file $outputfile -append
"Processing Beacons ends" | out-file $outputfile -append
"#-------------------------------" | out-file $outputfile -append

"#EG_SCRIPT_DONE#" | out-file $outputfile -append







# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQU4yuYA9GowLWyPs5o1zkSlRs/
# jPOgghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFCt4Z5EJJREYcnBB
# 1v8ogVSoTQmBMA0GCSqGSIb3DQEBAQUABIICACB/wVDUFDcma5X7Sg1z8+UIv3+P
# MolbnXKZF0Xu0KHx33wy2S1GYiKXVcS10RpQnX33RK0PB4NuADn0iphYGHK0FaFZ
# OfzkU3v0yM5qxVfOqal3AXZSeOEzixnnoI4I1Z6ZCI9f0/GoWIWvW9P4FBH9A0Ui
# fYM3PjcLGHY3LBqHc4u6nBfxEqI3WyGtL2vORUGxQVuDhao8jLDtJ9HVE/qG5O+S
# cyXIF111EnSI4rIfH/AENgafj/1j6ZWpS5HGRFiwR7d+0y01nj5lWmxomjSjULpt
# fkdjQmNWZuzo29ueEzZmha4iOp4vgkZKGIXzW/gn9wKRcE41zkb8xSZUoa9Ei2jj
# f6gz6bqWnWhI6ExpAe9AMJ+TpPGom2REnEwQbuG6tr3ORFwJCmPb8t9P5D78FYuf
# sUpqNccE2EVbU5bUxWFmp0IkzYzjasjkOZKEb1HVysgsxwQsh42tw//EyD1hy4l7
# aR5YEZj+K9AsREbtEGRhgu4qK/7z2bJrzUTCnYcfiBbg19pwi3k/UUqBm0kWEbI3
# 2ycArJ0ZCjwZontGwDBcjB/xCv7DNY59g04kg9j+Q10/gSrOWrlDICJwxh4Setww
# vok/pYgQ33lzkt1FHkFIHmN5vxawYW+XV+/B7BRbV33qrBceNs8cph68y9XBrKSL
# ydAHYD9l8B9KCZhhoYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MTA5MDExMjM2NThaMC8GCSqGSIb3DQEJBDEiBCDcpuzKgukd+Y/Uhwfu7vV5Z2L9
# mEg9S43XfyOyb70yEDANBgkqhkiG9w0BAQEFAASCAQCHyHVkc6eBx7FSeRDv0nsD
# TDEuABQ6TeAAzzd4Wf+qkRzWQzMieyKgGhEynFXN8ZRCGwQlCrmO78Xx/PyZW77r
# 2rKappx+vbR3+s7FW9wYmNwAjBSYTOF8ILq0K00f6NY3WD/cVeMMmbxTo2rOtj/t
# 2uj8ZF581zhf0PCaZRZDOZUUROFw6sDrKNo41jr0hgB/oj1iZWsbKk09ePYyDH6v
# Nikn7CHaLjYltlN7LoO3yYuoO2YULbWrYDl/r/azNbA0SiFAneCTdYZoWN8rS0Jk
# 5Cz3QJZSMxeLrT+1fA5jTMk2S2ac5zcwkIr7Fk7myLRn6Z7sw9jQytHbsbrWKEoy
# SIG # End signature block
