$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#REQUIRES -Version 2.0

<#
.DESCRIPTION
    This script is used to get FailoverCluster information
.NOTES  
    File Name      : FailoverCluster.ps1  
    Author         : T.R.Ranjithkumar
    Created        : 22 July 2013
    Prerequisite   : PowerShell V2 over Vista and upper
#>

Import-Module FailoverClusters -EA SilentlyContinue
$isPassive = "no"
if($args.Count -gt 0)
{
    $isPassive = $args[0]
}   
#Write-Host "isPassive" ($isPassive)

# Check the Clustere Service installed or not installed
$hostName = $env:computername
$isCluster = "False"
$clusterService = Get-Service -ComputerName $hostName -Name 'ClusSvc' -EA SilentlyContinue | Select-Object -expand Status 
if ($clusterService -ne $null)
{
    $isCluster = "True"
    Write-Host "Cluster Status"
    Write-Host "-----------------"
    if ($clusterService -eq "Running")
    {
        Write-Host ($isCluster, $clusterService) -Separator "~!~"
    }
    else
    {
        Write-Host ($isCluster, $clusterService) -Separator "~!~"
        return
    }
    Write-Host ""
}
else
{
    Write-Host "Cluster Service not installed"
    return
}
# Suppose the cluster service is not installed, further we are not going to do anything, so return
# Ends here

$No_of_nodes = 0

$Cluster = Get-Cluster
if($null -ne $Cluster)
{
    $ClusterName = $Cluster.Name
    $ClusterDomainName = $Cluster.Domain
}
else 
{
    return
}

#$keyClusetName = (Get-Cluster).Name

# Check the Active node
$isOwner = "False"
try
{
    $ClusterIP = [System.Net.Dns]::GetHostAddresses($ClusterName)
}
catch [Exception]
{
    #write-host $_.Exception.GetType().FullName;
    #write-host $_.Exception.Message;
}
if($null -eq $ClusterIP)
{
    return
}
$CurrentIPs = @();
get-wmiobject win32_networkadapterconfiguration | ? { $_.IPAddress -ne $null } | Sort-Object IPAddress -Unique | % { $CurrentIPs+=$_.IPAddress }
foreach ($tempIP in $CurrentIPs -split(" "))
{
    if ($ClusterIP -like $tempIP)
    {
        $isOwner = "True"
        break
    }
}

Write-Host "ActiveNode" ($ClusterName, $clusterIP, $isOwner, $hostName) -Separator "~!~"
if ($isOwner -like "false")
{
    if ($isPassive -like "no")
    {
        return
    }
}

#Write-Host "Nodes:" ($No_of_nodes)
# Suppose this is not an active node, return the code
# Ends here

# Cluster service is running and this is an active node
# further we are getting cluster information
Write-Host "Cluster Resource"
Write-Host "-----------------"
$ClusterResources = Get-ClusterResource
if($null -ne $ClusterResources)
{
    foreach($ClusterResource in $ClusterResources)
    {
        $ClusterResourceName = $ClusterName+":"+$ClusterResource.Name
        $ClusterResourceState = $ClusterResource.State
        $ClusterResourceCluster = $ClusterResource.Cluster
        $ClusterResourceOwnernode = $ClusterResource.OwnerNode
        $ClusterResourceResourcetype = $ClusterResource.ResourceType
        $ClusterResourceOwnergroup = $ClusterResource.OwnerGroup
        $ClusterResourceIsCoreResource = $ClusterResource.IsCoreResource
        Write-Host ($ClusterResourceName, $ClusterResourceState, $ClusterResourceCluster, $ClusterResourceOwnernode, $ClusterResourceResourcetype, $ClusterResourceOwnergroup,$ClusterResourceIsCoreResource) -Separator "~!~"
    }
}
Write-Host ""

$No_of_groups = 0
Write-Host "Cluster Groups"
Write-Host "---------------"
$ResourceGroups = Get-ClusterGroup
if ($null -ne $ResourceGroups)
{
    foreach($ResourceGroup in $ResourceGroups)
    {
        $ResourceGroupName = $ResourceGroup.Name
        if ($ResourceGroupName -like "Cluster Group" -or $ResourceGroupName -like "Available Storage")
        {
            continue // These two lists are default groups so skip this group
        }
        $ResourceGroupState = $ResourceGroup.State
        $ResourceGroupCluster = $ResourceGroup.Cluster
        $ResourceGroupOwnernode = $ResourceGroup.OwnerNode
        Write-Host ($ResourceGroupName, $ResourceGroupState, $ResourceGroupCluster, $ResourceGroupOwnernode) -Separator "~!~"
        $No_of_groups++
    }
}
Write-Host ""

#Write-Host "Groups:" ($No_of_groups)

Write-Host "Cluster Shared Volume"
Write-Host "----------------------"
$csvs = Get-ClusterSharedVolume
if($null -ne $csvs)
{
    foreach ($csv in $csvs)
    {
        $Name        = $ClusterName+":"+$csv.Name
        $state       = $csv.State
        $ownernode   = $csv.OwnerNode
        #Write-Host "State#####" ($state)
        if ($state -eq "Offline" -or $state -eq "Failed")
        {
            Write-Host ($Name, $state, $ownernode, "-", "0.00", "0.00", "0.00", "0.00") -Separator "~!~"
        }
        else
        {
            $csvinfos = $csv | select -Property Name -ExpandProperty SharedVolumeInfo
            foreach ($csvinfo in $csvinfos)
            {
                $Name        = $ClusterName+":"+$csv.Name
                $state       = $csv.State
                $ownernode   = $csv.OwnerNode
                $Path        = $csvinfo.FriendlyVolumeName
                [float]$Size        = ($csvinfo.Partition.Size / 1GB)
                [float]$FreeSpace   = ($csvinfo.Partition.FreeSpace / 1GB)
                [float]$UsedSpace   = ($csvinfo.Partition.UsedSpace / 1GB)
                if($Size -eq 0)
                {
                    [float]$PercentUsed = 0
                }
                else {
                    [float]$PercentUsed = ($UsedSpace /  $Size * 100)
                }    
                Write-Host ($Name, $state, $ownernode, $Path, $Size, $FreeSpace, $UsedSpace, $PercentUsed) -Separator "~!~"
            }
        }    
    }
}
Write-Host ""

Write-Host "Cluster Network Interface"
Write-Host "--------------------------"
$ClusterInterfaces = Get-ClusterNetworkInterface
if ($null -ne $ClusterInterfaces)
{
    foreach ($ClusterInterface in $ClusterInterfaces)
    {
        [String]$name = $ClusterInterface.Name
        $state = $ClusterInterface.State
        $cluster = $ClusterInterface.Cluster
        $network = $ClusterInterface.Network
        $node = $ClusterName+":"+$ClusterInterface.Node
        $adapter = $ClusterInterface.Adapter
        $adapterId = $ClusterInterface.AdapterId
        #$address = $Item.Address

        $address = ""
        $ipv4Address = ""
        $ipv6Address = ""

        $ipv4Addresses = $ClusterInterface.Ipv4Addresses
        if ($ipv4Addresses -ne $null -and $ipv4Addresses.count -gt 0)
        {
            $count = 0
            foreach ($ipv4 in $ipv4Addresses)
            {
                if ($count -eq 0)
                {
                    $ipv4Address = $ipv4
                }
                else
                {
                    $ipv4Address = $ipv4Address + "," + $ipv4
                }
                $count = $count + 1
            }
        }

        $ipv6Addresses = $ClusterInterface.Ipv6Addresses
        if ($ipv6Addresses -ne $null -and $ipv6Addresses.count -gt 0)
        {
            $count = 0
            foreach ($ipv6 in $ipv6Addresses)
            {
                if ($count -eq 0)
                {
                    $ipv6Address = $ipv6
                }
                else
                {
                    $ipv6Address = $ipv6Address + "," + $ipv6
                }
                $count = $count + 1
            }
        }

        if ($ipv4Address.length -gt 0 -and $ipv6Address.length -gt 0)
        {
            $address = $ipv4Address +","+$ipv6Address
        }
        elseif ($ipv4Address.length -gt 0)
        {
            $address = $ipv4Address
        }
        elseif ($ipv6Address.length -gt 0)
        {
            $address = $ipv6Address
        }
        Write-Host ($node.toLower(), $name, $state, $network, $adapter, $adapterId, $address) -Separator "~!~" 
    }
}
Write-Host ""

Write-Host "Cluster Node"
Write-Host "-------------"
$ClusterNodes = Get-ClusterNode
if ($null -ne $ClusterNodes)
{
    foreach ($ClusterNode in $ClusterNodes)
    {
        $name = $ClusterName+":"+$ClusterNode.Name
        $state = $ClusterNode.State
        $cluster = $ClusterNode.Cluster
        $nodename = $ClusterNode.NodeName
        $csdversion = $ClusterNode.CSDVersion
        Write-Host ($name, $state, $cluster) -Separator "~!~"
        $No_of_nodes++
    }
}
Write-Host ""

Write-Host "Network Cluster"
Write-Host "----------------"
$ClusterNetworks = Get-ClusterNetwork
if ($null -ne $ClusterNetworks)
{
    foreach ($ClusterNetwork in $ClusterNetworks)
    {
        $name = $ClusterName+":"+$ClusterNetwork.Name
        $state = $ClusterNetwork.State
        $cluster = $ClusterNetwork.Cluster
        $address = $ClusterNetwork.Address
        $addressmask = $ClusterNetwork.AddressMask
        Write-Host ($name, $state) -Separator "~!~" 
    }
}
Write-Host ""

$ClusterFQDN = $ClusterName+"."+$ClusterDomainName
Write-Host "Cluster Summary"
Write-Host "----------------"
Write-Host ($ClusterName, $No_of_groups, $No_of_nodes, $ClusterFQDN, $hostName) -Separator "~!~"
Write-Host ""

Write-Host "Available Disk"
Write-Host "----------------"
$AvailableStorageDisks = Get-ClusterGroup "Available Storage" | Get-ClusterResource
if ($null -ne $AvailableStorageDisks)
{
    foreach ($AvailableStorageDisk in $AvailableStorageDisks)
    {
        $name = $ClusterName+":"+$AvailableStorageDisk.Name
        $state = $AvailableStorageDisk.State
        $group = $AvailableStorageDisk.OwnerGroup
        $type = $AvailableStorageDisk.ResourceType
        Write-Host ($name, $state, $group, $type) -Separator "~!~" 
    }
}
Write-Host ""

Write-Host "Groups Resources"
Write-Host "---------------"
$ResourceGroups = Get-ClusterGroup
if ($null -ne $ResourceGroups)
{
    foreach($ResourceGroup in $ResourceGroups)
    {
        $ResourceGroupName = $ResourceGroup.Name
        if ($ResourceGroupName -like "Cluster Group" -or $ResourceGroupName -like "Available Storage")
        {
            continue // These two lists are default groups so skip this group
        } 
        $ResourceGroupState = $ResourceGroup.State
        $ResourceGroupOwnernode = $ResourceGroup.OwnerNode
        $Resources = Get-ClusterGroup $ResourceGroupName | Get-ClusterResource
        $ResourceGroupName = $ClusterName+":"+$ResourceGroup.Name
        foreach ($Resource in $Resources)
        {
            $ResourceName = $ResourceGroupName+":"+$Resource.Name
            $ResourceState = $Resource.State
            $ResourceCluster = $Resource.Cluster
            $ResourceOwnerNode = $Resource.OwnerNode
            $ResourceType = $Resource.ResourceType
            $ResourceOwnerGroup = $Resource.OwnerGroup 
            Write-Host ($ResourceGroupName, $ResourceGroupState, $ResourceGroupOwnernode, $ResourceName, $ResourceState, $ResourceCluster, $ResourceOwnerNode, $ResourceType, $ResourceOwnerGroup) -Separator "~!~"
        }
    }
}
Write-Host ""

Write-Host "Cluster Disks"
Write-Host "--------------"
$resources = Get-WmiObject -namespace root\MSCluster MSCluster_Resource -filter "Type='Physical Disk'"
if ($resources -ne $null)
{
    foreach($res in $resources)
    {
        $Name = $res.Name
        $Status = $res.State
        if($Status -eq 3 -or $Status -eq 4 -or $Status -eq 130)
        {
        
            Write-Host "Name           : " $ClusterName":"$Name
            Write-Host "Status         : " $Status
            Write-Host
        }
        else
        {
            $disks = $res.GetRelated("MSCluster_Disk")
            $disks | foreach {
            $_.GetRelated("MSCluster_DiskPartition") |
            select @{N="Name"; E={$($ClusterName)+":"+$res.Name}}, @{N="Status"; E={$res.State}}, @{N="IsSharedVolume"; E={$res.IsClusterSharedVolume}},Path, VolumeLabel, TotalSize, FreeSpace 
            }
        }
    }
}



# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQU7lPpfTCAdwELfJqkDgvarBLQ
# Md6gghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFLxDeTQGcimBDhmt
# DLuVn5nyoAowMA0GCSqGSIb3DQEBAQUABIICADpka0+DIB1oH0Itk1vTjXtjGUze
# IKD9QW45iumhsPp8dSR+s+LRnePZ5Cwg66uY7FG3DRi7jisD5d2FW3QDDnUquJos
# +XlyaUIVF+prho9pdoYGMCzlpIsh2U+5HO1IPXoV/Mu1bh37DDABiEy30yU+RX9f
# od6Jy2S4+dDzBFqXLfLZtJVj4KMCnW1rgkH/AMI+i4+PmVSURQcZHdEMaJ44NkXU
# 1yyfNNesTzw4HU7YSmE8aK+ypqacGFDOHNcUPd9vuUQ28fSd+fcY99ayzZcpcwyu
# C8ehJIj3PP0PVQFs0p45Xi/C8es/TUV5Nh7CFUMJ5EbEE+N8852ggxBadv/cBrnj
# QzVnACoQxUPmYBXW92r3MxNA+AmJozNI+9SvpYApDD/xYI+EtRZJHDgc3o/O8tvk
# N6Vsl5M3C0/UX0r2Lk6YkpzPE+xTU4NT9gJNB3+z/wDaoW+NvyPFzVYZMIDIiTQT
# 2QPTtTYaagnQM3XfNs5Z/+Y+rEF1qje/Ezbt7pnXz8VKsW9fXJBjjTn9mWq9Io12
# w3stze0PJgeBqsLrbvpEfSdluKHN7+zyOYM/uwNiM+CuVsPimR/etJaXc2O5Dftt
# y+3+bWt6lHpZXudqb+MKu5YZV0Q+JwTnOKQjiMjHhAWRkuSY3Cx7ArVUNiROfbUC
# 2V1HCczhfcOKoVpDoYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MTA5MDExMjM2MDhaMC8GCSqGSIb3DQEJBDEiBCCyOp3Qq7IZd188ERxcmL7HppXV
# kH8s4rbw8lspPlsW/jANBgkqhkiG9w0BAQEFAASCAQCXX0e0/0EqRoKkcWSlS9MW
# Pu9cpwsYJrQqxJLUUTy0VFb/tvdGZYSmvneeA6nJeLNd9hSOpWAVglOqtwlVK2k6
# ZrZZsEyIKtjO/Y93GLPLFDSoXnsAR/GG7zZPtrqHXWpnTgcxhsjzWO630Xe+VBtq
# AO8qTQTnUVnXv8FqS9VWjrbos2LPm9h0R36sAZsurBTUfHDs2wxVI7nezB1EB/NQ
# LkD71qbNNRWpdrKqCxHbqjmhxvxYGeNWddfZNPD5jp2OOxwQ0lqYCYcCYRwFC7ce
# 4LZpbmczCaRRb6vwN0u8i3eWO1ikgnLCDEikzdxX+EUSClHQ0wb4z6BYwJXtjz4Y
# SIG # End signature block
