$Global:ErrorActionPreference="SilentlyContinue"
$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

if($args.Count -eq 0)
{
	"Syntax: CitrixCloudMonitor.ps1 <CustomerId> <Path_of_secureclientfile> [Optional]<eG_Install_Dir> [Optional]<ProxySettings>"
	return
}

function GetBearerToken
{
	param (
	[Parameter(Mandatory=$true)]
	[string] $clientId,
	[Parameter(Mandatory=$true)]
	[string] $clientSecret
	)
  
	$postHeaders = @{"Content-Type"="application/json"}
	$body = @{
	"ClientId"=$clientId;
	"ClientSecret"=$clientSecret
	}
	$trustUrl = "https://trust.citrixworkspacesapi.net/root/tokens/clients"

	$response = Invoke-RestMethod -Uri $trustUrl -Method POST -Body (ConvertTo-Json $body) -Headers $postHeaders
	$bearerToken = $response.token

	return $bearerToken;
}

Add-PSSnapin Citrix*
$ProxyHost = 'none'
$ProxyUser = 'none'
$ProxyPwd = 'none'

if($args.Length -eq 2)
{
	$CustomerId = $args[0]
	$SecureClientFile = $args[1]
}
elseif($args.Length -eq 4)
{
	$CustomerId = $args[0]
	$SecureClientFile = $args[1]
	$eGInstallDir = $args[2]
	# The format is https://ip:port#user#password
	$ProxyArr = $args[3].ToString().Split('#') 
	if($ProxyArr.Length -eq 3)
	{
		$ProxyHost = $ProxyArr[0]
		$ProxyUser = $ProxyArr[1]
		$ProxyPwd = $ProxyArr[2]
		$OEPath = $eGInstallDir + "\lib\OE.exe"
		$ProxyPwd = & $OEPath $ProxyPwd
	}
}

if($ProxyHost -ne 'none')
{
	[system.net.webrequest]::defaultwebproxy = new-object system.net.webproxy($ProxyHost)
	if($ProxyUser -ne 'none' -and $ProxyPwd -ne 'none')
	{
		$proxyCred = New-Object -TypeName System.Management.Automation.PSCredential -argumentlist $ProxyUser, $(convertto-securestring $ProxyPwd -asplaintext -force)
		[system.net.webrequest]::defaultwebproxy.credentials = $proxyCred   #[System.Net.CredentialCache]::DefaultNetworkCredentials
		[system.net.webrequest]::defaultwebproxy.BypassProxyOnLocal = $true
	}
}

Set-XDCredentials -CustomerId $CustomerId -SecureClientFile $SecureClientFile -ProfileType CloudApi -StoreAs "default"
Get-XDCredentials -ProfileName "default" | out-null

<#
Function Get-BrokerMachinesPaged
{
	param (
	$DesktopGroupUid,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "MachineName")
	$desktops = @()
	$desktops += Get-BrokerMachine -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($desktops.count / $total * 100, 1)
		#Write-Host  "Getting Broker Machines. Retrieved $($desktops.count) of $total machines completed in $remainingPerc%"
		$last = $desktops[-1].MachineName
		$desktops += Get-BrokerMachine -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { MachineName -gt $last } @argumentList
	}
	return $desktops
}

Function Get-BrokerApplicationsPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($applications.count / $total * 100, 1)
		#Write-Host  "Getting Broker Applications. Retrieved $($applications.count) of $total applications completed in $remainingPerc%"
		$last = $applications[-1].Uid
		$applications += Get-BrokerApplication -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { Uid -gt $last } @argumentList
	}
	return $applications
}

Function Get-BrokerSessionsPaged
{
	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$sessions = @()
	$sessions += Get-BrokerSession -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue @argumentList
	$total = $RecordCount.TotalAvailableResultCount
	while ($RecordCount.TotalAvailableResultCount -gt 250)
	{
		$remainingPerc= [math]::round($sessions.count / $total * 100, 1)
		#Write-Host  "Getting Broker Sessions. Retrieved $($sessions.count) of $total sessions completed in $remainingPerc%"
		$last = $sessions[-1].Uid
		$sessions += Get-BrokerSession -ReturnTotalRecordCount:$true -ErrorVariable RecordCount -ErrorAction SilentlyContinue -Filter { Uid -gt $last } @argumentList
	}
	return $sessions
}
#>

Function Get-BrokerMachinesPaged
{
	param (
	$DesktopGroupUid,
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$desktops = @()
	$LastUid = 0;
	do
	{
		$NewDesktops = @(Get-BrokerMachine -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewDesktops.Count -gt 0)
		{
			$desktops += $NewDesktops
			$LastUid = $desktops[-1].Uid
		}
	} while ($NewDesktops.Count -ge $MaxRecordCount)
	return $desktops
}

Function Get-BrokerApplicationsPaged
{
	param (
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$applications = @()
	$LastUid = 0;
	do
	{
		$NewApplications = @(Get-BrokerApplication -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewApplications.Count -gt 0)
		{
			$applications += $NewApplications
			$LastUid = $applications[-1].Uid
		}
	} while ($NewApplications.Count -ge $MaxRecordCount)
	return $applications
}

Function Get-BrokerSessionsPaged
{
	param (
	$MaxRecordCount = 250
	)

	$argumentList = $PSBoundParameters
	$argumentList.Add("SortBy", "Uid")
	$sessions = @()
	$LastUid = 0;
	do
	{
		$NewSessions = @(Get-BrokerSession -MaxRecordCount $MaxRecordCount -ErrorAction SilentlyContinue -Filter {Uid -gt $LastUid} @argumentList)
		if ($NewSessions.Count -gt 0)
		{
			$sessions += $NewSessions
			$LastUid = $sessions[-1].Uid
		}
	} while ($NewSessions.Count -ge $MaxRecordCount)
	return $sessions
}

$ZoneControllersHash = @{}
$ConfiguredZones = Get-ConfigZone -EA 0
foreach($Zone in $ConfiguredZones)
{
	$ZoneControllersHash.Add($Zone.Name,$Zone.ControllerSids)
}

Write-Host "Site"
Write-Host "----"
$Site = Get-BrokerSite
if ($Site -ne $null)
{
	$LicenseEdition = $Site.LicenseEdition
	if ($LicenseEdition -eq $null -or $LicenseEdition.length -eq 0)
	{
		$LicenseEdition = "-"
	}

	$LicenseModel = $Site.LicenseModel
	if ($LicenseModel -eq $null -or $LicenseModel.length -eq 0)
	{
		$LicenseModel = "-"
	}

	$LicensingGraceHoursLeft = $Site.LicensingGraceHoursLeft
	if($LicensingGraceHoursLeft -eq $null -or $LicensingGraceHoursLeft.length -eq 0)
	{
		$LicensingGraceHoursLeft = "-"
	}

	$IsSecondaryBroker = $Site.IsSecondaryBroker
	if ($IsSecondaryBroker -eq $null -or $IsSecondaryBroker.length -eq 0)
	{
		$IsSecondaryBroker = "-"
	}

	$LocalHostCacheEnabled = $Site.LocalHostCacheEnabled
	if ($LocalHostCacheEnabled -eq $null -or $LocalHostCacheEnabled.length -eq 0)
	{
		$LocalHostCacheEnabled = "-"
	}

	$ReuseMachinesWithoutShutdownInOutageAllowed = $Site.ReuseMachinesWithoutShutdownInOutageAllowed
	if ($ReuseMachinesWithoutShutdownInOutageAllowed -eq $null -or $ReuseMachinesWithoutShutdownInOutageAllowed.length -eq 0)
	{
		$ReuseMachinesWithoutShutdownInOutageAllowed = "-"
	}

	$TotalUniqueLicenseUsers = -5
	if ($Site.TotalUniqueLicenseUsers -ne $null)
	{
		$TotalUniqueLicenseUsers = $Site.TotalUniqueLicenseUsers
	}

	$PeakConcurrentLicenseUsers = -5
	if ($Site.PeakConcurrentLicenseUsers -ne $null)
	{
		$PeakConcurrentLicenseUsers = $Site.PeakConcurrentLicenseUsers
	}

	Write-Host ($Site.Name,$Site.LicenseServerName,$Site.LicenseServerPort,$Site.LicensedSessionsActive,
	$LicenseEdition,$LicenseModel,$Site.LicensingBurnInDate,$Site.DnsResolutionEnabled,$Site.SecureIcaRequired,
	$Site.TrustRequestsSentToTheXmlServicePort,
	$Site.ConnectionLeasingEnabled,$Site.LicensingGracePeriodActive,$Site.LicensingOutOfBoxGracePeriodActive,$LicensingGraceHoursLeft,
	$IsSecondaryBroker,$LocalHostCacheEnabled,$ReuseMachinesWithoutShutdownInOutageAllowed,$TotalUniqueLicenseUsers,$PeakConcurrentLicenseUsers) -Separator ";"
}

Write-Host "Connectors"
Write-Host "----------"

$EdgeServers = Get-ConfigEdgeServer
if ($EdgeServers -ne $null)
{
	foreach ($EdgeServer in $EdgeServers)
	{
		Write-Host ($EdgeServer.MachineAddress,$EdgeServer.ZoneName) -Separator ";"
	}
}

Write-Host "Hypervisor Details"
Write-Host "------------------"
$HypConnPath = "XDHyp:\Connections\"
$HypConnections = Get-ChildItem $HypConnPath
if ($HypConnections -ne $null)
{
	foreach ($HypConn in $HypConnections)
	{
		$ConnectionType = $HypConn.ConnectionType
		$HypervisorConnectionName = $HypConn.HypervisorConnectionName
		$HypervisorAddress = $HypConn.HypervisorAddress
		if ($HypervisorAddress -eq $null -or $HypervisorAddress.length -eq 0)
		{
			$HypervisorAddress = "-"
		}
		else
		{
			$HypvConn = ""
			$count = 0
			foreach ($key in $HypervisorAddress)
			{
				if ($count -eq 0)
				{
					$HypvConn = $key
				}
				else
				{
					$HypvConn = $HypvConn + "#" + $key
				}
				$count = $count + 1
			}
			if ($HypvConn -eq $null -or $HypvConn.length -eq 0)
			{
				$HypervisorAddress = "-"
			}
			else
			{
				$HypervisorAddress = $HypvConn
			}
		}
		$MaintenanceMode = $HypConn.MaintenanceMode
		$Persistent = $HypConn.Persistent
		$SupportsLocalStorageCaching = $HypConn.SupportsLocalStorageCaching
		$SupportsPvsVMs = $HypConn.SupportsPvsVMs
		$UserName = $HypConn.UserName
		$Scopes = $HypConn.Scopes
		if ($Scopes -eq $null -or $Scopes.length -eq 0)
		{
			$Scopes = "-"
		}
		$HypZone = $HypConn.ZoneName
		if($HypZone -eq $null -or $HypZone.length -eq 0)
		{
			$HypZone = "-"
		}
		# Hypervisor State
		# Unavailable - The broker is unable to contact the hypervisor.
		# InMaintenanceMode - The hosting server is in maintenance mode.
		# On - The broker is in contact with the hypervisor.
		$HyperConns = Get-BrokerHypervisorConnection -HypHypervisorConnectionUid $HypConn.HypervisorConnectionUid
		if($HyperConns -ne $null)
		{
			$State = $HyperConns.State
			$PreferredController = $HyperConns.PreferredController
			$IsReady = $HyperConns.IsReady
			$MachineCount = $HyperConns.MachineCount
			Write-Host ($HypervisorConnectionName,$State,$ConnectionType,$HypervisorAddress,$PreferredController,$MaintenanceMode,
			$Persistent,$SupportsLocalStorageCaching,$SupportsPvsVMs,$UserName,$Scopes,$HypZone,$IsReady,$MachineCount) -Separator ";"
		}
	}
}

Write-Host "Controllers"
Write-Host "-----------"
$Controllers = Get-BrokerController
if($Controllers -ne $null)
{
	foreach($Controller in $Controllers)
	{
		$DNSName = $Controller.DNSName
		$ControllerSid = $Controller.SID
		if ($DNSName -eq $null -or $DNSName.length -eq 0)
		{
			$DNSName = "-"
		}
		$ControllerZoneName = "-"
		if($ZoneControllersHash.Count -gt 0)
		{
			$ZoneKeys = $ZoneControllersHash.KEYS.GetEnumerator()
			foreach($ZoneKey in $ZoneKeys)
			{
				$ZoneValues = $ZoneControllersHash[$ZoneKey]
				if($ZoneValues.Contains($ControllerSid) -eq $true)
				{
					$ControllerZoneName = $ZoneKey
				}
			}
		}
		$LicensingGraceState = $Controller.LicensingGraceState
		if ($LicensingGraceState -eq $null -or $LicensingGraceState.length -eq 0)
		{
			$LicensingGraceState = "-"
		}
		$LicensingServerState = $Controller.LicensingServerState
		if ($LicensingServerState -eq $null -or $LicensingServerState.length -eq 0)
		{
			$LicensingServerState = "-"
		}
		Write-Host($DNSName,$Controller.MachineName,$Controller.State,$Controller.DesktopsRegistered,
		$Controller.ControllerVersion,$Controller.OSType,$Controller.OSVersion,$Controller.LastActivityTime,
		$Controller.LastStartTime,$ControllerZoneName,$LicensingGraceState,$LicensingServerState) -Separator ";"
	}
}

Write-Host "Controller Services"
Write-Host "-------------------"
if($Controllers -ne $null)
{
	foreach($Controller in $Controllers)
	{
		$DNSName = $Controller.DNSName
		if ($DNSName -eq $null -or $DNSName.length -eq 0)
		{
			continue
		}
		#$DNSName = $DNSName.Substring($DNSName.indexOf('.'))
		$State = $Controller.State
		if ($State -eq $null -or $State.length -eq 0)
		{
			continue
		}
		if ($State -eq "Off" -or $State -eq "Failed")
		{
			continue
		}
		$BrokerStatus = Get-BrokerServiceStatus -AdminAddress $DNSName
		if ($BrokerStatus -eq $null -or $BrokerStatus.length -eq 0)
		{
			$BrokerStatus = "-"
		}
		$ConfigStatus = Get-ConfigServiceStatus -AdminAddress $DNSName
		if ($ConfigStatus -eq $null -or $ConfigStatus.length -eq 0)
		{
			$ConfigStatus = "-"
		}
		Write-Host($DNSName,$BrokerStatus,$ConfigStatus) -Separator ";"
	}
}

Write-Host "Machine Catalogs"
Write-Host "----------------"
$Catalogs = Get-BrokerCatalog -MaxRecordCount 999
if($Catalogs -ne $null)
{
	foreach($Catalog in $Catalogs)
	{
		$Description = "-"
		$PvsAddress = $Catalog.PvsAddress 
		if ($PvsAddress -eq $null -or $PvsAddress.length -eq 0)
		{
			$PvsAddress = "-"
		}
		$PvsDomain = $Catalog.PvsDomain
		if ($PvsDomain -eq $null -or $PvsDomain.length -eq 0)
		{
			$PvsDomain = "-"
		}
		$Description = $Catalog.Description
		if ($Description -eq $null -or $Description.trim().length -eq 0)
		{
			$Description = "-"
		}
		$Scopes = $Catalog.Scopes
		if ($Scopes -eq $null -or $Scopes.length -eq 0)
		{
			$Scopes = "All"
		}
		$ZoneName = $Catalog.ZoneName
		if ($ZoneName -eq $null -or $ZoneName.length -eq 0)
		{
			$ZoneName = "-"
		}
		Write-Host ($Catalog.Name,$Catalog.AllocationType,$Catalog.AssignedCount,$Catalog.AvailableAssignedCount,$Catalog.AvailableCount,
		$Catalog.AvailableUnassignedCount,$Catalog.UnassignedCount,$Catalog.UsedCount,$Catalog.MachinesArePhysical,$PvsAddress,
		$PvsDomain,$Catalog.PersistUserChanges,$Catalog.ProvisioningType,$Catalog.SessionSupport,$Description,$Scopes,$ZoneName) -Separator ";"
	}
}

#BrowserName and Name Map Hashtbale
$AppNameHash = @{}
$AppAdminNameHash = @{}
#$PublishedAppInstances = Get-BrokerApplication -MaxRecordCount 2147483647
$PublishedAppInstances = Get-BrokerApplicationsPaged
if ($PublishedAppInstances -ne $null)
{
	foreach($PublishedAppInstance in $PublishedAppInstances)
	{
		$BrowserName = $PublishedAppInstance.BrowserName
		if ($BrowserName -ne $null -and $BrowserName.length -gt 0)
		{
			$Name = $PublishedAppInstance.Name
			if ($Name -eq $null -or $Name.length -eq 0)
			{
				$Name = $BrowserName
			}
			$AppAdminNameHash.Add($Name,$PublishedAppInstance)
			$Name = $Name + "(" + $BrowserName + ")"
			$AppNameHash.Add($BrowserName,$Name)
		}
	}
}

Write-Host "Broker Machines"
Write-Host "---------------"

#$BrokerMachines = Get-BrokerMachine -MaxRecordCount 2147483647
$DesktopGroups = Get-BrokerDesktopGroup -MaxRecordCount 999
if ($DesktopGroups  -ne $null)
{
	foreach ($DesktopGroup in $DesktopGroups)
	{
		$BrokerMachines = Get-BrokerMachinesPaged -DesktopGroupUid $DesktopGroup.Uid
		if($BrokerMachines -ne $null)
		{
			foreach($BrokerMachine in $BrokerMachines)
			{
				$HostedMachineName = $BrokerMachine.HostedMachineName
				if ($HostedMachineName -eq $null -or $HostedMachineName.length -eq 0)
				{
					$HostedMachineName = "-"
				}
				$HypervisorConnectionName = $BrokerMachine.HypervisorConnectionName
				if ($HypervisorConnectionName -eq $null -or $HypervisorConnectionName.length -eq 0)
				{
					$HypervisorConnectionName = "-"
				}
				$PublishedApplications = $BrokerMachine.PublishedApplications
				if ($PublishedApplications -eq $null -or $PublishedApplications.length -eq 0)
				{
					$PublishedApplications = "-"
				}
				else
				{
					$PublishApps = ""
					$count = 0
					foreach ($key in $PublishedApplications)
					{
						if ($count -eq 0)
						{
							$PublishApps = $AppNameHash.Get_Item($key)
						}
						else
						{
							$PublishApps = $PublishApps + "#" + $AppNameHash.Get_Item($key)
						}
						$count = $count + 1
					}
					if ($PublishApps -eq $null -or $PublishApps.length -eq 0)
					{
						$PublishedApplications = "-"
					}
					else
					{
						$PublishedApplications = $PublishApps
					}
				}
				$SessionUserName = $BrokerMachine.SessionUserName
				if ($SessionUserName -eq $null -or $SessionUserName.length -eq 0)
				{
					$SessionUserName = "-"
				}
				$IPAddress = $BrokerMachine.IPAddress
				if ($IPAddress -eq $null -or $IPAddress.length -eq 0)
				{
					$IPAddress = "-"
				}
				$OSType = $BrokerMachine.OSType
				if ($OSType -eq $null -or $OSType.length -eq 0)
				{
					$OSType = "-"
				}
				$OSVersion = $BrokerMachine.OSVersion
				if ($OSVersion -eq $null -or $OSVersion.length -eq 0)
				{
					$OSVersion = "-"
				}
				$ControllerDNSName = $BrokerMachine.ControllerDNSName
				if ($ControllerDNSName -eq $null -or $ControllerDNSName.length -eq 0)
				{
					$ControllerDNSName = "-"
				}
				$HostingServerName = $BrokerMachine.HostingServerName
				if ($HostingServerName -eq $null -or $HostingServerName.length -eq 0)
				{
					$HostingServerName = "-"
				}
				$AgentVersion = $BrokerMachine.AgentVersion
				if ($AgentVersion -eq $null -or $AgentVersion.length -eq 0)
				{
					$AgentVersion = "-"
				}
				$LastConnectionFailure = $BrokerMachine.LastConnectionFailure
				if ($LastConnectionFailure -eq $null)
				{
					$LastConnectionFailure = "-"
				}
				$LastConnectionTime = $BrokerMachine.LastConnectionTime
				if ($LastConnectionTime -eq $null)
				{
					$LastConnectionTime = "-"
				}
				$LastDeregistrationReason = $BrokerMachine.LastDeregistrationReason
				if ($LastDeregistrationReason -eq $null -or $LastDeregistrationReason.length -eq 0)
				{
					$LastDeregistrationReason = "-"
				}
				$LastDeregistrationTime = $BrokerMachine.LastDeregistrationTime
				if ($LastDeregistrationTime -eq $null)
				{
					$LastDeregistrationTime = "-"
				}
				$LastErrorReason = $BrokerMachine.LastErrorReason
				if ($LastErrorReason -eq $null -or $LastErrorReason.length -eq 0)
				{
					$LastErrorReason = "-"
				}
				$LastErrorTime = $BrokerMachine.LastErrorTime
				if ($LastErrorTime -eq $null)
				{
					$LastErrorTime = "-"
				}
				$ImageOutOfDate = $BrokerMachine.ImageOutOfDate
				if ($ImageOutOfDate -eq $null)
				{
					$ImageOutOfDate = "-"
				}
				$PublishedName = $BrokerMachine.PublishedName
				if ($PublishedName -eq $null -or $PublishedName.length -eq 0)
				{
					$PublishedName = "-"
				}
				$DesktopConditions = $BrokerMachine.DesktopConditions
				if ($DesktopConditions -eq $null -or $DesktopConditions.length -eq 0)
				{
					$DesktopConditions = "-"
				}
				$FaultState = $BrokerMachine.FaultState
				if ($FaultState -eq $null -or $FaultState.length -eq 0)
				{
					$FaultState = "-"
				}
				$LoadIndex = $BrokerMachine.LoadIndex
				if ($LoadIndex -eq $null -or $LoadIndex.length -eq 0)
				{
					$LoadIndex = "-"
				}
				$LoadIndexes = $BrokerMachine.LoadIndexes
				if ($LoadIndexes -eq $null -or $LoadIndexes.length -eq 0)
				{
					$LoadIndexes = "-"
				}
				else
				{
					$LoadIdxes = ""
					$count = 0
					foreach ($key in $LoadIndexes)
					{
						if ($count -eq 0)
						{
							$LoadIdxes = $key
						}
						else
						{
							$LoadIdxes = $LoadIdxes + "#" + $key
						}
						$count = $count + 1
					}
					if ($LoadIdxes -eq $null -or $LoadIdxes.length -eq 0)
					{
						$LoadIndexes = "-"
					}
					else
					{
						$LoadIndexes = $LoadIdxes
					}
				}

				$Tags = $BrokerMachine.Tags
				$BrokerMachineTags = "-"
				if ($Tags -ne $null -and $Tags.length -gt 0)
				{
					$TagCount = 0
					foreach ($Tag in $Tags)
					{
						if ($TagCount -eq 0)
						{
							$BrokerMachineTags = $Tag
						}
						else
						{
							$BrokerMachineTags = $BrokerMachineTags + "#" + $Tag
						}
						$TagCount = $TagCount + 1
					}
				}

				Write-Host($BrokerMachine.CatalogName,$BrokerMachine.DesktopGroupName,$BrokerMachine.AllocationType,$BrokerMachine.MachineName,$BrokerMachine.PowerState,
				$BrokerMachine.RegistrationState,$BrokerMachine.SummaryState,$BrokerMachine.DNSName,$ControllerDNSName,$HostedMachineName,$HypervisorConnectionName,
				$HostingServerName,$BrokerMachine.InMaintenanceMode,$BrokerMachine.IsAssigned,$BrokerMachine.IsPhysical,
				$BrokerMachine.MachineInternalState,$OSType,$OSVersion,$BrokerMachine.PersistUserChanges,$BrokerMachine.ProvisioningType,
				$PublishedApplications,$BrokerMachine.SessionCount,$BrokerMachine.SessionsEstablished,$BrokerMachine.SessionsPending,
				$BrokerMachine.SessionSupport,$SessionUserName,$IPAddress,$AgentVersion,$LastConnectionFailure,$LastConnectionTime,
				$LastDeregistrationReason,$LastDeregistrationTime,$LastErrorReason,$LastErrorTime,$ImageOutOfDate,$PublishedName,
				$BrokerMachine.PowerActionPending,$BrokerMachine.WillShutdownAfterUse,$FaultState,$LoadIndex,$LoadIndexes,$BrokerMachine.VMToolsState,$BrokerMachine.WindowsConnectionSetting,$BrokerMachineTags,$DesktopConditions) -Separator ";"
			}
		}
	}
}


Write-Host "Delivery Groups"
Write-Host "---------------"
#$DesktopGroups = Get-BrokerDesktopGroup -MaxRecordCount 2147483647
if($DesktopGroups -ne $null)
{
	foreach($DesktopGroup in $DesktopGroups)
	{
		$Description = $DesktopGroup.Description
		if ($Description -eq $null -or $Description.trim().length -eq 0)
		{
			$Description = "-"
		}
		$PublishedName = $DesktopGroup.PublishedName
		if ($PublishedName -eq $null -or $PublishedName.length -eq 0)
		{
			$PublishedName = "-"
		}

		$Tags = $DesktopGroup.Tags
		$DesktopGroupTags = "-"
		if ($Tags -ne $null -and $Tags.length -gt 0)
		{
			$TagCount = 0
			foreach ($Tag in $Tags)
			{
				if ($TagCount -eq 0)
				{
					$DesktopGroupTags = $Tag
				}
				else
				{
					$DesktopGroupTags = $DesktopGroupTags + "#" + $Tag
				}
				$TagCount = $TagCount + 1
			}
		}

		Write-Host ($DesktopGroup.Name,$Description,$DesktopGroup.Enabled,$DesktopGroup.InMaintenanceMode,$DesktopGroup.TotalDesktops,
		$DesktopGroup.DesktopsAvailable,$DesktopGroup.DesktopsDisconnected,$DesktopGroup.DesktopsInUse,
		$DesktopGroup.DesktopsNeverRegistered,$DesktopGroup.DesktopsUnregistered,$DesktopGroup.DesktopsPreparing,
		$DesktopGroup.DesktopKind,$DesktopGroup.SecureIcaRequired,$DesktopGroup.SessionSupport,
		$DesktopGroup.ShutdownDesktopsAfterUse,$PublishedName,$DesktopGroup.AutomaticPowerOnForAssigned,
		$DesktopGroup.AutomaticPowerOnForAssignedDuringPeak,$DesktopGroup.DeliveryType,
		$DesktopGroup.OffPeakBufferSizePercent,$DesktopGroup.PeakBufferSizePercent,
		$DesktopGroup.Sessions,$DesktopGroup.TotalApplications,$DesktopGroup.TurnOnAddedMachine,$DesktopGroupTags) -Separator ";"
	}
}

Write-Host "Sessions"
Write-Host "--------"
# SessionState
# Valid values are Other, PreparingSession, Connected, Active, Disconnected, Reconnecting, NonBrokeredSession, and Unknown.

#$BrokerSessions = Get-BrokerSession -MaxRecordCount 2147483647
$BrokerSessions = Get-BrokerSessionsPaged
if ($BrokerSessions -ne $null)
{
	foreach($BrokerSession in $BrokerSessions)
	{
		$IPAddress = $BrokerSession.IPAddress
		if ($IPAddress -eq $null -or $IPAddress.length -eq 0)
		{
			$IPAddress = "-"
		}
		$OSType = $BrokerSession.OSType
		if ($OSType -eq $null -or $OSType.length -eq 0)
		{
			$OSType = "-"
		}
		$HostedMachineName = $BrokerSession.HostedMachineName
		if ($HostedMachineName -eq $null -or $HostedMachineName.length -eq 0)
		{
			$HostedMachineName = "-"
		}
		$HostingServerName = $BrokerSession.HostingServerName
		if ($HostingServerName -eq $null -or $HostingServerName.length -eq 0)
		{
			$HostingServerName = "-"
		}
		$HypervisorConnectionName = $BrokerSession.HypervisorConnectionName
		if ($HypervisorConnectionName -eq $null -or $HypervisorConnectionName.length -eq 0)
		{
			$HypervisorConnectionName = "-"
		}
		$UserName = $BrokerSession.UserName
		if ($UserName -eq $null -or $UserName.length -eq 0)
		{
			$UserName = $BrokerSession.$BrokeringUserName
			if ($UserName -eq $null -or $UserName.length -eq 0)
			{
				$UserName = "-"
			}
		}
		$ClientAddress = $BrokerSession.ClientAddress
		if ($ClientAddress -eq $null -or $ClientAddress.length -eq 0)
		{
			$ClientAddress = "-"
		}
		$ClientName = $BrokerSession.ClientName
		if ($ClientName -eq $null -or $ClientName.length -eq 0)
		{
			$ClientName = "-"
		}
		$ClientVersion = $BrokerSession.ClientVersion
		if ($ClientVersion -eq $null -or $ClientVersion.length -eq 0)
		{
			$ClientVersion = "-"
		}
		$ClientPlatform = $BrokerSession.ClientPlatform
		if ($ClientPlatform -eq $null -or $ClientPlatform.length -eq 0)
		{
			$ClientPlatform = "-"
		}
		$SessionType = $BrokerSession.SessionType
		if ($SessionType -eq $null -or $SessionType.length -eq 0)
		{
			$SessionType = "-"
		}
		$BrokeringTime = $BrokerSession.BrokeringTime
		if ($BrokeringTime -eq $null)
		{
			$BrokeringTime = "-"
		}
		$ConnectedViaHostName = $BrokerSession.ConnectedViaHostName
		if ($ConnectedViaHostName -eq $null -or $ConnectedViaHostName.length -eq 0)
		{
			$ConnectedViaHostName = "-"
		}
		$ConnectedViaIP = $BrokerSession.ConnectedViaIP
		if ($ConnectedViaIP -eq $null -or $ConnectedViaIP.length -eq 0)
		{
			$ConnectedViaIP = "-"
		}
		$LaunchedViaHostName = $BrokerSession.LaunchedViaHostName
		if ($LaunchedViaHostName -eq $null -or $LaunchedViaHostName.length -eq 0)
		{
			$LaunchedViaHostName = "-"
		}
		$LaunchedViaIP = $BrokerSession.LaunchedViaIP
		if ($LaunchedViaIP -eq $null -or $LaunchedViaIP.length -eq 0)
		{
			$LaunchedViaIP = "-"
		}
		$Protocol = $BrokerSession.Protocol
		if ($Protocol -eq $null -or $Protocol.length -eq 0)
		{
			$Protocol = "-"
		}
		$ControllerDNSName = $BrokerSession.ControllerDNSName
		if ($ControllerDNSName -eq $null -or $ControllerDNSName.length -eq 0)
		{
			$ControllerDNSName = "-"
		}
		$LicenseId = $BrokerSession.LicenseId
		if ($LicenseId -eq $null -or $LicenseId -eq 0)
		{
			$LicenseId = "-"
		}
		$BrokeringDuration = $BrokerSession.BrokeringDuration
		if ($BrokeringDuration -eq $null)
		{
			$BrokeringDuration = "-"
		}
		$EstablishmentDuration = $BrokerSession.EstablishmentDuration
		if ($EstablishmentDuration -eq $null)
		{
			$EstablishmentDuration = "-"
		}
		$EstablishmentTime = $BrokerSession.EstablishmentTime
		if ($EstablishmentTime -eq $null)
		{
			$EstablishmentTime = "-"
		}
		$SessionStateChangeTime = $BrokerSession.SessionStateChangeTime
		if ($SessionStateChangeTime -eq $null)
		{
			$SessionStateChangeTime = "-"
		}
		$StartTime = $BrokerSession.StartTime
		if ($StartTime -eq $null)
		{
			$StartTime = "-"
		}
		$DeviceId = $BrokerSession.DeviceId
		if ($DeviceId -eq $null)
		{
			$DeviceId = "-"
		}
		$HardwareId = $BrokerSession.HardwareId
		if ($HardwareId -eq $null)
		{
			$HardwareId = "-"
		}
		$AgentVersion = $BrokerSession.AgentVersion
		if ($AgentVersion -eq $null -or $AgentVersion.length -eq 0)
		{
			$AgentVersion = "-"
		}

		$ApplicationsInUse = $BrokerSession.ApplicationsInUse
		if ($ApplicationsInUse -eq $null -or $ApplicationsInUse.length -eq 0)
		{
			$ApplicationsInUse = "-"
		}
		else
		{
			$AppsInUse = ""
			$count = 0
			foreach ($key in $ApplicationsInUse)
			{
				if ($count -eq 0)
				{
					$AppsInUse = $key
				}
				else
				{
					$AppsInUse = $AppsInUse + "#" + $key
				}
				$count = $count + 1
			}
			if ($AppsInUse -eq $null -or $AppsInUse.length -eq 0)
			{
				$ApplicationsInUse = "-"
			}
			else
			{
				$ApplicationsInUse = $AppsInUse
			}
		}
		$AppState = $BrokerSession.AppState
		if ($AppState -eq $null -or $AppState.length -eq 0)
		{
			$AppState = "-"
		}
		$LogoffInProgress = $BrokerSession.LogoffInProgress
		if ($LogoffInProgress -eq $null -or $LogoffInProgress.length -eq 0)
		{
			$LogoffInProgress = "-"
		}
		$LogonInProgress = $BrokerSession.LogonInProgress
		if ($LogonInProgress -eq $null -or $LogonInProgress.length -eq 0)
		{
			$LogonInProgress = "-"
		}
		$SessionKey = $BrokerSession.SessionKey
		if ($SessionKey -eq $null -or $SessionKey.length -eq 0)
		{
			$SessionKey = "-"
		}
		$IsAnonymousUser = $Broker.IsAnonymousUser
		if ($IsAnonymousUser -eq $null -or $IsAnonymousUser.length -eq 0)
		{
			$IsAnonymousUser = "-"
		}

		Write-Host ($BrokerSession.Uid,$BrokerSession.MachineName,$BrokerSession.DNSName,$IPAddress,$OSType,
		$BrokerSession.CatalogName,$BrokerSession.DesktopGroupName,$HostedMachineName,$HostingServerName,
		$HypervisorConnectionName,$UserName,$ClientAddress,$ClientName,$ClientVersion,$ConnectedViaHostName,$ConnectedViaIP,
		$LaunchedViaHostName,$LaunchedViaIP,$Protocol,$BrokerSession.SessionState,$BrokeringDuration,$EstablishmentDuration,$EstablishmentTime,
		$SessionStateChangeTime,$StartTime,$BrokeringTime,$LicenseId,$BrokerSession.Hidden,$BrokerSession.AutonomouslyBrokered,
		$DeviceId,$HardwareId,$AgentVersion,$BrokerSession.SessionSupport,$ControllerDNSName,$ApplicationsInUse,$AppState,
		$LogoffInProgress,$LogonInProgress,$SessionKey,$ClientPlatform,$SessionType,$IsAnonymousUser) -Separator ";"
	}
}

Write-Host "Application Instances"
Write-Host "---------------------"
if ($BrokerSessions -ne $null)
{
	foreach($Session in $BrokerSessions)
	{
		if($Session)
		{
			$ControllerDNSName = $Session.ControllerDNSName
			$UserName = $Session.UserName
			if ($UserName -eq $null -or $UserName.length -eq 0)
			{
				$UserName = "-"
			}
			$ClientName = $Session.ClientName
			if ($ClientName -eq $null -or $ClientName.length -eq 0)
			{
				$ClientName = "-"
			}
			$ClientAddress = $Session.ClientAddress
			if ($ClientAddress -eq $null -or $ClientAddress -eq 0)
			{
				$ClientAddress = "-"
			}
			$StartTime = $Session.StartTime
			if ($StartTime -eq $null)
			{
				$StartTime = "-"
			}
			$BrokeringTime = $Session.BrokeringTime
			if ($BrokeringTime -eq $null)
			{
				$BrokeringTime = "-"
			}
			$HostedMachineName = $Session.HostedMachineName
			if ($HostedMachineName -eq $null -or $HostedMachineName.length -eq 0)
			{
				$HostedMachineName = "-"
			}
			$MachineName = $Session.MachineName
			if ($MachineName -eq $null -or $MachineName.length -eq 0)
			{
				$MachineName = "-"
			}
			$DesktopGroupName = $Session.DesktopGroupName
			if ($DesktopGroupName -eq $null -or $DesktopGroupName.length -eq 0)
			{
				$DesktopGroupName = "-"
			}
			$CatalogName = $Session.CatalogName
			if ($CatalogName -eq $null -or $CatalogName.length -eq 0)
			{
				$CatalogName = "-"
			}
			$SessionState = $Session.SessionState
			if ($SessionState -eq $null -or $SessionState.length -eq 0)
			{
				$SessionState = "-"
			}
			$SessionSupport = $Session.SessionSupport
			if ($SessionSupport -eq $null -or $SessionSupport.length -eq 0)
			{
				$SessionSupport = "-"
			}
			$Protocol = $Session.Protocol
			if ($Protocol -eq $null -or $Protocol.length -eq 0)
			{
				$Protocol = "-"
			}

			#$AppInstances = Get-BrokerApplicationInstance -SessionUid $Session.Uid -EA 0
			$AppInstances = $Session.ApplicationsInUse
			if ($AppInstances -ne $null)
			{	
				foreach($AppInstance in $AppInstances)
				{
					#$Application = Get-BrokerApplication -Uid $AppInstance.ApplicationUid
					$Application = $AppAdminNameHash.Get_Item($AppInstance)
					if ($Application -ne $null)
					{
						$Tags = $Application.Tags
						$ApplicationTags = "-"
						if ($Tags -ne $null -and $Tags.length -gt 0)
						{
							$TagCount = 0
							foreach ($Tag in $Tags)
							{
								if ($TagCount -eq 0)
								{
									$ApplicationTags = $Tag
								}
								else
								{
									$ApplicationTags = $ApplicationTags + "#" + $Tag
								}
								$TagCount = $TagCount + 1
							}
						}
						#Write-Host($ControllerDNSName,$Application.Uid,$Application.Name,$AppInstance.Instances,$MachineName,$HostedMachineName,
						#	   $DesktopGroupName,$CatalogName,$AppInstance.SessionUid,$UserName,$ClientName,$ClientAddress,
						#	   $Protocol,$SessionState,$StartTime,$BrokeringTime,$SessionSupport,$ApplicationTags) -Separator ";"

						$Instances = 1
						Write-Host($ControllerDNSName,$Application.Uid,$Application.Name,$Instances,$MachineName,$HostedMachineName,
							   $DesktopGroupName,$CatalogName,$Session.Uid,$UserName,$ClientName,$ClientAddress,
							   $Protocol,$SessionState,$StartTime,$BrokeringTime,$SessionSupport,$ApplicationTags) -Separator ";"
					}
				}
			}
		}
	}
}

Write-Host "Published Applications"
Write-Host "----------------------"
#$PublishedAppInstances = Get-BrokerApplication -MaxRecordCount 2147483647
if ($PublishedAppInstances -ne $null)
{
	foreach($PublishedAppInstance in $PublishedAppInstances)
	{
		$Description = $PublishedAppInstance.Description
		if ($Description -eq $null -or $Description.trim().length -eq 0)
		{
			$Description = "-"
		}
		$DisplayName = "-"
		$WorkingDirectory = $PublishedAppInstance.WorkingDirectory
		if ($WorkingDirectory -eq $null -or $WorkingDirectory.length -eq 0)
		{
			$WorkingDirectory = "-"
		}

		$CommandLineExecutable = $PublishedAppInstance.CommandLineExecutable
		$CommandLineArgs = $PublishedAppInstance.CommandLineArguments
		$AppVCommandLineArgs = "-"
		if ($CommandLineExecutable -ne $null -and ($CommandLineExecutable -ieq "CtxAppVLauncher.exe"))
		{
			$MetadataMap = $PublishedAppInstance.MetadataMap
			if ($MetadataMap -ne $null)
			{
				$AppID = $MetadataMap.get_Item("Citrix_DesktopStudio_AppID")
				$PkgID = $MetadataMap.get_Item("Citrix_DesktopStudio_PackageId")
				if ($AppID -ne $null -and $AppID.length -gt 0 -and $PkgID -ne $null -and $PkgID.length -gt 0)
				{
					$Idx = $PkgID.LastIndexOf("_")
					if ($Idx -gt 0)
					{
						$PkgID = $PkgID.Substring(0,$Idx)
					}
					$AppVCommandLineArgs = "CtxAppVLauncher.exe" + "#" + $AppID + "#" + $PkgID
				}
			}
		}
		if ($CommandLineArgs)
		{
			$CommandLineExecutable = $CommandLineExecutable + " " + $CommandLineArgs
		}
		$Tags = $PublishedAppInstance.Tags
		$PublishedAppInstanceTags = "-"
		if ($Tags -ne $null -and $Tags.length -gt 0)
		{
			$TagCount = 0
			foreach ($Tag in $Tags)
			{
				if ($TagCount -eq 0)
				{
					$PublishedAppInstanceTags = $Tag
				}
				else
				{
					$PublishedAppInstanceTags = $PublishedAppInstanceTags + "#" + $Tag
				}
				$TagCount = $TagCount + 1
			}
		}

		$NameAndBrowserName = $PublishedAppInstance.Name + "(" + $PublishedAppInstance.BrowserName + ")"
		Write-Host($PublishedAppInstance.Uid,$NameAndBrowserName,$DisplayName,$Description,$PublishedAppInstance.ApplicationType,
		$PublishedAppInstance.CpuPriorityLevel,$PublishedAppInstance.Enabled,$PublishedAppInstance.Visible,
		$CommandLineExecutable,$WorkingDirectory,$AppVCommandLineArgs,$PublishedAppInstanceTags) -Separator ";"
	}
}

Write-Host "Zones"
Write-Host "------"
foreach($Zone in $ConfiguredZones)
{
	Write-Host($Zone.Name,$Zone.IsPrimary) -Separator ";"
}


Write-Host "Broker Tag Usage"
Write-Host "----------------"
$BrokerTagUsage = Get-BrokerTagUsage
if ($BrokerTagUsage -ne $null)
{
	foreach ($TagUsage in $BrokerTagUsage)
	{
		Write-Host ($TagUsage.TagUUID,$TagUsage.TagName,$TagUsage.TaggedDesktopGroups,$TagUsage.TaggedMachines,$TagUsage.TaggedApplicationGroups,
		$TagUsage.TaggedApplications,$TagUsage.TotalTaggedObjects,$TagUsage.TotalTagRestrictedObjects,$TagUsage.TagRestrictedApplicationGroups,
		$TagUsage.TagRestrictedEntitlementPolicyRules,$TagUsage.TagRestrictedRebootSchedules,$TagUsage.UnknownTaggedObjects,$TagUsage.UnknownTagRestrictedObjects) -Separator ";"
	}
}

Write-Host "Date Time Format"
Write-Host "----------------"
$ShortDatePattern = (Get-Culture).datetimeformat.ShortDatePattern
$LongTimePattern = (Get-Culture).datetimeformat.LongTimePattern
Write-Host "$ShortDatePattern $LongTimePattern"

Remove-PSSnapin citrix*



# SIG # Begin signature block
# MIIldgYJKoZIhvcNAQcCoIIlZzCCJWMCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUgymp8myajfzrJaWq1tYODS6C
# 64iggh6cMIIFMDCCBBigAwIBAgIQBAkYG1/Vu2Z1U0O1b5VQCDANBgkqhkiG9w0B
# AQsFADBlMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVk
# IElEIFJvb3QgQ0EwHhcNMTMxMDIyMTIwMDAwWhcNMjgxMDIyMTIwMDAwWjByMQsw
# CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cu
# ZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQg
# Q29kZSBTaWduaW5nIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA
# +NOzHH8OEa9ndwfTCzFJGc/Q+0WZsTrbRPV/5aid2zLXcep2nQUut4/6kkPApfmJ
# 1DcZ17aq8JyGpdglrA55KDp+6dFn08b7KSfH03sjlOSRI5aQd4L5oYQjZhJUM1B0
# sSgmuyRpwsJS8hRniolF1C2ho+mILCCVrhxKhwjfDPXiTWAYvqrEsq5wMWYzcT6s
# cKKrzn/pfMuSoeU7MRzP6vIK5Fe7SrXpdOYr/mzLfnQ5Ng2Q7+S1TqSp6moKq4Tz
# rGdOtcT3jNEgJSPrCGQ+UpbB8g8S9MWOD8Gi6CxR93O8vYWxYoNzQYIH5DiLanMg
# 0A9kczyen6Yzqf0Z3yWT0QIDAQABo4IBzTCCAckwEgYDVR0TAQH/BAgwBgEB/wIB
# ADAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMweQYIKwYBBQUH
# AQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQwYI
# KwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFz
# c3VyZWRJRFJvb3RDQS5jcnQwgYEGA1UdHwR6MHgwOqA4oDaGNGh0dHA6Ly9jcmw0
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwOqA4oDaG
# NGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RD
# QS5jcmwwTwYDVR0gBEgwRjA4BgpghkgBhv1sAAIEMCowKAYIKwYBBQUHAgEWHGh0
# dHBzOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwCgYIYIZIAYb9bAMwHQYDVR0OBBYE
# FFrEuXsqCqOl6nEDwGD5LfZldQ5YMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6en
# IZ3zbcgPMA0GCSqGSIb3DQEBCwUAA4IBAQA+7A1aJLPzItEVyCx8JSl2qB1dHC06
# GsTvMGHXfgtg/cM9D8Svi/3vKt8gVTew4fbRknUPUbRupY5a4l4kgU4QpO4/cY5j
# DhNLrddfRHnzNhQGivecRk5c/5CxGwcOkRX7uq+1UcKNJK4kxscnKqEpKBo6cSgC
# PC6Ro8AlEeKcFEehemhor5unXCBc2XGxDI+7qPjFEmifz0DLQESlE/DmZAwlCEIy
# sjaKJAL+L3J+HNdJRZboWR3p+nRka7LrZkPas7CM1ekN3fYBIM6ZMWM9CBoYs4Gb
# T8aTEAb8B4H6i9r5gkn3Ym6hU/oSlBiFLpKR6mhsRDKyZqHnGKSaZFHvMIIFsTCC
# BJmgAwIBAgIQASQK+x44C4oW8UtxnfTTwDANBgkqhkiG9w0BAQwFADBlMQswCQYD
# VQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGln
# aWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0Ew
# HhcNMjIwNjA5MDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEV
# MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29t
# MSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3
# DQEBAQUAA4ICDwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZ
# wuEppz1Yq3aaza57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4V
# pX6+n6lXFllVcq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAd
# YyktzuxeTsiT+CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3
# T6cw2Vbuyntd463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjU
# N6QuBX2I9YI+EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNda
# SaTC5qmgZ92kJ7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtm
# mnTK3kse5w5jrubU75KSOp493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyV
# w4/3IbKyEbe7f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3
# AeEPlAwhHbJUKSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYi
# Cd98THU/Y+whX8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmp
# sh3lGwIDAQABo4IBXjCCAVowDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7Nfj
# gtJxXWRM3y5nP+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNt
# yA8wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHkGCCsGAQUF
# BwEBBG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMG
# CCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRB
# c3N1cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDwwOqA4oDaGNGh0dHA6Ly9jcmwz
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwIAYDVR0g
# BBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBDAUAA4IBAQCa
# FgKlAe+B+w20WLJ4ragjGdlzN9pgnlHXy/gvQLmjH3xATjM+kDzniQF1hehiex1W
# 4HG63l7GN7x5XGIATfhJelFNBjLzxdIAKicg6okuFTngLD74dXwsgkFhNQ8j0O01
# ldKIlSlDy+CmWBB8U46fRckgNxTA7Rm6fnc50lSWx6YR3zQz9nVSQkscnY2W1ZVs
# RxIUJF8mQfoaRr3esOWRRwOsGAjLy9tmiX8rnGW/vjdOvi3znUrDzMxHXsiVla3R
# y7sqBiD5P3LqNutFcpJ6KXsUAzz7TdZIcXoQEYoIdM1sGwRc0oqVA3ZRUFPWLvdK
# RsOuECxxTLCHtic3RGBEMIIGMzCCBRugAwIBAgIQCJp0nrgtw+wn6mXq2/g1MTAN
# BgkqhkiG9w0BAQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQg
# SW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2Vy
# dCBTSEEyIEFzc3VyZWQgSUQgQ29kZSBTaWduaW5nIENBMB4XDTIxMDUyNTAwMDAw
# MFoXDTI0MDUyOTIzNTk1OVowcTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5ldyBK
# ZXJzZXkxDzANBgNVBAcTBklzZWxpbjEdMBsGA1UEChMUZUcgSW5ub3ZhdGlvbnMs
# IEluYy4xHTAbBgNVBAMTFGVHIElubm92YXRpb25zLCBJbmMuMIICIjANBgkqhkiG
# 9w0BAQEFAAOCAg8AMIICCgKCAgEAxGUJ0prjbfxcmgZGjd7JDxOs7sySHvaQ3wGR
# 2KbD3aZageSE+kG0tKkEsQ4na8bnNViY/zF4Pbo0ZkYtMzJIk0AZXDTMvXr/mEvm
# KxCbiJFTCpKkmZ4sa5BZfO7igIvcNSa0zII2a5jiQJFy85j+f9I4EgTo9OcdVeIN
# XhgQ2xZ6TjEK+pbhqmVXvr8DB26JclOaed0L5Vs7+CbGzZuKifgxL2i8d5FDzkhZ
# SZfdCtGtEvE5pLesXSWfpzUddvCNRisIPGB7mg+Rln2XvUZyyaRURtlx11pYfJ/K
# NYAkUCL23rnh38/maxtSvdeioYBj4xLyt7poQBG+b0uzCjgRADe3/k0NtiVwJIo0
# ZEsUyDANp7JIIjMePzsOMYzfMPyi8clAwfBYc2XhSuUcX3yKf4kpRFmClYfK5Lst
# GQRNbONjpsCBTEgZuwEfkdOU0rmaMywZFVLHEpmpYZePz7M3VhN+aV56kR/efo8e
# D81E5VDQqmJiGWKq/s9jhbkYH7g2uhL2MXb/uRshVDSOkFRvIID/l+yDJogDBpae
# 3x2ov5YvjY8Zo8RF0tXvyS0rfsDvANZw7mptzHn0Blmw5lug3F2CVJpNbzb1HmAH
# 6k0yFHmWYCoUANbBB0YR/k4JvgY2byttF7cQPJyl7UI9Cu4KTAg7ROmGkwdLhmXo
# PCBco0ECAwEAAaOCAcQwggHAMB8GA1UdIwQYMBaAFFrEuXsqCqOl6nEDwGD5LfZl
# dQ5YMB0GA1UdDgQWBBT5yESaz9gTSS8aBNa1r34MMA42SjAOBgNVHQ8BAf8EBAMC
# B4AwEwYDVR0lBAwwCgYIKwYBBQUHAwMwdwYDVR0fBHAwbjA1oDOgMYYvaHR0cDov
# L2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5jcmwwNaAzoDGG
# L2h0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFzc3VyZWQtY3MtZzEuY3Js
# MEsGA1UdIAREMEIwNgYJYIZIAYb9bAMBMCkwJwYIKwYBBQUHAgEWG2h0dHA6Ly93
# d3cuZGlnaWNlcnQuY29tL0NQUzAIBgZngQwBBAEwgYQGCCsGAQUFBwEBBHgwdjAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tME4GCCsGAQUFBzAC
# hkJodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRTSEEyQXNzdXJl
# ZElEQ29kZVNpZ25pbmdDQS5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsF
# AAOCAQEAlAjncFPvLKywT/4DECG7bHbtiVl+uumfRj2YrFuChsgkv1PmQpgfypxC
# 3g+ErV9yRV8+XXSCwOaKJ3v6RLnfphLhEwYc2+0Qs/Nlib5NAxDGuIczAIeXOc5k
# RRpvFsQ2XSNtM7XL1tLDm6p/VG7BoUAyqRXsMPdWbTkN/9ndCmGSsqcxjG2ud8O6
# Vhte9J5LaHBVk3lIZAMtH6ACdo5QTrM49nbIU8QGuRYNXZKRLAUu6IgD6WJKMVfZ
# XWlyfD8dZ2r3ej6Q1uAO/Nbtd397T+BVQrDWMOG8+GeRiJwoevxbIWh0SenZOUrA
# q9vTJaSvFMSvctkJm/oxLUcUdEGS3zCCBq4wggSWoAMCAQICEAc2N7ckVHzYR6z9
# KGYqXlswDQYJKoZIhvcNAQELBQAwYjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERp
# Z2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMY
# RGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MB4XDTIyMDMyMzAwMDAwMFoXDTM3MDMy
# MjIzNTk1OVowYzELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2IFRp
# bWVTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAMaG
# NQZJs8E9cklRVcclA8TykTepl1Gh1tKD0Z5Mom2gsMyD+Vr2EaFEFUJfpIjzaPp9
# 85yJC3+dH54PMx9QEwsmc5Zt+FeoAn39Q7SE2hHxc7Gz7iuAhIoiGN/r2j3EF3+r
# GSs+QtxnjupRPfDWVtTnKC3r07G1decfBmWNlCnT2exp39mQh0YAe9tEQYncfGpX
# evA3eZ9drMvohGS0UvJ2R/dhgxndX7RUCyFobjchu0CsX7LeSn3O9TkSZ+8OpWNs
# 5KbFHc02DVzV5huowWR0QKfAcsW6Th+xtVhNef7Xj3OTrCw54qVI1vCwMROpVymW
# Jy71h6aPTnYVVSZwmCZ/oBpHIEPjQ2OAe3VuJyWQmDo4EbP29p7mO1vsgd4iFNmC
# KseSv6De4z6ic/rnH1pslPJSlRErWHRAKKtzQ87fSqEcazjFKfPKqpZzQmiftkaz
# nTqj1QPgv/CiPMpC3BhIfxQ0z9JMq++bPf4OuGQq+nUoJEHtQr8FnGZJUlD0UfM2
# SU2LINIsVzV5K6jzRWC8I41Y99xh3pP+OcD5sjClTNfpmEpYPtMDiP6zj9NeS3YS
# UZPJjAw7W4oiqMEmCPkUEBIDfV8ju2TjY+Cm4T72wnSyPx4JduyrXUZ14mCjWAkB
# KAAOhFTuzuldyF4wEr1GnrXTdrnSDmuZDNIztM2xAgMBAAGjggFdMIIBWTASBgNV
# HRMBAf8ECDAGAQH/AgEAMB0GA1UdDgQWBBS6FtltTYUvcyl2mi91jGogj57IbzAf
# BgNVHSMEGDAWgBTs1+OC0nFdZEzfLmc/57qYrhwPTzAOBgNVHQ8BAf8EBAMCAYYw
# EwYDVR0lBAwwCgYIKwYBBQUHAwgwdwYIKwYBBQUHAQEEazBpMCQGCCsGAQUFBzAB
# hhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQQYIKwYBBQUHMAKGNWh0dHA6Ly9j
# YWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRSb290RzQuY3J0MEMG
# A1UdHwQ8MDowOKA2oDSGMmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2Vy
# dFRydXN0ZWRSb290RzQuY3JsMCAGA1UdIAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG
# /WwHATANBgkqhkiG9w0BAQsFAAOCAgEAfVmOwJO2b5ipRCIBfmbW2CFC4bAYLhBN
# E88wU86/GPvHUF3iSyn7cIoNqilp/GnBzx0H6T5gyNgL5Vxb122H+oQgJTQxZ822
# EpZvxFBMYh0MCIKoFr2pVs8Vc40BIiXOlWk/R3f7cnQU1/+rT4osequFzUNf7WC2
# qk+RZp4snuCKrOX9jLxkJodskr2dfNBwCnzvqLx1T7pa96kQsl3p/yhUifDVinF2
# ZdrM8HKjI/rAJ4JErpknG6skHibBt94q6/aesXmZgaNWhqsKRcnfxI2g55j7+6ad
# cq/Ex8HBanHZxhOACcS2n82HhyS7T6NJuXdmkfFynOlLAlKnN36TU6w7HQhJD5TN
# OXrd/yVjmScsPT9rp/Fmw0HNT7ZAmyEhQNC3EyTN3B14OuSereU0cZLXJmvkOHOr
# pgFPvT87eK1MrfvElXvtCl8zOYdBeHo46Zzh3SP9HSjTx/no8Zhf+yvYfvJGnXUs
# HicsJttvFXseGYs2uJPU5vIXmVnKcPA3v5gA3yAWTyf7YGcWoWa63VXAOimGsJig
# K+2VQbc61RWYMbRiCQ8KvYHZE/6/pNHzV9m8BPqC3jLfBInwAM1dwvnQI38AC+R2
# AibZ8GV2QqYphwlHK+Z/GqSFD/yYlvZVVCsfgPrA8g4r5db7qS9EFUrnEw4d2zc4
# GqEr9u3WfPwwggbGMIIErqADAgECAhAKekqInsmZQpAGYzhNhpedMA0GCSqGSIb3
# DQEBCwUAMGMxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7
# MDkGA1UEAxMyRGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1l
# U3RhbXBpbmcgQ0EwHhcNMjIwMzI5MDAwMDAwWhcNMzMwMzE0MjM1OTU5WjBMMQsw
# CQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xJDAiBgNVBAMTG0Rp
# Z2lDZXJ0IFRpbWVzdGFtcCAyMDIyIC0gMjCCAiIwDQYJKoZIhvcNAQEBBQADggIP
# ADCCAgoCggIBALkqliOmXLxf1knwFYIY9DPuzFxs4+AlLtIx5DxArvurxON4XX5c
# Nur1JY1Do4HrOGP5PIhp3jzSMFENMQe6Rm7po0tI6IlBfw2y1vmE8Zg+C78KhBJx
# bKFiJgHTzsNs/aw7ftwqHKm9MMYW2Nq867Lxg9GfzQnFuUFqRUIjQVr4YNNlLD5+
# Xr2Wp/D8sfT0KM9CeR87x5MHaGjlRDRSXw9Q3tRZLER0wDJHGVvimC6P0Mo//8Zn
# zzyTlU6E6XYYmJkRFMUrDKAz200kheiClOEvA+5/hQLJhuHVGBS3BEXz4Di9or16
# cZjsFef9LuzSmwCKrB2NO4Bo/tBZmCbO4O2ufyguwp7gC0vICNEyu4P6IzzZ/9KM
# u/dDI9/nw1oFYn5wLOUrsj1j6siugSBrQ4nIfl+wGt0ZvZ90QQqvuY4J03ShL7BU
# dsGQT5TshmH/2xEvkgMwzjC3iw9dRLNDHSNQzZHXL537/M2xwafEDsTvQD4ZOgLU
# MalpoEn5deGb6GjkagyP6+SxIXuGZ1h+fx/oK+QUshbWgaHK2jCQa+5vdcCwNiay
# CDv/vb5/bBMY38ZtpHlJrYt/YYcFaPfUcONCleieu5tLsuK2QT3nr6caKMmtYbCg
# QRgZTu1Hm2GV7T4LYVrqPnqYklHNP8lE54CLKUJy93my3YTqJ+7+fXprAgMBAAGj
# ggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8E
# DDAKBggrBgEFBQcDCDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEw
# HwYDVR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYDVR0OBBYEFI1kt4kh
# /lZYRIRhp+pvHDaP3a8NMFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwzLmRp
# Z2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3Rh
# bXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEBBIGDMIGAMCQGCCsGAQUFBzABhhhodHRw
# Oi8vb2NzcC5kaWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0dHA6Ly9jYWNlcnRz
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1l
# U3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcNAQELBQADggIBAA0tI3Sm0fX46kuZPwHk
# 9gzkrxad2bOMl4IpnENvAS2rOLVwEb+EGYs/XeWGT76TOt4qOVo5TtiEWaW8G5iq
# 6Gzv0UhpGThbz4k5HXBw2U7fIyJs1d/2WcuhwupMdsqh3KErlribVakaa33R9QIJ
# T4LWpXOIxJiA3+5JlbezzMWn7g7h7x44ip/vEckxSli23zh8y/pc9+RTv24KfH7X
# 3pjVKWWJD6KcwGX0ASJlx+pedKZbNZJQfPQXpodkTz5GiRZjIGvL8nvQNeNKcEip
# tucdYL0EIhUlcAZyqUQ7aUcR0+7px6A+TxC5MDbk86ppCaiLfmSiZZQR+24y8fW7
# OK3NwJMR1TJ4Sks3KkzzXNy2hcC7cDBVeNaY/lRtf3GpSBp43UZ3Lht6wDOK+Eoo
# jBKoc88t+dMj8p4Z4A2UKKDr2xpRoJWCjihrpM6ddt6pc6pIallDrl/q+A8GQp3f
# BmiW/iqgdFtjZt5rLLh4qk1wbfAs8QcVfjW05rUMopml1xVrNQ6F1uAszOAMJLh8
# UgsemXzvyMjFjFhpr6s94c/MfRWuFL+Kcd/Kl7HYR+ocheBFThIcFClYzG/Tf8u+
# wQ5KbyCcrtlzMlkI5y2SoRoR/jKYpl0rl+CL05zMbbUNrkdjOEcXW28T2moQbh9J
# t0RbtAgKh1pZBHYRoad3AhMcMYIGRDCCBkACAQEwgYYwcjELMAkGA1UEBhMCVVMx
# FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNv
# bTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmlu
# ZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYBBAGCNwIB
# DDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFC5pWF5kgrJCMNSmkbP3qVk+
# KBD3MA0GCSqGSIb3DQEBAQUABIICAIFvflAOQYXylkd/lwlISiQb1SMDPOfLJR2O
# jb+b/9SH49WdpZETCNZN3czD52HmFNbwTMkpJV24Jbf4B0yWFpLJTXiexG0TlcU7
# azGDoefD67NWcCGQD6rlenSLc5vY25UrA+fcO4Vz34JxVAWXWfJgTmOmnRRdYPvh
# c2qo/5/wTFxNvKuQM2jFJvAIw5EydGYA7RQCw021Vs+AsT0/yhBImLjE4a7DrhJM
# d3bAHwEowF26Z1FUTfOsbtiUV0dxgPs9U9o4mbtgMdHKMqSfgXtJwzsPMGNGKsRB
# NZ7U6xB2XjEZgAtYdSxtc89JNCZrrd/gNKm+TrQ12UaX5dv/IHZNvcP0Whr2Sdd0
# ha/GnYjhj4oYiqpi9N9BZhjfRmVbHpr8Yh67OZEaB39ONsYI2HmIp6bgkcdSJzsq
# g9/4WViJbnNpc1BGivMGjItiKcuySE3zYbw/aFtAlPip8kNpK6RrWvFRLiCDwOnm
# ehGJzM3LdnyngFBSJsrsKDi0jtts9FzJO7hUDNooMpRi2gKU0g7iV8hv0bT4gXLZ
# NJzXR0XthZPN/Yl8rA4J0tCeR/9AFapbhQSitNSMAiNgU7zNxKOurteSqGaZ48Rj
# bxG+f0r8srxSxiJPuYYykyQqSzhBiPDLfn5vhy+i8tauhuz/2jJh87T6xOA/aOPE
# j3I5NIOWoYIDIDCCAxwGCSqGSIb3DQEJBjGCAw0wggMJAgEBMHcwYzELMAkGA1UE
# BhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2Vy
# dCBUcnVzdGVkIEc0IFJTQTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQCnpK
# iJ7JmUKQBmM4TYaXnTANBglghkgBZQMEAgEFAKBpMBgGCSqGSIb3DQEJAzELBgkq
# hkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTIyMDcxMjEwMzkxNVowLwYJKoZIhvcN
# AQkEMSIEIOMNZtVF19hhgFy5CcHSFIJXBrqITGhrrvHzxlyOGJYnMA0GCSqGSIb3
# DQEBAQUABIICAHuGdzU645/Igs4jquvXEh/nG8SxrgcVB4cE3Kkm1xtbCBckF8nU
# pFLSRsFpbdVsNqDQhVXa82xR+2sE9CCufrcQtncqdWf5VhgziHW85+Yn1ppZqVRK
# T59ZZBOzKqXh2QgzBxR4OmsfLS7AUokoEVyG+l7bs5yNR+Ju55sjZ5sSbNmPSUWM
# ZtwAOcv7BkqS9rIUb+tJkm4fK1M0g2lepBZThyazAnq0mi7JClfpnF73C0Z8JMjV
# 8HuT15B2EKxpoPTkSiKQq6EzR1jTVIeFsSOwiLXQL/rOOcp3gHSPQQic0UoSLz9z
# 4Z6YeXNaagu7kwuJcV5UxE70jFot18CQSzQv4QNX2ryC32Q90F6PrJpkoZcveMv7
# 08ksuqH3pMjQvtOHPorAZMT/W2xe5JdbIcYHwY7Y5wD+8CsINfUBcjB7/TSurjzQ
# pgu+pkRsRmo+nNujcA5HSod1KWkuz9W1+VdEPoUpTkqdE0ZLDbdchjsQfTHRu3hZ
# rM86MBLDTonQq93I9YvK34J/vuPyc7k3/96rWy2tlPUYaWCOSO7GKGga7E+S5B2M
# QuPnhdUTbQE+dYf/NwXQL59DeUa9+FT3ipzt2I3xcjURgcLDJ9vEY5c3jg137FqB
# RocyLPeJT70SV6U4h5+PjvyScgnAfBrdThgjwsqhDNB2p/OyOtOqHSN1
# SIG # End signature block
