[CmdletBinding(SupportsShouldProcess = $True)]
param(
	[parameter(ValueFromPipeline = $false, ValueFromPipelineByPropertyName = $true)]
	[switch] $IncludeSystem
)
function Set-ModuleStatus { 
	[CmdletBinding(SupportsShouldProcess = $True)]
	param	(
		[parameter(ValueFromPipeline = $true, ValueFromPipelineByPropertyName = $true, Mandatory = $true, HelpMessage = "No module name specified!")] 
		[ValidateNotNullOrEmpty()]
		[string] $name
	)
	PROCESS{
		# Executes once for each pipeline object
		# the $_ variable represents the current input object		
		if (!(Get-Module -name "$name")) { 
			if (Get-Module -ListAvailable | Where-Object Name -eq "$name") { 
				Import-Module -Name "$name" 
				# module was imported
				# return $true
			} else {
				# module was not available
				# return $false
			}
		} else {
			# Write-Output "$_ module already imported"
			# return $true
		} 
	} # end PROCESS
} # end function Set-ModuleStatus


function Get-Data
{
	param ($Server,$DBName)
	if ($IncludeSystem){
		[string] $whereClause = $null
		}
		else{
			[string] $whereClause = "where R.UserAtHost not like 'RtcApplication-%' "
		}
	$connstring = "server=$Server\$DBName;database=rtcdyn;trusted_connection=true;"
	$command = New-Object System.Data.SqlClient.SqlCommand
	if ($pool -and (Get-CsService -PoolFqdn $Pool -Registrar).Version -eq 6)
	{
		# SQL query for Lync Server 2013
		$command.CommandText = "Select (cast (RE.ClientApp as varchar (100))) as ClientVersion, R.UserAtHost as 								UserName, RA.Fqdn `
								From rtcdyn.dbo.RegistrarEndpoint RE `
								Inner Join rtcdyn.dbo.Endpoint EP on RE.EndpointId = EP.EndpointId `
								Inner Join rtc.dbo.Resource R on R.ResourceId = RE.OwnerId `
								Inner Join rtcdyn.dbo.Registrar RA on EP.RegistrarId = RA.RegistrarId `
								$whereClause `
								Order By ClientVersion, UserName"
	}
	else
	{
		# SQL query for Lync Server 2010
		$command.CommandText = "Select (cast (RE.ClientApp as varchar (100))) as ClientVersion, R.UserAtHost as 								UserName, FE.Fqdn `
								From rtcdyn.dbo.RegistrarEndpoint RE `
								Inner Join rtcdyn.dbo.Endpoint EP on RE.EndpointId = EP.EndpointId `
								Inner Join rtc.dbo.Resource R on R.ResourceId = RE.OwnerId `
								Inner Join rtcdyn.dbo.FrontEnd FE on EP.RegistrarId = FE.FrontEndId `
								$whereClause `
								Order By ClientVersion, UserName"
	}
	$connection = New-Object System.Data.SqlClient.SqlConnection
	$connection.ConnectionString = $connstring
   	$connection.Open()
	$command.Connection = $connection
	
	[object] $sqladapter = New-Object System.Data.SqlClient.SqlDataAdapter
	$sqladapter.SelectCommand = $command
 
	[object] $results = New-Object System.Data.Dataset
	$recordcount = $sqladapter.Fill($results)
	$connection.Close()
	return $Results.Tables[0]
}

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

Set-ModuleStatus -name Lync
$ShowFullClient = $true
$domainname = ([System.DirectoryServices.ActiveDirectory.Domain]::GetComputerDomain()).name
$hostname = $env:computername
$fqdnname = $hostname + "." + $domainname
$backenddbname = "rtclocal"

$feServer = Get-CsComputer -Identity $fqdnname -ea 0

if ($feServer)
{
	$pool = $feServer.pool
	$data = Get-Data -Server $feServer.identity -DBName $backenddbname
	if ($data)
	{
		$OverallRecords = $data
	}
}

if (! $OverallRecords)
{
	Write-Host "Nothing returned from query!" 
 	exit
}
else
{
	$count = 0 
	$userHash = @{}
	$clientHash = @{}
	$serverHash = @{}
	$UserList = @{}
	$OverallRecords | ForEach-object {
	$UserList = $_.UserName

	if (! $userHash.ContainsKey($_.UserName))
	{
		$userHash.add($_.UserName, 1)
	}
	else
	{
		$userHash.set_item($_.UserName, ($userHash.get_item($_.UserName) + 1))
	}	
	
	#if (! $serverHash.ContainsKey($_.fqdn))
	#{
	#	$serverHash.add($_.fqdn, 1)
	#} 
	#else
	#{
	#	$serverHash.set_item($_.fqdn, ($serverHash.get_item($_.fqdn) + 1))
	#}
	
	if ($_.ClientVersion.contains('(') -and (! $ShowFullClient))
	{
		# Get rid of extraneous verbage
		$clientName = $_.ClientVersion.substring(0, $_.ClientVersion.IndexOf('('))			
	}
	else
	{
		# Have a client name with no extraneous verbage or $ShowFullClient switch specified
		$clientName = $_.ClientVersion
		$clientName = $clientName.replace("Microsoft ","")
		$clientName = $clientName.replace("Office ","")
		$clientName = $clientName.replace("AndroidLync","Android")
		$clientName = $clientName.replace("iPadLync","iPad")
		$clientName = $clientName.replace("iPhoneLync","iPhone")
		$clientName = $clientName.replace("WPLync","WP")
	}
	
	if (! $clientHash.ContainsKey($clientName))
	{
		$clientHash.add($clientName, 1)
	}
	else
	{
		$clientHash.set_item($ClientName, ($clientHash.get_item($ClientName) + 1))
	}
	$count++
 	}
}

Write-Host ("{0, -26}{1, -41}{2, 11}" -f "Client Version", "Agent", "Connections")
Write-Host "------------------------------------------------------------------------------"

ForEach ($key in $clientHash.keys | Sort-Object -Descending)
{
	$index = $key.indexof(" ")
 
	if ($index -eq "-1")
	{
		# No second part
		$first = $key
		$second = " "
	}
	else
	{
		# Client version/agent has two main parts
		$first = $key.substring(0, $index)
		$second = $key.substring($index + 1)
	}
 
	$value = $clientHash.$key
	"{0,-26}{1,-45}{2,7}" -f $first, $second, $value
}
Write-Host "------------------------------------------------------------------------------"

Write-Host "`r`n`r`n"
Write-Host ("{0,-41}{1,15}" -f "Front End Servers", "Connections")
Write-Host "--------------------------------------------------------"
ForEach ($key in ($serverHash.keys | Sort-Object))
{
	$value = $serverHash.$key
	[string]$Percent = "("+"{0:P2}" -f ($value/$count)+")"
	"{0,-40}{1,6} {2,9}" -f $key, $value, $Percent
}
Write-Host "--------------------------------------------------------"
"{0,-40}{1,6}" -f "Total connections", $count 

$UniqueUsers = $userHash.count
Write-Host "`r`n`r`n"
Write-Host ("{0,-41}{1,15}" -f "Total Unique Users/Clients", "Total")
Write-Host "--------------------------------------------------------"
"{0,-41}{1,15}" -f "Users Connected", $userHash.count
[int]$TotalUsers = (Get-CsUser | Measure-Object).count
[int]$TotalEVUsers = (Get-CsUser -filter {EnterpriseVoiceEnabled -eq $True} | Measure-Object).Count
[string]$TotalPercent = "{0:P2}" -f ($UniqueUsers/$TotalUsers)
Write-Host "`b`b`b`b`b`b`b`b`b`b`b`b`b`b`b`b`b`b`b" -NoNewline
"{0,-41}{1,15}" -f "Lync Enabled Users", $TotalUsers
"{0,-41}{1,15}" -f "Voice Enabled Users", $TotalEVUsers
"{0,-41}{1,15}" -f "Percentage of Enabled Users Connected", $TotalPercent
"{0,-41}{1,15}" -f "Client Versions Connected", $clientHash.count
Write-Host "--------------------------------------------------------"
Write-Host "`r`n`r`n"
		
Write-Host ("{0,-45}{1,-11}" -f "Connected Users", "Connections")
Write-Host "--------------------------------------------------------"
ForEach ($key in $userHash.keys | Sort-Object)
{
	$value = $userHash.$key
	"{0,-45}{1,11}" -f $key, $value
}
Write-Host "--------------------------------------------------------"



