﻿#requires -Version 3.0
#requires -Module DHCPServer
#This File is in Unicode format.  Do not edit in an ASCII editor. Notepad++ UTF-8-BOM

#region help text
<#
.SYNOPSIS
	Creates a complete inventory of a Microsoft 2012+ DHCP server.
.DESCRIPTION
	Creates a complete inventory of a Microsoft 2012+ DHCP server using Microsoft 
	PowerShell, Word, plain text or HTML.
	
	Script requires at least PowerShell version 3 but runs best in version 5.
	
	Requires the DHCPServer module.
	Can be run on a DHCP server or on a Windows 8.x or Windows 10 computer with RSAT installed.
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520
	
	For Windows Server 2003, 2008 and 2008 R2, use the following to export and import the 
	DHCP data:
		Export from the 2003, 2008 or 2008 R2 server:
			netsh dhcp server export C:\DHCPExport.txt all
			
			Copy the C:\DHCPExport.txt file to the 2012+ server.
			
		Import on the 2012+ server:
			netsh dhcp server import c:\DHCPExport.txt all
			
		The script can now be run on the 2012+ DHCP server to document the older DHCP 
		information.

	For Windows Server 2008 and Server 2008 R2, the 2012+ DHCP Server PowerShell cmdlets 
	can be used for the export and import.
		From the 2012+ DHCP server:
			Export-DhcpServer -ComputerName 2008R2Server.domain.tld -Leases -File 
			C:\DHCPExport.xml 
			
			Import-DhcpServer -ComputerName 2012Server.domain.tld -Leases -File 
			C:\DHCPExport.xml -BackupPath C:\dhcp\backup\ 
			
			Note: The c:\dhcp\backup path must exist before the Import-DhcpServer 
			cmdlet is run.
	
	Using netsh is much faster than using the PowerShell export and import cmdlets.
	
	Processing of IPv4 Multicast Scopes is only available with Server 2012 R2 DHCP.
	
	Word and PDF Documents include a Cover Page, Table of Contents and Footer.
	
.PARAMETER ComputerName
	DHCP server to run the script against.
	The computername is used for the report title.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
.PARAMETER AllDHCPServers
	The script will process all Authorized DHCP servers that are online.
	"All DHCP Servers" is used for the report title.
	This parameter is disabled by default.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
	This parameter has an alias of ALL.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  This script creates a Word, PDF, HTML or 
	formatted text document.
#>
#endregion


#region script parameters
[CmdletBinding(SupportsShouldProcess = $False, ConfirmImpact = "None", DefaultParameterSetName = "Word") ]

Param(
	[parameter(Mandatory=$False)] 
	[string]$ComputerName="LocalHost",
	
	[parameter(Mandatory=$False)] 
	[Alias("ALL")]
	[Switch]$AllDHCPServers=$False
	)
#endregion

#region initialize variables 
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False
$scriptStartIdentifier = "*** Start - DHCP config data ***"
$scriptEndIdentifier = "*** End - DHCP config data ***"
$global:output = ""
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

#region general script functions
Function InsertBlankLine
{
	Line 0 ""
}

Function TestComputerName
{
	Param([string]$Cname)
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		##Write-Verbose "$(Get-Date): Testing to see if $($CName) is online and reachable"
		If(Test-Connection -ComputerName $CName -quiet)
		{
			##Write-Verbose "$(Get-Date): Server $($CName) is online."
		}
		Else
		{
			##Write-Verbose "$(Get-Date): Computer $($CName) is offline"
			$ErrorActionPreference = $SaveEAPreference
			#Write-Error "`n`n`t`tComputer $($CName) is offline.`n`t`tScript cannot continue.`n`n"
			Exit
		}
	}

	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
		##Write-Verbose "$(Get-Date): Computer name has been renamed from localhost to $($CName)"
		##Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			##Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
			$ErrorActionPreference = $SaveEAPreference
			#Write-Error "`n`n`t`tComputer $($CName) is not a DHCP Server.`n`n`t`tRerun the script using -ComputerName with a valid DHCP server name.`n`n`t`tScript cannot continue.`n`n"
			Exit
		}
	}

	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip)
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
			##Write-Verbose "$(Get-Date): Computer name has been renamed from $($ip) to $($CName)"
			##Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
			$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
			If($? -and $Null -ne $results)
			{
				#the computer is a dhcp server
				##Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
				Return $CName
			}
			ElseIf(!$? -or $Null -eq $results)
			{
				#the computer is not a dhcp server
				#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
				$ErrorActionPreference = $SaveEAPreference
				Write-Error "`n`n`t`tComputer $($CName) is not a DHCP Server.`n`n`t`tRerun the script using -ComputerName with a valid DHCP server name.`n`n`t`tScript cannot continue.`n`n"
				Exit
			}
		}
		Else
		{
			Write-Warning "Unable to resolve $($CName) to a hostname"
		}
	}
	Else
	{
		#Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
			$ErrorActionPreference = $SaveEAPreference
			Write-Error "`n`n`t`tComputer $($CName) is not a DHCP Server.`n`n`t`tRerun the script using -ComputerName with a valid DHCP server name.`n`n`t`tScript cannot continue.`n`n"
			Exit
		}
	}
	Return $CName
}

Function TestComputerName2
{
	Param([string]$Cname)
	
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		##Write-Verbose "$(Get-Date): Testing to see if $($CName) is online and reachable"
		If(Test-Connection -ComputerName $CName -quiet)
		{
			##Write-Verbose "$(Get-Date): Server $($CName) is online."
		}
		Else
		{
			#Write-Verbose "$(Get-Date): Computer $($CName) is offline"
			Write-Output "$(Get-Date): Computer $($CName) is offline" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}

	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
		##Write-Verbose "$(Get-Date): Computer name has been renamed from localhost to $($CName)"
		#Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
			Write-Output "$(Get-Date): Computer $($CName) is not a DHCP Server" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}

	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip)
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
			##Write-Verbose "$(Get-Date): Computer name has been renamed from $($ip) to $($CName)"
			#Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
			$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
			If($? -and $Null -ne $results)
			{
				#the computer is a dhcp server
				#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
				Return $CName
			}
			ElseIf(!$? -or $Null -eq $results)
			{
				#the computer is not a dhcp server
				#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
				Write-Output "$(Get-Date): Computer $($CName) is not a DHCP Server" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
				Return "BAD"
			}
		}
		Else
		{
			#Write-Verbose "$(Get-Date): Unable to resolve $($CName) to a hostname"
			Write-Output "$(Get-Date): Unable to resolve $($CName) to a hostname" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}
	Else
	{
		#Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
			Write-Output "$(Get-Date): Computer $($CName) is not a DHCP Server" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}

	#Write-Verbose "$(Get-Date): "
	Return $CName
}
#endregion

#region line output functions
Function line
#for creating the formatted text report
{
	Param( [int]$tabs = 0, [string]$name = '', [string]$value = '', [string]$newline = "`r`n", [switch]$nonewline )
	While( $tabs -gt 0 ) { $Global:Output += "`t"; $tabs--; }
	If( $nonewline )
	{
		$Global:Output += $name + $value
	}
	Else
	{
		$Global:Output += $name + $value + $newline
	}
}
#endregion

#region DHCP script functions

Function ProcessServerProperties
{
	#Write-Verbose "$(Get-Date): Server Properties and Configuration"
	#Write-Verbose "$(Get-Date): "

	#Write-Verbose "$(Get-Date): Getting DHCP server information"

	$Global:Output = ""
	
	$tmp = $Script:DHCPServerName.Split(".")
	$NetBIOSName = $tmp[0]

	Line 0 "DHCP Server Information: " $NetBIOSName
	Line 1 "Server name`t: " $Script:DHCPServerName

	$DHCPDB = Get-DHCPServerDatabase -ComputerName $Script:DHCPServerName -EA 0

	If($? -and $Null -ne $DHCPDB)
	{
		Line 1 "Database path`t: " $DHCPDB.FileName.SubString(0,($DHCPDB.FileName.LastIndexOf('\')))
		Line 1 "Backup path`t: " $DHCPDB.BackupPath
	}
	Else
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving DHCP Server Database information"
	}
	InsertBlankLine

	$DHCPDB = $Null

	[bool]$Script:GotServerSettings = $False
	$Script:ServerSettings = Get-DHCPServerSetting -ComputerName $Script:DHCPServerName -EA 0

	If($? -and $Null -ne $Script:ServerSettings)
	{
		$Script:GotServerSettings = $True
		#some properties of $Script:ServerSettings will be needed later
		If($Script:ServerSettings.IsAuthorized)
		{
			Line 1 "DHCP server is authorized"
		}
		Else
		{
			Line 1 "DHCP server is not authorized"
		}
	}
	Else
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving DHCP Server setting information"
	}
	Write-Output $Global:Output
	InsertBlankLine
	[gc]::collect() 
}

Function ProcessIPv4Bindings
{
	$IPv4Bindings = Get-DHCPServerV4Binding -ComputerName $Script:DHCPServerName -EA 0 | Sort-Object IPAddress

	$Global:Output = ""

	If($? -and $Null -ne $IPv4Bindings)
	{
		Line 0 "Connections and server bindings"
	
		ForEach($IPv4Binding in $IPv4Bindings)
		{
			If($IPv4Binding.BindingState)
			{
				Line 1 "Enabled " -NoNewLine
			}
			Else
			{
				Line 1 "Disabled " -NoNewLine
			}
			Line 0 "$($IPv4Binding.IPAddress) $($IPv4Binding.InterfaceAlias)"
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving IPv4 server bindings"
	}
	Else
	{
		Line 1 "There were no IPv4 server bindings"
	}
	$IPv4Bindings = $Null
	[gc]::collect() 

	Write-Output $Global:Output

	InsertBlankLine
}

Function ProcessIPv6Bindings
{
	#Write-Verbose "$(Get-Date): `tGetting IPv6 bindings"
	$IPv6Bindings = Get-DHCPServerV6Binding -ComputerName $Script:DHCPServerName -EA 0 | Sort-Object IPAddress

	$Global:Output = ""

	If($? -and $Null -ne $IPv6Bindings)
	{
		WriteWordLine 0 0 "Connections and server bindings:"
		ForEach($IPv6Binding in $IPv6Bindings)
		{
			If($IPv6Binding.BindingState)
			{
				Line 1 "Enabled " -NoNewLine
			}
			Else
			{
				Line 1 "Disabled " -NoNewLine
			}
			Line 0 "$($IPv6Binding.IPAddress) $($IPv6Binding.InterfaceAlias)"
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving IPv6 server bindings"
	}
	Else
	{
		Line 1 "There were no IPv6 server bindings"
	}
	$IPv6Bindings = $Null
	[gc]::collect() 
	Write-Output $Global:Output
}

Function ProcessIPv4Properties
{
	$Global:Output = ""

	#Write-Verbose "$(Get-Date): Getting IPv4 properties"
	Line 0 ""
	Line 0 "IPv4"
	Line 0 "Properties"

	#Write-Verbose "$(Get-Date): `tGetting IPv4 general settings"
	Line 1 "General"

	[bool]$Script:GotAuditSettings = $False
	$Script:AuditSettings = Get-DHCPServerAuditLog -ComputerName $Script:DHCPServerName -EA 0

	If($? -and $Null -ne $Script:AuditSettings)
	{
		$Script:GotAuditSettings = $True
		If($Script:AuditSettings.Enable)
		{
			Line 2 "DHCP audit logging is enabled"
		}
		Else
		{
			Line 2 "DHCP audit logging is disabled"
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving audit log settings"
	}
	Else
	{
		Line 0 "There were no audit log settings"
	}
	[gc]::collect() 

	#"HKLM:\SYSTEM\CurrentControlSet\Services\DHCPServer\Parameters" "BootFileTable"

	#Define the variable to hold the BOOTP Table
	$BOOTPKey="SYSTEM\CurrentControlSet\Services\DHCPServer\Parameters" 

	#Create an instance of the Registry Object and open the HKLM base key
	$reg=[microsoft.win32.registrykey]::OpenRemoteBaseKey('LocalMachine',$Script:DHCPServerName) 

	#Drill down into the BOOTP key using the OpenSubKey Method
	$regkey1=$reg.OpenSubKey($BOOTPKey) 

	#Retrieve an array of string that contain all the subkey names
	If($Null -ne $regkey1)
	{
		$Script:BOOTPTable = $regkey1.GetValue("BootFileTable") 
	}
	Else
	{
		$Script:BOOTPTable = $Null
	}

	If($Null -ne $Script:BOOTPTable)
	{
		Line 2 "Show the BOOTP table folder is enabled"
	}

	#DNS settings
	#Write-Verbose "$(Get-Date): `tGetting IPv4 DNS settings"
	Line 1 "DNS"

	$DNSSettings = Get-DHCPServerV4DnsSetting -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $DNSSettings)
	{
		GetDNSSettings $DNSSettings "A"
	}
	Else
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving IPv4 DNS Settings for DHCP server $Script:DHCPServerName"
	}
	$DNSSettings = $Null
	[gc]::collect() 

	#now back to some server settings
	#Write-Verbose "$(Get-Date): `tGetting IPv4 NAP settings"
	Line 1 "Network Access Protection"

	If($Script:GotServerSettings)
	{
		Line 2 "Network Access Protection is " -NoNewLine
		If($Script:ServerSettings.NapEnabled)
		{
			Line 0 "Enabled on all scopes"
		}
		Else
		{
			Line 0 "Disabled on all scopes"
		}
		Line 2 "DHCP server behavior when NPS is unreachable: " -NoNewLine
		Switch($Script:ServerSettings.NpsUnreachableAction)
		{
			"Full"			{Line 0 "Full Access"}
			"Restricted"	{Line 0 "Restricted Access"}
			"NoAccess"		{Line 0 "Drop Client Packet"}
			Default			{Line 0 "Unable to determine NPS unreachable action: $($Script:ServerSettings.NpsUnreachableAction)"}
		}
	}

	#filters
	#Write-Verbose "$(Get-Date): `tGetting IPv4 filters"
	Line 1 "Filters"

	$MACFilters = Get-DHCPServerV4FilterList -ComputerName $Script:DHCPServerName -EA 0

	If($? -and $Null -ne $MACFilters)
	{
		Line 2 "Enable Allow list`t: " -NoNewLine
		If($MACFilters.Allow)
		{
			Line 0 "Enabled"
		}
		Else
		{
			Line 0 "Disabled"
		}
		Line 0
		Line 2 "Enable Deny list`t: " -NoNewLine
		If($MACFilters.Deny)
		{
			Line 0 "Enabled"
		}
		Else
		{
			Line 0 "Disabled"
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving MAC filters for DHCP server $Script:DHCPServerName"
	}
	Else
	{
		Line 2 "There were no MAC filters for DHCP server $Script:DHCPServerName"
	}
	$MACFilters = $Null
	[gc]::collect() 

	#failover
	#Write-Verbose "$(Get-Date): `tGetting IPv4 Failover"
	Line 1 "Failover"

	$Failovers = Get-DHCPServerV4Failover -ComputerName $Script:DHCPServerName -EA 0

	If($? -and $Null -ne $Failovers)
	{
		ForEach($Failover in $Failovers)
		{
			#Write-Verbose "$(Get-Date):`t`tProcessing failover $($Failover.Name)"
			Line 2 "Relationship name: " $Failover.Name
					
			Line 2 "State of the server`t: " -NoNewLine
			Switch($Failover.State)
			{
				"NoState" 
				{
					Line 0 "No State"
				}
				"Normal" 
				{
					Line 0 "Normal"
				}
				"Init" 
				{
					Line 0 "Initializing"
				}
				"CommunicationInterrupted" 
				{
					Line 0 "Communication Interrupted"
				}
				"PartnerDown" 
				{
					Line 0 "Normal"
				}
				"PotentialConflict" 
				{
					Line 0 "Potential Conflict"
				}
				"Startup" 
				{
					Line 0 "Starting Up"
				}
				"ResolutionInterrupted" 
				{
					Line 0 "Resolution Interrupted"
				}
				"ConflictDone" 
				{
					Line 0 "Conflict Done"
				}
				"Recover" 
				{
					Line 0 "Recover"
				}
				"RecoverWait" 
				{
					Line 0 "Recover Wait"
				}
				"RecoverDone" 
				{
					Line 0 "Recover Done"
				}
				Default 
				{
					Line 0 "Unable to determine server state"
				}
			}
					
			Line 2 "Partner Server`t`t: " $Failover.PartnerServer
			Line 2 "Mode`t`t`t: " $Failover.Mode
			Line 2 "Message Authentication`t: " -NoNewLine
			If($Failover.EnableAuth)
			{
				Line 0 "Enabled"
			}
			Else
			{
				Line 0 "Disabled"
			}
					
			If($Failover.Mode -eq "LoadBalance")
			{
				$tmp = (100 - $($Failover.LoadBalancePercent))
				Line 2 "Local server`t`t: $($Failover.LoadBalancePercent)%"
				Line 2 "Partner Server`t`t: $($tmp)%"
				$tmp = $Null
			}
			Else
			{
				Line 2 "Role of this server`t: " $Failover.ServerRole
				Line 2 "Addresses reserved for standby server: $($Failover.ReservePercent)%"
			}
					
			#skip a row for spacing
			Line 0 ""
		}
	}
	Else
	{
		Line 2 "There was no Failover configured for DHCP server $Script:DHCPServerName"
	}
	$Failovers = $Null
	[gc]::collect() 

	#Advanced
	#Write-Verbose "$(Get-Date): `tGetting IPv4 advanced settings"
	Line 1 "Advanced"

	If($Script:GotServerSettings)
	{
		Line 2 "Conflict detection attempts`t: " $Script:ServerSettings.ConflictDetectionAttempts
	}

	If($Script:GotAuditSettings)
	{
		Line 2 "Audit log file path`t`t: " $Script:AuditSettings.Path
	}

	#added 18-Jan-2016
	#get dns update credentials
	#Write-Verbose "$(Get-Date): `tGetting DNS dynamic update registration credentials"
	$DNSUpdateSettings = Get-DhcpServerDnsCredential -ComputerName $Script:DHCPServerName -EA 0

	If($? -and $Null -ne $DNSUpdateSettings)
	{
		Line 2 "DNS dynamic update registration credentials: "
		Line 3 "User name`t: " $DNSUpdateSettings.UserName
		Line 3 "Domain`t`t: " $DNSUpdateSettings.DomainName
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving DNS Update Credentials for DHCP server $Script:DHCPServerName"
	}
	Else
	{
		Line 2 "There were no DNS Update Credentials for DHCP server $Script:DHCPServerName"
	}
	$DNSUpdateSettings = $Null
	[gc]::collect() 

	Write-Output $Global:Output
}

Function ProcessIPv4Statistics
{
	#Write-Verbose "$(Get-Date): Getting IPv4 Statistics"
	$Global:Output = ""

	Line 1 "Statistics"
	[gc]::collect() 

	$Statistics = Get-DHCPServerV4Statistics -ComputerName $Script:DHCPServerName -EA 0

	If($? -and $Null -ne $Statistics)
	{
		$UpTime = $(Get-Date) - $Statistics.ServerStartTime
		$Str = [string]::format("{0} days, {1} hours, {2} minutes, {3} seconds", `
			$UpTime.Days, `
			$UpTime.Hours, `
			$UpTime.Minutes, `
			$UpTime.Seconds)
		[int]$InUsePercent = "{0:N0}" -f $Statistics.PercentageInUse
		[int]$AvailablePercent = "{0:N0}" -f $Statistics.PercentageAvailable
		
		Line 2 "Description" -NoNewLine
		Line 3 "Details"

		Line 2 "Start Time:" -NoNewLine
		Line 3 $Statistics.ServerStartTime
		Line 2 "Up Time:" -NoNewLine
		Line 3 $Str
		Line 2 "Discovers:" -NoNewLine
		Line 3 $Statistics.Discovers
		Line 2 "Offers:" -NoNewLine
		Line 4 $Statistics.Offers
		Line 2 "Delayed Offers:" -NoNewLine
		Line 3 $Statistics.DelayedOffers
		Line 2 "Requests:" -NoNewLine
		Line 3 $Statistics.Requests
		Line 2 "Acks:" -NoNewLine
		Line 4 $Statistics.Acks
		Line 2 "Nacks:" -NoNewLine
		Line 4 $Statistics.Naks
		Line 2 "Declines:" -NoNewLine
		Line 3 $Statistics.Declines
		Line 2 "Releases:" -NoNewLine
		Line 3 $Statistics.Releases
		Line 2 "Total Scopes:" -NoNewLine
		Line 3 $Statistics.TotalScopes
		Line 2 "Scopes w/delay configured:" -NoNewLine
		Line 1 $Statistics.ScopesWithDelayConfigured
		Line 2 "Total Addresses:" -NoNewLine
		$tmp = "{0:N0}" -f $Statistics.TotalAddresses
		Line 2 $tmp
		Line 2 "In Use:" -NoNewLine
		Line 4 "$($Statistics.AddressesInUse) ($($InUsePercent))%"
		Line 2 "Available:" -NoNewLine
		$tmp = "{0:N0}" -f $Statistics.AddressesAvailable
		Line 3 "$($tmp) ($($AvailablePercent))%"
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving IPv4 statistics"
	}
	Else
	{
		Line 0 "There were no IPv4 statistics"
	}
	$Statistics = $Null
	[gc]::collect() 

	Write-Output $Global:Output
}

Function GetDNSSettings
{
	Param([object]$DNSSettings, [string]$As)
	
	Line 2 "Enable DNS dynamic updates`t`t`t: " -NoNewLine
	If($DNSSettings.DynamicUpdates -eq "Never")
	{
		Line 0 "Disabled"
	}
	ElseIf($DNSSettings.DynamicUpdates -eq "OnClientRequest")
	{
		Line 0 "Enabled"
		Line 2 "Dynamically update DNS $($As) & PTR records only if requested by the DHCP clients"
	}
	ElseIf($DNSSettings.DynamicUpdates -eq "Always")
	{
		Line 0 "Enabled"
		Line 2 "Always dynamically update DNS $($As) & PTR records"
	}
	Line 2 "Discard $($As) & PTR records when lease deleted`t: " -NoNewLine
	If($DNSSettings.DeleteDnsRROnLeaseExpiry)
	{
		Line 0 "Enabled"
	}
	Else
	{
		Line 0 "Disabled"
	}
	Line 2 "Name Protection`t`t`t`t`t: " -NoNewLine
	If($DNSSettings.NameProtection)
	{
		Line 0 "Enabled"
	}
	Else
	{
		Line 0 "Disabled"
	}
}

Function ProcessIPv6Properties
{
	$Global:Output = ""
	#IPv6

	Line 0 "IPv6"
	Line 0 "Properties"

	#Write-Verbose "$(Get-Date): Getting IPv6 properties"
	#Write-Verbose "$(Get-Date): `tGetting IPv6 general settings"
	Line 1 "General"

	If($Script:GotAuditSettings)
	{
		If($Script:AuditSettings.Enable)
		{
			Line 2 "DHCP audit logging is enabled"
		}
		Else
		{
			Line 2 "DHCP audit logging is disabled"
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving audit log settings"
	}
	Else
	{
		Line 2 "There were no audit log settings"
	}

	#DNS settings
	#Write-Verbose "$(Get-Date): `tGetting IPv6 DNS settings"
	Line 1 "DNS"
	$DNSSettings = Get-DHCPServerV6DnsSetting -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $DNSSettings)
	{
		GetDNSSettings $DNSSettings "AAAA"
	}
	Else
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving IPv6 DNS Settings for DHCP server $Script:DHCPServerName"
	}
	$DNSSettings = $Null

	#Advanced
	#Write-Verbose "$(Get-Date): `tGetting IPv6 advanced settings"
	Line 1 "Advanced"
	If($Script:GotAuditSettings)
	{
		Line 2 "Audit log file path " $Script:AuditSettings.Path
	}
	$Script:AuditSettings = $Null

	#added 18-Jan-2016
	#get dns update credentials
	#Write-Verbose "$(Get-Date): `tGetting DNS dynamic update registration credentials"
	$DNSUpdateSettings = Get-DhcpServerDnsCredential -ComputerName $Script:DHCPServerName -EA 0

	If($? -and $Null -ne $DNSUpdateSettings)
	{
		Line 2 "DNS dynamic update registration credentials: "
		Line 3 "User name`t: " $DNSUpdateSettings.UserName
		Line 3 "Domain`t`t: " $DNSUpdateSettings.DomainName
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving DNS Update Credentials for DHCP server $Script:DHCPServerName"
	}
	Else
	{
		Line 2 "There were no DNS Update Credentials for DHCP server $Script:DHCPServerName"
	}
	$DNSUpdateSettings = $Null
	[gc]::collect() 
	
	Line 1 "Statistics"
	$Statistics = Get-DHCPServerV6Statistics -ComputerName $Script:DHCPServerName -EA 0

	If($? -and $Null -ne $Statistics)
	{
		$UpTime = $(Get-Date) - $Statistics.ServerStartTime
		$Str = [string]::format("{0} days, {1} hours, {2} minutes, {3} seconds", `
			$UpTime.Days, `
			$UpTime.Hours, `
			$UpTime.Minutes, `
			$UpTime.Seconds)
		[int]$InUsePercent = "{0:N0}" -f $Statistics.PercentageInUse
		[int]$AvailablePercent = "{0:N0}" -f $Statistics.PercentageAvailable

		Line 2 "Description" -NoNewLine
		Line 2 "Details"

		Line 2 "Start Time: " -NoNewLine
		Line 2 $Statistics.ServerStartTime
		Line 2 "Up Time: " -NoNewLine
		Line 2 $Str
		Line 2 "Solicits: " -NoNewLine
		Line 2 $Statistics.Solicits
		Line 2 "Advertises: " -NoNewLine
		Line 2 $Statistics.Advertises
		Line 2 "Requests: " -NoNewLine
		Line 2 $Statistics.Requests
		Line 2 "Replies: " -NoNewLine
		Line 2 $Statistics.Replies
		Line 2 "Renews: " -NoNewLine
		Line 2 $Statistics.Renews
		Line 2 "Rebinds: " -NoNewLine
		Line 2 $Statistics.Rebinds
		Line 2 "Confirms: " -NoNewLine
		Line 2 $Statistics.Confirms
		Line 2 "Declines: " -NoNewLine
		Line 2 $Statistics.Declines
		Line 2 "Releases: " -NoNewLine
		Line 2 $Statistics.Releases
		Line 2 "Total Scopes: " -NoNewLine
		Line 2 $Statistics.TotalScopes
		Line 2 "Total Addresses: " -NoNewLine
		$tmp = "{0:N0}" -f $Statistics.TotalAddresses
		Line 1 $tmp
		Line 2 "In Use: " -NoNewLine
		Line 2 "$($Statistics.AddressesInUse) ($($InUsePercent)%)"
		Line 2 "Available: " -NoNewLine
		$tmp = "{0:N0}" -f $Statistics.AddressesAvailable 
		Line 2 "$($tmp) ($($AvailablePercent)%)"
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Line 0 "Error retrieving IPv6 statistics"
	}
	Else
	{
		Line 0 "There were no IPv6 statistics"
	}
	
	$Statistics = $Null

	Write-Output $Global:Output
}
#endregion

#region script setup function
Function ProcessScriptSetup
{
	$script:startTime = Get-Date
	
	#pre 1.40
	#$ComputerName = TestComputerName $ComputerName
	#$Script:DHCPServerName = $ComputerName
	
	#change for 1.40 and -AllDHCPServers
	$Script:DHCPServerNames = @()
	If($AllDHCPServers -eq $False)
	{
		##Write-Verbose "$(Get-Date): Resolving computer name"
		$ComputerName = TestComputerName $ComputerName
		$Script:DHCPServerNames += $ComputerName
	}
	Else
	{
		#Write-Verbose "$(Get-Date): Retrieving all DHCP servers in domain"
		$ComputerName = "All DHCP Servers"
		
		$ALLServers = Get-DHCPServerInDc -EA 0
		
		If($Null -eq $AllServers)
		{
			#oops no DHCP servers
			Write-Error "Unable to retrieve any DHCP servers.  Script cannot continue"
			Exit
		}
		Else
		{
			[int]$cnt = 0
			If($AllServers -is [array])
			{
				$cnt = $AllServers.Count
				#Write-Verbose "$(Get-Date): $($cnt) DHCP servers were found"
			}
			Else
			{
				$cnt = 1
				#Write-Verbose "$(Get-Date): $($cnt) DHCP server was found"
			}
			
			$Script:BadDHCPErrorFile = "$($pwd.Path)\BadDHCPServers_$(Get-Date -f yyyy-MM-dd_HHmm).txt"

			ForEach($Server in $AllServers)
			{
				$Result = TestComputerName2 $Server.DnsName
				
				If($Result -ne "BAD")
				{
					$Script:DHCPServerNames += $Result
				}
			}
			#Write-Verbose "$(Get-Date): $($Script:DHCPServerNames.Count) DHCP servers will be processed"
			#Write-Verbose "$(Get-Date): "
		}
	}
}
#endregion

#Script begins

ProcessScriptSetup

Write-Host $scriptStartIdentifier

ForEach($DHCPServer in $Script:DHCPServerNames)
{
	$Script:DHCPServerName = $DHCPServer
	ProcessServerProperties
	ProcessIPv4Bindings
	ProcessIPv6Bindings
	ProcessIPv4Properties
	ProcessIPv4Statistics
	ProcessIPv6Properties
	InsertBlankLine
}
If(-Not $isErrorOccured)
{
	Write-Host $scriptEndIdentifier
}
#endregion