#requires -Version 3.0
#requires -Module DHCPServer
#This File is in Unicode format.  Do not edit in an ASCII editor. Notepad++ UTF-8-BOM

#region help text
<#
.SYNOPSIS
	Creates a complete inventory of a Microsoft 2012+ DHCP server.
.DESCRIPTION
	Creates a complete inventory of a Microsoft 2012+ DHCP server using Microsoft 
	PowerShell, Word, plain text or HTML.
	
	Script requires at least PowerShell version 3 but runs best in version 5.
	
	Requires the DHCPServer module.
	Can be run on a DHCP server or on a Windows 8.x or Windows 10 computer with RSAT installed.
		
	Remote Server Administration Tools for Windows 8 
		http://www.microsoft.com/en-us/download/details.aspx?id=28972
		
	Remote Server Administration Tools for Windows 8.1 
		http://www.microsoft.com/en-us/download/details.aspx?id=39296
		
	Remote Server Administration Tools for Windows 10
		http://www.microsoft.com/en-us/download/details.aspx?id=45520
	
	For Windows Server 2003, 2008 and 2008 R2, use the following to export and import the 
	DHCP data:
		Export from the 2003, 2008 or 2008 R2 server:
			netsh dhcp server export C:\DHCPExport.txt all
			
			Copy the C:\DHCPExport.txt file to the 2012+ server.
			
		Import on the 2012+ server:
			netsh dhcp server import c:\DHCPExport.txt all
			
		The script can now be run on the 2012+ DHCP server to document the older DHCP 
		information.

	For Windows Server 2008 and Server 2008 R2, the 2012+ DHCP Server PowerShell cmdlets 
	can be used for the export and import.
		From the 2012+ DHCP server:
			Export-DhcpServer -ComputerName 2008R2Server.domain.tld -Leases -File 
			C:\DHCPExport.xml 
			
			Import-DhcpServer -ComputerName 2012Server.domain.tld -Leases -File 
			C:\DHCPExport.xml -BackupPath C:\dhcp\backup\ 
			
			Note: The c:\dhcp\backup path must exist before the Import-DhcpServer 
			cmdlet is run.
	
	Using netsh is much faster than using the PowerShell export and import cmdlets.
	
	Processing of IPv4 Multicast Scopes is only available with Server 2012 R2 DHCP.
	
	Word and PDF Documents include a Cover Page, Table of Contents and Footer.
	
.PARAMETER ComputerName
	DHCP server to run the script against.
	The computername is used for the report title.
	ComputerName can be entered as the NetBIOS name, FQDN, localhost or IP Address.
	If entered as localhost, the actual computer name is determined and used.
	If entered as an IP address, an attempt is made to determine and use the actual 
	computer name.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
.PARAMETER AllDHCPServers
	The script will process all Authorized DHCP servers that are online.
	"All DHCP Servers" is used for the report title.
	This parameter is disabled by default.
	
	If both ComputerName and AllDHCPServers are used, AllDHCPServers is used.
	This parameter has an alias of ALL.

.INPUTS
	None.  You cannot pipe objects to this script.
.OUTPUTS
	No objects are output from this script.  This script creates a Word, PDF, HTML or 
	formatted text document.
#>
#endregion
#region script parameters
[CmdletBinding(SupportsShouldProcess = $False, ConfirmImpact = "None", DefaultParameterSetName = "Word") ]
Param(
	[parameter(Mandatory=$False)] 
	[string]$ComputerName="LocalHost",
	
	[parameter(Mandatory=$False)] 
	[Alias("ALL")]
	[Switch]$AllDHCPServers=$False
	)
#endregion

#region initialize variables
$ErrorActionPreference = 'SilentlyContinue'
$global:isErrorOccured = $False
$scriptStartIdentifier = "*** Start - DHCP config data ***"
$scriptEndIdentifier = "*** End - DHCP config data ***"
$delimiter = "@@@"
$notApplicable = "N/A"
#endregion

$OutputEncoding = [console]::InputEncoding = [console]::OutputEncoding = New-Object System.Text.UTF8Encoding
$PSDefaultParameterValues = @{'*:Encoding' = 'utf8'}

Function TestComputerName
{
	Param([string]$Cname)
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		If(Test-Connection -ComputerName $CName -quiet)
		{
			##Write-Verbose "$(Get-Date): Server $($CName) is online."
		}
		Else
		{
			$ErrorActionPreference = $SaveEAPreference
			#Write-Error "`n`n`t`tComputer $($CName) is offline.`n`t`tScript cannot continue.`n`n"
			Exit
		}
	}
	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
		
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			
			$ErrorActionPreference = $SaveEAPreference
			#Write-Error "`n`n`t`tComputer $($CName) is not a DHCP Server.`n`n`t`tRerun the script using -ComputerName with a valid DHCP server name.`n`n`t`tScript cannot continue.`n`n"
			Exit
		}
	}
	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip)
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
			
			$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
			If($? -and $Null -ne $results)
			{
				#the computer is a dhcp server
				
				Return $CName
			}
			ElseIf(!$? -or $Null -eq $results)
			{
				#the computer is not a dhcp server
				
				$ErrorActionPreference = $SaveEAPreference
				Write-Error "`n`n`t`tComputer $($CName) is not a DHCP Server.`n`n`t`tRerun the script using -ComputerName with a valid DHCP server name.`n`n`t`tScript cannot continue.`n`n"
				Exit
			}
		}
		Else
		{
			Write-Warning "Unable to resolve $($CName) to a hostname"
		}
	}
	Else
	{	
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
		
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			$ErrorActionPreference = $SaveEAPreference
			Write-Error "`n`n`t`tComputer $($CName) is not a DHCP Server.`n`n`t`tRerun the script using -ComputerName with a valid DHCP server name.`n`n`t`tScript cannot continue.`n`n"
			Exit
		}
	}
	Return $CName
}

Function TestComputerName2
{
	Param([string]$Cname)	
	If(![String]::IsNullOrEmpty($CName)) 
	{
		#get computer name
		#first test to make sure the computer is reachable
		If(Test-Connection -ComputerName $CName -quiet)
		{
			##Write-Verbose "$(Get-Date): Server $($CName) is online."
		}
		Else
		{
			#Write-Verbose "$(Get-Date): Computer $($CName) is offline"
			Write-Output "$(Get-Date): Computer $($CName) is offline" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}
	#if computer name is localhost, get actual computer name
	If($CName -eq "localhost")
	{
		$CName = $env:ComputerName
		
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			Write-Output "$(Get-Date): Computer $($CName) is not a DHCP Server" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}
	#if computer name is an IP address, get host name from DNS
	$ip = $CName -as [System.Net.IpAddress]
	If($ip)
	{
		$Result = [System.Net.Dns]::gethostentry($ip)
		
		If($? -and $Null -ne $Result)
		{
			$CName = $Result.HostName
		
			$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
			If($? -and $Null -ne $results)
			{
				#the computer is a dhcp server
				Return $CName
			}
			ElseIf(!$? -or $Null -eq $results)
			{
				#the computer is not a dhcp server
				Write-Output "$(Get-Date): Computer $($CName) is not a DHCP Server" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
				Return "BAD"
			}
		}
		Else
		{
			Write-Output "$(Get-Date): Unable to resolve $($CName) to a hostname" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}
	Else
	{
		#Write-Verbose "$(Get-Date): Testing to see if $($CName) is a DHCP Server"
		$results = Get-DHCPServerVersion -ComputerName $CName -EA 0
		If($? -and $Null -ne $results)
		{
			#the computer is a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is a DHCP Server"
			Return $CName
		}
		ElseIf(!$? -or $Null -eq $results)
		{
			#the computer is not a dhcp server
			#Write-Verbose "$(Get-Date): Computer $($CName) is not a DHCP Server"
			Write-Output "$(Get-Date): Computer $($CName) is not a DHCP Server" | Out-File $Script:BadDHCPErrorFile -Append 4>$Null
			Return "BAD"
		}
	}
	#Write-Verbose "$(Get-Date): "
	Return $CName
}
#endregion

#region DHCP script functions

Function ProcessIPv4Statistics
{
	$Statistics = Get-DHCPServerV4Statistics -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $Statistics)
	{
		$totalScopes = $Statistics.TotalScopes
		$totalAddresses = $Statistics.TotalAddresses
		Write-host  "Server Name: $($Script:DHCPServerName) $($delimiter)  protocolVersion : IPV4 $($delimiter)  ParentScopeType/Name : $($notApplicable)  $($delimiter)  ScopeType :  $($notApplicable) $($delimiter) Scopename:  $($notApplicable) $($delimiter) status : $($notApplicable) $($delimiter)  TotalScopes : $($totalScopes) $($delimiter)  TotalAddresses : $($totalAddresses) "	
		$totalScopes = ""
		$totalAddresses = ""
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Host"Error retrieving IPv4 statistics"
	}
	$Statistics = $Null
	[gc]::collect() 
}

Function ProcessIPv4SuperScopeStatistics
{
	Write-Verbose "$(Get-Date): Getting IPv4 Superscopes"
	$IPv4Superscopes = Get-DHCPServerV4Superscope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv4Superscopes)
	{
		ForEach($IPv4Superscope in $IPv4Superscopes)
		{
			If(![string]::IsNullOrEmpty($IPv4Superscope.SuperscopeName))
			{
				
				$Statistics = Get-DHCPServerV4SuperscopeStatistics -ComputerName $Script:DHCPServerName -Name $IPv4Superscope.SuperscopeName -EA 0
				If($? -and $Null -ne $Statistics)
				{
					[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
				}
				ElseIf(!$?)
				{
					$global:isErrorOccured = $True
					Write-Host"Error retrieving IPv4 statistics"
				}
				Write-Host "Server Name: $($Script:DHCPServerName) $($delimiter)  ProtocolVersion : IPV4 $($delimiter) ParentScopeType/Name : $($notApplicable)  $($delimiter) ScopeType : SuperScope $($delimiter) Scopename : $($IPv4Superscope.SuperscopeName) $($delimiter) status : $($notApplicable) $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) totalScopes :  $($Statistics.NumScopes )"
				$TotalAddresses = ""
				
			}
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Host "Error retrieving IPv4 Superscopes"
	}
	
	$IPv4Superscopes = $Null
	[gc]::collect() 
}

Function ProcessIPv4SuperScopeChildScopeStatistics
{
	$IPv4Superscopes = Get-DHCPServerV4Superscope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv4Superscopes)
	{
		ForEach($IPv4Superscope in $IPv4Superscopes)
		{
			If(![string]::IsNullOrEmpty($IPv4Superscope.SuperscopeName))
			{
			    $xScopeIds = $IPv4Superscope.ScopeId
				ForEach($xScopeId in $xScopeIds)
				{
					$IPv4Scope = Get-DHCPServerV4Scope -ComputerName $Script:DHCPServerName -ScopeId $xScopeId -EA 0
					If($? -and $Null -ne $IPv4Scope)
					{
						$ScopeIPScopeName = "[$($IPv4Scope.ScopeId)] $($IPv4Scope.Name)"
						$IPv4ScoepState = $IPv4Scope.state
						$IPV4ScopeStatistics   =  Get-DhcpServerv4ScopeStatistics -ComputerName $Script:DHCPServerName -ScopeId $xScopeId -EA 0
						If($? -and $Null -ne $Statistics)
						{
							[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
						}
						ElseIf(!$?)
						{
							$global:isErrorOccured = $True
							Write-Host"Error retrieving IPv4 statistics"
						}
						Write-Host "Server Name: $($Script:DHCPServerName) $($delimiter) ProtocolVersion : IPV4 $($delimiter) ParentScopeType/Name : SuperScope/$($IPv4Superscope.SuperscopeName) $($delimiter) ScopeType : Scope $($delimiter)  Scopename: $($ScopeIPScopeName) $($delimiter) TotalAddresses : $($TotalScopeAddresses) $($delimiter) status : $($IPv4ScoepState) $($delimiter) totalScopes :  $($notApplicable)"
						$TotalScopeAddresses = ""
						$IPv4ScoepState = ""
					}
					Else
					{
						$global:isErrorOccured = $True
						Write-Host "Error retrieving Superscope data for scope $($IPv4Scope)"
					}
				}
		    }
	    }
    }
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Host "Error retrieving IPv4 Superscopes"
	}
	
	$IPv4Superscopes = $Null
	[gc]::collect() 
}

Function ProcessIPv4ScopeStatistics
{
	$IPv4Scopes = Get-DHCPServerV4Scope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv4Scopes)
	{
		ForEach($IPv4Scope in $IPv4Scopes)
		{
			$ScopeIPScopeName = "[$($IPv4Scope.ScopeId)] $($IPv4Scope.Name)"
			$Ipv4ScopeState = $IPv4Scope.state
		   
			$IPV4ScopeStatistics   =  Get-DhcpServerv4ScopeStatistics -ComputerName $Script:DHCPServerName -ScopeId $IPv4Scope.ScopeId -EA 0
			
			If($? -and $Null -ne $IPV4ScopeStatistics)
			{
					[decimal]$TotalScopeAddresses = ($IPV4ScopeStatistics.AddressesFree + $IPV4ScopeStatistics.AddressesInUse)
			}
			ElseIf(!$?)
			{
				$global:isErrorOccured = $True
				Write-Host "Error retrieving scope statistics : $ScopeIPScopeName"
			}					
			Write-Host "Server Name: $($Script:DHCPServerName) $($delimiter) ProtocolVersion : IPV4 $($delimiter)  ParentScopeType/Name : $($notApplicable)  $($delimiter)  ScopeType : Scope $($delimiter) Scopename: $($ScopeIPScopeName) $($delimiter) TotalAddresses : $($TotalScopeAddresses) $($delimiter) status : $($Ipv4ScopeState) $($delimiter) totalScopes :  $($notApplicable)"
			$TotalScopeAddresses = ""
			$Ipv4ScopeState = ""
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Host "Error retrieving IPv4 scopes "
	}
	
	$IPv4Scopes = $Null
	[gc]::collect() 
}
Function ProcessIPv4MulticastScopesStatistics
{
	$CmdletName = "Get-DHCPServerV4MulticastScope"
	If(Get-Command $CmdletName -Module "DHCPServer" -EA 0)
	{
		$IPv4MulticastScopes = Get-DHCPServerV4MulticastScope -ComputerName $Script:DHCPServerName -EA 0

		If($? -and $Null -ne $IPv4MulticastScopes)
		{
			ForEach($IPv4MulticastScope in $IPv4MulticastScopes)
			{	
				$multicastScopeState = $IPv4MulticastScope.state	
				$Statistics = Get-DHCPServerV4MulticastScopeStatistics -ComputerName $Script:DHCPServerName -Name $IPv4MulticastScope.Name -EA 0
				If($? -and $Null -ne $Statistics)
				{
					[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
				}
				ElseIf(!$?)
				{
					$global:isErrorOccured = $True
					Write-Host "Error retrieving scope statistics"
				}	
				Write-Host "Server Name : $($Script:DHCPServerName) $($delimiter) ProtocolVersion : IPV4 $($delimiter)  ParentScopeType/Name : $($notApplicable) $($delimiter) ScopeType : MulticastScope $($delimiter)  Scopename : $($IPv4MulticastScope.Name) $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter) status : $($multicastScopeState) $($delimiter) totalScopes :  $($notApplicable)"
				$TotalAddresses = ""
				$multicastScopeState = ""
				$Statistics = $Null
			}
		}
		ElseIf(!$?)
		{
			$global:isErrorOccured = $True
			Write-Host "Error retrieving IPv4 Multicast scopes"
		}
		
		$IPv4MulticastScopes = $Null
	}
	[gc]::collect() 
}

Function ProcessIPv6Statistics
{
	$Statistics = Get-DHCPServerV6Statistics -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $Statistics)
	{
		$totalScopes = $Statistics.TotalScopes
		$totalAddresses = $Statistics.TotalAddresses
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Host "Error retrieving IPV6 statistics"
	}
	
    Write-host  "Server Name: $($Script:DHCPServerName) $($delimiter)  protocolVersion : IPV6 $($delimiter) ParentScopeType/Name : $($notApplicable) $($delimiter)  ScopeType : $($notApplicable) $($delimiter) Scopename: $($notApplicable) $($delimiter) TotalScopes : $($totalScopes) $($delimiter)  TotalAddresses : $($totalAddresses) $($delimiter) status : $($notApplicable) "
    $totalScopes = ""
	$totalAddresses = ""
	$Statistics = $Null
	[gc]::collect() 
}

Function ProcessIPv6ScopesStatistics
{	
	$IPv6Scopes = Get-DHCPServerV6Scope -ComputerName $Script:DHCPServerName -EA 0
	If($? -and $Null -ne $IPv6Scopes)
	{
		ForEach($IPv6Scope in $IPv6Scopes)
		{
			$scopename  = "[$($IPv6Scope.Prefix)] $($IPv6Scope.Name)"
			$Ipv6State = $IPv6Scope.state
			$Statistics = Get-DHCPServerV6ScopeStatistics -ComputerName $Script:DHCPServerName -Prefix $IPv6Scope.Prefix -EA 0
			If($? -and $Null -ne $Statistics)
			{		
					[decimal]$TotalAddresses = ($Statistics.AddressesFree + $Statistics.AddressesInUse)
			}
			ElseIf(!$?)
			{
				$global:isErrorOccured = $True
				Write-Host "Error retrieving IPv6 scope statistics"
			}
		
			Write-host  "Server Name: $($Script:DHCPServerName) $($delimiter)  protocolVersion : IPV6 $($delimiter) ParentScopeType/Name : $($notApplicable) $($delimiter) ScopeType : Scope $($delimiter) Scopename : $($scopename ) $($delimiter) TotalAddresses : $($TotalAddresses) $($delimiter)  status : $($Ipv6State) $($delimiter) totalScopes :  $($notApplicable)"
			$TotalAddresses = ""
			$Ipv6State = ""
		}
	}
	ElseIf(!$?)
	{
		$global:isErrorOccured = $True
		Write-Host "Error retrieving IPv6 scopes"
	}
	$IPv6Scopes = $Null
	[gc]::collect() 
}
#endregion

#region script setup function
Function ProcessScriptSetup
{
	$script:startTime = Get-Date
	
	$Script:DHCPServerNames = @()
	If($AllDHCPServers -eq $False)
	{
	
		$ComputerName = TestComputerName $ComputerName
		$Script:DHCPServerNames += $ComputerName
	}
	Else
	{
				$ComputerName = "All DHCP Servers"
		$ALLServers = Get-DHCPServerInDc -EA 0
		If($Null -eq $AllServers)
		{
			Write-Error "Unable to retrieve any DHCP servers.  Script cannot continue"
			Exit
		}
		Else
		{
			[int]$cnt = 0
			If($AllServers -is [array])
			{
				$cnt = $AllServers.Count
				#Write-Verbose "$(Get-Date): $($cnt) DHCP servers were found"
			}
			Else
			{
				$cnt = 1
				#Write-Verbose "$(Get-Date): $($cnt) DHCP server was found"
			}
			$Script:BadDHCPErrorFile = "$($pwd.Path)\BadDHCPServers_$(Get-Date -f yyyy-MM-dd_HHmm).txt"
			ForEach($Server in $AllServers)
			{
				$Result = TestComputerName2 $Server.DnsName
				
				If($Result -ne "BAD")
				{
					$Script:DHCPServerNames += $Result
				}
			}
			#Write-Verbose "$(Get-Date): $($Script:DHCPServerNames.Count) DHCP servers will be processed"
			#Write-Verbose "$(Get-Date): "
		}
	}
}
#endregion

#region script core
#Script begins

ProcessScriptSetup
Write-Host $scriptStartIdentifier
ForEach($DHCPServer in $Script:DHCPServerNames)
{
	#Write-Verbose "$(Get-Date): Processing DHCP Server: $($DHCPServer)"
	$Script:DHCPServerName = $DHCPServer	
	ProcessIPv4Statistics
	ProcessIPv4SuperScopeStatistics
	ProcessIPv4SuperScopeChildScopeStatistics
	ProcessIPv4ScopeStatistics
	ProcessIPv4MulticastScopesStatistics
	ProcessIPv6Statistics
	ProcessIPv6ScopesStatistics

}
If(-Not $isErrorOccured)
{
	Write-Host $scriptEndIdentifier
}
#endregion
# SIG # Begin signature block
# MIIbjAYJKoZIhvcNAQcCoIIbfTCCG3kCAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUEnUJtWERDdSuOiqYmMEm4hbq
# TuagghWiMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHOFADw3TANBgkqhkiG9w0B
# AQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEwMTAwMDAwMFoXDTMxMDEw
# NjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMTCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpja1HXOhFCvQp1dU2UtAxQ
# tSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9NmqB+in43Stwhd4CGPN4
# bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaPxqPhLxs6t2HWc+xObTOK
# fF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsOjizVI9r0TXhG4wODMSlK
# XAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7EdMMKbaYK02/xWVLwfoYer
# vnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmEvQECAwEAAaOCAbgwggG0
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsG
# AQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkwJwYIKwYBBQUHAgEWG2h0
# dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSMEGDAWgBT0tuEgHf4prtLk
# YaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgtHUQ23eNqerwwcQYDVR0f
# BGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
# ZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEyLWFz
# c3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYIKwYBBQUHMAGGGGh0dHA6
# Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
# ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJRFRpbWVzdGFtcGluZ0NB
# LmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7znGucgDo5nRv1CclF0CiNH
# o6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJmTe1ppA/2uHDPYuj1UUp4
# eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3Y18jUmmDgvoaU+2QzI2h
# F3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74wOFcz8QRcucbZEnYIpp1
# FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPWH1ub9y4bTxMd90oNcX6X
# t/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHLKNpbh6aKLzCCBTAwggQY
# oAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEx
# MC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUgU2lnbmluZyBD
# QTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAPjTsxx/DhGvZ3cH0wsx
# SRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5idQ3Gde2qvCchqXYJawO
# eSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQdLEoJrskacLCUvIUZ4qJ
# RdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+rHCiq85/6XzLkqHlOzEc
# z+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE86xnTrXE94zRICUj6whk
# PlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pzINAPZHM8np+mM6n9Gd8l
# k9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8CAQAwDgYDVR0PAQH/BAQD
# AgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUFBwEBBG0wazAkBggrBgEF
# BQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRw
# Oi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0Eu
# Y3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3JsNC5kaWdpY2VydC5jb20v
# RGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2hjRodHRwOi8vY3JsMy5k
# aWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsME8GA1UdIARI
# MEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
# Y2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQWBBRaxLl7KgqjpepxA8Bg
# +S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzANBgkqhkiG
# 9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwtOhrE7zBh134LYP3DPQ/E
# r4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GOYw4TS63XX0R58zYUBor3
# nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEoAjwukaPAJRHinBRHoXpo
# aK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhCMrI2iiQC/i9yfhzXSUWW
# 6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOBm0/GkxAG/AeB+ova+YJJ
# 92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEwggQZoAMCAQICEAqhJdbW
# Mht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
# BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIG
# A1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAw
# MFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lD
# ZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGln
# aUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCCASIwDQYJKoZI
# hvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6gpnFOVQoV7YjSsQOB0Uz
# URB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvuzKb2Mln+
# X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oiPhisEeTwmQNtO4V8CdPu
# XciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287DxgaqwvB8z9
# 8OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgPhH+fMRTWrdXyZMt7HgXQ
# hBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
# ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1bjAfBgNVHSMEGDAWgBRF
# 66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB
# /wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEFBQcBAQRtMGswJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
# dENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDovL2NybDQuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0cDovL2Ny
# bDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDBQBgNV
# HSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cu
# ZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggEB
# AHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
# KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIlHsS6HHssIeLWWywUNUME
# aLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN1
# 1ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHKeZR+WfyMD+NvtQEm
# tmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR7
# 9VYzIi8iNrJLokqV2PWmjlIwggYzMIIFG6ADAgECAhAImnSeuC3D7CfqZerb+DUx
# MA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lD
# ZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcgQ0EwHhcNMjEwNTI1MDAw
# MDAwWhcNMjQwNTI5MjM1OTU5WjBxMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEPMA0GA1UEBxMGSXNlbGluMR0wGwYDVQQKExRlRyBJbm5vdmF0aW9u
# cywgSW5jLjEdMBsGA1UEAxMUZUcgSW5ub3ZhdGlvbnMsIEluYy4wggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDEZQnSmuNt/FyaBkaN3skPE6zuzJIe9pDf
# AZHYpsPdplqB5IT6QbS0qQSxDidrxuc1WJj/MXg9ujRmRi0zMkiTQBlcNMy9ev+Y
# S+YrEJuIkVMKkqSZnixrkFl87uKAi9w1JrTMgjZrmOJAkXLzmP5/0jgSBOj05x1V
# 4g1eGBDbFnpOMQr6luGqZVe+vwMHbolyU5p53QvlWzv4JsbNm4qJ+DEvaLx3kUPO
# SFlJl90K0a0S8Tmkt6xdJZ+nNR128I1GKwg8YHuaD5GWfZe9RnLJpFRG2XHXWlh8
# n8o1gCRQIvbeueHfz+ZrG1K916KhgGPjEvK3umhAEb5vS7MKOBEAN7f+TQ22JXAk
# ijRkSxTIMA2nskgiMx4/Ow4xjN8w/KLxyUDB8FhzZeFK5RxffIp/iSlEWYKVh8rk
# uy0ZBE1s42OmwIFMSBm7AR+R05TSuZozLBkVUscSmalhl4/PszdWE35pXnqRH95+
# jx4PzUTlUNCqYmIZYqr+z2OFuRgfuDa6EvYxdv+5GyFUNI6QVG8ggP+X7IMmiAMG
# lp7fHai/li+NjxmjxEXS1e/JLSt+wO8A1nDuam3MefQGWbDmW6DcXYJUmk1vNvUe
# YAfqTTIUeZZgKhQA1sEHRhH+Tgm+BjZvK20XtxA8nKXtQj0K7gpMCDtE6YaTB0uG
# Zeg8IFyjQQIDAQABo4IBxDCCAcAwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt
# 9mV1DlgwHQYDVR0OBBYEFPnIRJrP2BNJLxoE1rWvfgwwDjZKMA4GA1UdDwEB/wQE
# AwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRw
# Oi8vY3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOg
# MYYvaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5j
# cmwwSwYDVR0gBEQwQjA2BglghkgBhv1sAwEwKTAnBggrBgEFBQcCARYbaHR0cDov
# L3d3dy5kaWdpY2VydC5jb20vQ1BTMAgGBmeBDAEEATCBhAYIKwYBBQUHAQEEeDB2
# MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wTgYIKwYBBQUH
# MAKGQmh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1
# cmVkSURDb2RlU2lnbmluZ0NBLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEB
# CwUAA4IBAQCUCOdwU+8srLBP/gMQIbtsdu2JWX666Z9GPZisW4KGyCS/U+ZCmB/K
# nELeD4StX3JFXz5ddILA5oone/pEud+mEuETBhzb7RCz82WJvk0DEMa4hzMAh5c5
# zmRFGm8WxDZdI20ztcvW0sObqn9UbsGhQDKpFeww91ZtOQ3/2d0KYZKypzGMba53
# w7pWG170nktocFWTeUhkAy0foAJ2jlBOszj2dshTxAa5Fg1dkpEsBS7oiAPpYkox
# V9ldaXJ8Px1navd6PpDW4A781u13f3tP4FVCsNYw4bz4Z5GInCh6/FshaHRJ6dk5
# SsCr29MlpK8UxK9y2Qmb+jEtRxR0QZLfMYIFVDCCBVACAQEwgYYwcjELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIENvZGUg
# U2lnbmluZyBDQQIQCJp0nrgtw+wn6mXq2/g1MTAJBgUrDgMCGgUAoHAwEAYKKwYB
# BAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
# NwIBCzEOMAwGCisGAQQBgjcCARUwIwYJKoZIhvcNAQkEMRYEFI8FkasH2AV9O7DU
# 8F6xHDvY0ICxMA0GCSqGSIb3DQEBAQUABIICACv/lKbf8rgX51+MDP6LyUaKR/2u
# /J4dyUxw/4J+sEbI6JifaFE/kutxGTeAzC7DGJqrOEpfVSGc2VO50aKqF+cvdrAy
# pxh2E2lNP2xG3hbkXnV/7gQmtHwpxSDwyMxVeeA9DeNAYYTP8RrKyMDO8AE7v+oB
# NpEOszwVDV64Pn0lTxiO5WszrgXjY6a9v4Hgbu+U7YJoGaJwqyHtBXkkCMS8myTI
# NmApqqIv6cXzyHo8gAyFnxHniYx19dw71CZlvgmIuWkM7VO03Ln9USpBM9uBGHUQ
# VVYnN2aDu+nTTsdvMJZzqB4HQ24BG+6J8lDJGg2+6WETyQC7P1PDD7yIJeXupaza
# kXlDU/mZEfMTjtFBilqfAJxcfWYfQwaggcOEdEiqG8nv6V77LY4CgxMEQFFq73cr
# htyng4y3Q7S+MYQUWDXq4XcE0QWPYFIBZiyXdP1uoGRL9zpuzg0ILAd8kB/Q2uPz
# nlYGPSySMf18E6Nb2vLuymxB8LWUPSTLFiqSx+yiZ0WTkK4Xr1d7Ts9QodVeuDZF
# gTMKdfH7sTjtkjgl5eowtoyse8KcvxwpMBWz+HIjzqdk3ePA1APq3KU9hdhU8EqF
# h9bVod+9nNZwlJUbob35Q+loxsAHZ/DH5+joj5M02pSm4GK9fLT17cdgSheorcTX
# 6M4X0iAvwpsFUbn3oYICMDCCAiwGCSqGSIb3DQEJBjGCAh0wggIZAgEBMIGGMHIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
# RCBUaW1lc3RhbXBpbmcgQ0ECEA1CSuC+Ooj/YEAhzhQA8N0wDQYJYIZIAWUDBAIB
# BQCgaTAYBgkqhkiG9w0BCQMxCwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0y
# MTA5MTYwNjUxMDlaMC8GCSqGSIb3DQEJBDEiBCBqbnZPgKMiQwUttdFKn8wYLXpu
# UAaVpoaMr59xnCHWqzANBgkqhkiG9w0BAQEFAASCAQC5kA2JhdG3TasIdMoZrKvw
# c/GSQWd2vbSw/ETqh4X8pB/V+ns8UySqvkub+feyZgBVYme5a9Zcqv7EISWWZMJ+
# 0ChrEOIRozy6DcgGlOYomT3c/D/ugoBp+PO1iPbeamnHB7yDjonT2nn/yCPBIQ/I
# GF2OGOkdYrtxahgoKuDu+ld536oDGwIAinCTtBSpXnUI3ZSqencl/W5n2fPZ74uB
# LDm9EBwyS6pkI9HmwHYTAztj/aQ9ZKJsvrTW+yh/4CnIl+My5KVGwoSeszPjzEXQ
# sWHFnpr7dy7wXs2dHpCQ7n7NIBWXP7Od8zyuq6hZdmHnwV+/eXH6NcQa1GY4w3zh
# SIG # End signature block
