'use strict'


const v8 = require('v8');
const fs = require('fs');
const path = require('path');
const semver = require('semver');

const logger = require('../logger');
const appInfo = require('../app-info');
const appUtils = require('../app-utils');

const appendMsg = 'memory-profiler:';
const dumpPath = '/datastorage/memorydump/';

function MemoryProfiler(agent) {
  this.agent = agent;
  this.isRunning = false;
  this.fullDumpPath = path.normalize(appInfo.homeDirectory + dumpPath);

  if (semver.gte(process.version, '11.13.0')) {
    this.coreProfiler = true;
    logger.info(appendMsg, 'core v8 will be used to take snapshot');
    return;
  }

  try {
    this.v8Profiler = require("v8-profiler-next");
    logger.info(appendMsg, 'v8-profiler-next npm required successfully...');
  } catch (e) {
    logger.error(appendMsg, "v8-profiler-next  couldn't be required", e);
  }
}

module.exports = MemoryProfiler;

MemoryProfiler.prototype.takeSnapshot = function () {
  if (!this.coreProfiler && !this.v8Profiler) {
    logger.debug(appendMsg, "v8-profiler-next is not found, so can't do the Snapshot");
    return;
  }

  if (this.isRunning) {
    logger.debug(appendMsg, "don't profiler more than one at a time");
    return;
  }

  const _self = this;
  try {
    _ensureDirectoryExistence(this.fullDumpPath);
    const doDump = this.coreProfiler ? _doV8Snapshot : _doSnapshot;
    doDump.call(this, (e, data) => {
      this.isRunning = false;
      if (e) return;
      _processDump.call(_self, data);
    });

    this.isRunning = true;
  } catch (err) {
    logger.error(appendMsg, "Error while snapshoting", err);
  }
}

function _doV8Snapshot(cb) {
  try {
    logger.info(appendMsg, `snapshot started`);
    const start = Date.now();
    const snapshotStream = v8.getHeapSnapshot();
    logger.info(appendMsg, `snapshot completed and took ${(Date.now() - start) / 1000} seconds`);

    appUtils.streamToString(snapshotStream).then(result => {
      logger.info(appendMsg, `snapshot export completed...!`);
      if (cb) cb(null, result);
    }).catch(err => {
      logger.error(appendMsg, "Error while exporting snapshot", err);
      if (cb) cb(true);
    });
  } catch (e) {
    _deleteSnapshot.call(this, snapshot);
    logger.error(appendMsg, "Error while snapshoting", e);
    if (cb) cb(true);
  }
}

function _doSnapshot(cb) {
  try {
    logger.info(appendMsg, `snapshot started`);
    const start = Date.now();
    const snapshot = this.v8Profiler.takeSnapshot();
    logger.info(appendMsg, `snapshot completed and took ${(Date.now() - start) / 1000} seconds`);

    snapshot.export(function (err, result) {
      logger.info(appendMsg, `snapshot export completed...!`);
      _deleteSnapshot.call(this, snapshot);
      if (err) {
        logger.error(appendMsg, "Error while exporting snapshot", err);
        if (cb) cb(true);
        return;
      }

      logger.debug(appendMsg, `snapshot completed`);
      if (cb) cb(null, result);
    });
  } catch (e) {
    _deleteSnapshot.call(this, snapshot);
    logger.error(appendMsg, "Error while snapshoting", e);
    if (cb) cb(true);
  }
}

function _processDump(data) {
  const fileName = `${this.fullDumpPath}/m_${Date.now()}.heapsnapshot`;
  fs.writeFile(fileName, data, function (err) {
    if (!err) {
      logger.info(appendMsg, `The dump is writen in ${fileName}`);
    } else {
      logger.error(appendMsg, `Error while writing a dump`, err);
    }
  });
}

function _deleteSnapshot(snapshot) {
  if (snapshot) snapshot.delete();
  this.v8Profiler.deleteAllSnapshots();
}

function _ensureDirectoryExistence(filePath) {
  if (!filePath) throw new Error("File Path not found...!");
  if (fs.existsSync(filePath)) return true;
  fs.mkdirSync(filePath, { recursive: true });
}
