'use strict'

const fs = require('fs');
const path = require('path');
const winston = require('winston');
const DailyRotateFile = require('winston-daily-rotate-file');

const appInfo = require('./app-info');
let config = require('./default_config');
const threads = require('./utils/threads');
const safeStringify = require('./utils/stringify');

let SPLAT = null;
const logger = {};
let logFormat = null;
let tid = threads.threadId;
const logDirectory = appInfo.logDirectory;
const BTM_PATH = logDirectory + '/btm';
const STARTUP_PATH = logDirectory + '/startup';
tid = tid > 0 ? ` - [${tid}]` : '';
const levels = { "silly": 0, "debug": 1, "verbose": 2, "info": 3, "warn": 4, "error": 5 };

if ((process.argv && process.argv.length && process.argv.indexOf("--clear-eg-logs") > -1) || process.env.CLEAR_EG_LOGS_ON_START == "true") {
  _emptyLogs();
}

//Always winston libray is exists beacuse we bundle default npm lib with source
try {
  Object.keys(levels).forEach(level => {
    logger[level] = function () {
      if (levels[level] < levels[logger.level]) return;
      winston[level].apply(this, arguments);
    }
  });

  SPLAT = require('triple-beam').SPLAT;
  logFormat = winston.format.printf(info => {
    if (levels[info.level] < levels[logger.level]) return info;

    const { timestamp, level, message } = info;
    const rest = info[SPLAT] || [];
    const wid = global.workerId > -1 ? ' - ' + global.workerId : '';
    let result = `${timestamp} - ${process.pid + wid + tid} - ${level}: ${info.stack ? formatObject(info.stack) : formatObject(message)}`;
    let isConsole = false;
    const lastEle = rest[rest.length - 1];

    if (lastEle && lastEle.isConsoleLog) {
      rest.pop();
      isConsole = true;
    }

    if (rest.length && !info.stack) {
      const msg = rest.map(formatObject).join(' ')
      result += ` ${msg}`;
      if (isConsole) consoleLog(msg);
    }

    return result;
  });

  if (threads.isMainThread) {
    init(STARTUP_PATH);
  }
} catch (e) {
  consoleError('Winston Error', e);
}

function formatObject(param) {
  if (!param) return param;
  if (typeof param === 'string') return param;

  if (param instanceof Error) {
    return param.stack ? param.stack : safeStringify(param, null, 2);
  }

  return logger.level === 'silly' ? safeStringify(param, null, 2) : safeStringify(param);
};

function init(logPath) {
  ensureDirectoryExistence(logPath);

  try {
    const transports = [
      new DailyRotateFile({
        filename: `${logPath}_%DATE%`,
        datePattern: 'YYYY',
        extension: ".log",
        maxSize: config.log_file_size + 'm', // If using the units,add 'k', 'm', or 'g' as the suffix.
        maxFiles: config.log_file_count,
        auditFile: logDirectory + '/winston_audit.json',
      })
    ];

    if (config.enable_console_log) {
      transports.push(
        new winston.transports.Console()
      );
    }

    winston.configure({
      exitOnError: false,
      handleRejections: true,
      format: winston.format.combine(
        winston.format.timestamp({
          format: 'YYYY/MM/DD HH:mm:ss:SSSS',
        }),
        logFormat
      ),
      transports
    });

    if (config.log_level) {
      setLevel(config.log_level);
    }

  } catch (e) {
    //Note:-Hide the console 
    console.error('Logger init', e)
  }
}

//Level : error,warn,info,verbose,debug 
function setLevel(level) {
  logger.level = level;
  winston.level = level;
  config.log_level = level;
}

function setConfig(_config) {
  config = _config || config;
  init(BTM_PATH)
}

function ensureDirectoryExistence(filePath) {
  var dirname = path.dirname(filePath);
  if (fs.existsSync(dirname)) return true;
  fs.mkdirSync(dirname);
}

function _emptyLogs() {
  if (!fs.existsSync(logDirectory)) return;
  const files = fs.readdirSync(logDirectory) || [];
  for (const file of files) {
    fs.unlinkSync(path.join(logDirectory, file));
  }
  process._rawDebug("Logs folder (" + logDirectory + ") cleared successfully...!");
}

function consoleLog(msg, msg2, msg3) {
  process._rawDebug(`${timestamp()} INFO [eG] ${msg}`, formatObject(msg2) || '', formatObject(msg3) || '');
}

function consoleError(msg, err) {
  console.error(`${timestamp()} ERROR [eG] ${msg}`, err || '');
}

function timestamp() {
  const d = new Date();
  const offset = -d.getTimezoneOffset();

  return pad(d.getDate()) + "-" +
    pad(d.getMonth() + 1) + "-" +
    pad(d.getFullYear()) + " " +
    pad(d.getHours()) + ":" +
    pad(d.getMinutes()) + ":" +
    pad(d.getSeconds()) + " " +
    ((offset >= 0 ? "+" : "-") + pad(parseInt(offset / 60)) + ":" + pad(offset % 60))
}

function pad(n) {
  return n < 10 ? "0" + n : n;
}

logger.setLevel = setLevel;
logger.setConfig = setConfig;
logger.consoleLog = consoleLog;
logger.consoleError = consoleError;
logger.doConsole = { isConsoleLog: true };
module.exports = logger;