'use strict'

const Path = require('path');

const Cache = require('./cache');
const logger = require('./logger');
const info = require('./app-info');
const Runtime = require('./runtime');
const appUtils = require('./app-utils');
const Collector = require('./collector');
const Discovery = require('./discovery');
const packageJson = require('../package.json');
const configReader = require('./config-reader');
const ConfigWatcher = require('./config-watcher');
const DEFAULT_CONFIG = require('./default_config');
const Instrumentation = require('./instrumentation');
const InfraManager = require('./infra/infra-manager');
const ErrorHandler = require('./instrumentation/error-handler');
const ErrorReporter = require('./instrumentation/error-reporter');
const ThreadInstrumentation = require('./worker-threads/thread-instrumentation');
const appendMsg = 'Agent:';

module.exports = Agent;
const STATES = {
  'STARTING': 'starting',
  'RUNNING': 'running',
  'STOPPED': 'stopped',
  'ERROR': 'error'
};

function Agent() {
  this.version = '1.0.0';
  this.state = STATES.STOPPED;
  this.config = Object.assign({}, DEFAULT_CONFIG);
  this.info = info;
  this.metadata = _getCache.call(this);
  this.discovery = new Discovery(this);
  this.infraManager = new InfraManager(this);
  this.collector = new Collector(this);
  this.errorHandler = new ErrorHandler(this);
  this.errorReporter = new ErrorReporter(this);
  this._instrumentation = new Instrumentation(this);
  this.configWatcher = new ConfigWatcher(this);
  this.threadInstrumentation = new ThreadInstrumentation(this);

  this._httpClient = null;
  this.errorReporter.start();
  this._externalTransport = null;
}

/**
 * Start the agent
 * @param {Object} options contains config details appname, key, log level etc 
 * 
 * Load default config
 * Start agent to collecting process level metrics
 */
Agent.prototype.start = function (options) {
  if (global.EGURKHA) {
    logger.info(appendMsg, 'Do not start ' + EGURKHA + ' more than once');
    return this;
  }

  options = options || {};
  logger.consoleLog(`Node.js version                         : ${info.nodeVersion}`);
  logger.consoleLog(`Node.js Monitor Log directive path      : ${info.logDirectory}`);
  logger.consoleLog(`Node.js Monitor home directive path     : ${info.homeDirectory}`);
  logger.consoleLog(`Node.js Monitor version                 : ${packageJson.version}, Built on ${packageJson.buildDate}`);

  logger.info(appendMsg, 'starting...!');
  this.setConfig(options);
  appInfo(this);
  logger.info(appendMsg, 'Config log level:', this.getConfig('log_level'));
  logger.setConfig(this.config);
  this.threadInstrumentation.start();
  this._instrumentation.start();

  if (!options.unique_component_id) {
    /* if unique_component_id is not found then start the discovery */
    logger.info(appendMsg, "GUID is not set in the profiler options");
    const guid = this.metadata.get('GUID');
    const nickPort = this.metadata.get('NICK_PORT');

    if (guid) {
      this.config.component_id = nickPort;
      this.config.unique_component_id = guid;
      logger.info(appendMsg, `GUID: (${guid}) and nick port ${nickPort} are found from ${this.metadata.path}`);
    } else {
      logger.info(appendMsg, "BTM is going stop because the GUID is not found");
      this._instrumentation.stop();
      this.discovery.start();
    }
  } else {
    configReader.load(this);
  }

  const _self = this;
  this.runtime = new Runtime(this);
  this.runtime.start(_ => {
    _self.collector.start();
    _self.state = STATES.STARTING;
    global.EGURKHA = 'EG-Node-Monitor';

    if (!options.unique_component_id) {
      logger.info(appendMsg, 'Inframetrics is not enabled because the GUID is not found');
      return;
    }

    _self.infraManager.start();
  });

  logger.consoleLog(`NODE.JS MONITOR IS ENABLED SUCCESSFULLY.`);
  return this;
}

/**
 * Stop the agent
 */
Agent.prototype.stop = function (cb) {
  try {

    this.state = STATES.STOPPED;
    this.infraManager.stop();
    logger.info(appendMsg, 'Agent stopped')

    if (cb && typeof cb === 'function') {
      cb();
    }
  } catch (e) {
    logger.error(appendMsg, 'error while agent stop ', e)
  }
}

function _getCache() {
  const pkgJson = this.info.targetAppPackageJson || {};
  const startingFile = this.info.startingFile && Path.basename(this.info.startingFile) || "";
  let cacheName = '';

  if (pkgJson.name || pkgJson.description) {
    cacheName = (pkgJson.name || pkgJson.description);
    if (startingFile) cacheName += '-' + startingFile;
  } else if (this.config.component_id) {
    cacheName = this.config.component_id;
  } else {
    cacheName = process.ppid || process.pid;
  }

  cacheName = cacheName.toString().replace(/\./g, "-");
  return new Cache(`${cacheName}.json`);
}

Agent.prototype.stopTracing = function () {
  try {
    this.infraManager.stop();
    this.errorReporter.stop();
    this._instrumentation.stop();
    this.threadInstrumentation.stop();
    this.stopedTracing = true;
    logger.info(appendMsg, 'Agent stopped the tracing');
  } catch (e) {
    logger.error(appendMsg, 'error while stop tracing', e)
  }
}

Agent.prototype.startTracing = function () {
  this.infraManager.start();
  this._instrumentation.start();

  try {
    this.errorReporter.start();
    this.threadInstrumentation.start();
    if (this.stopedTracing) {
      logger.info(appendMsg, 'Agent started the tracing');
    }

    this.stopedTracing = false;
  } catch (e) {
    logger.error(appendMsg, 'error while starting error Reporter', e)
  }
}

Agent.prototype.isTransactionFound = function () {
  return !!this._instrumentation.currTransaction();
}

/**
 * To start a new transaction
 */
Agent.prototype.startTransaction = function () {
  if (!this.stopedTracing) {
    return this._instrumentation.startTransaction.apply(this._instrumentation, arguments);
  } else {
    logger.error(appendMsg, 'even after stopped the tracing, some module is trying to create a transaction. Need to fix this.', new Error());
  }
}
/**
 * To end a running transaction
 */
Agent.prototype.endTransaction = function () {
  return this._instrumentation.endTransaction.apply(this._instrumentation, arguments)
}

/**
 * To start a new span
 */
Agent.prototype.startSpan = function (name, type, _pointCutName) {
  return this._instrumentation.startSpan(name, type, _pointCutName);
}

Agent.prototype.currTransaction = function () {
  return this._instrumentation.currTransaction();
}

/**
 * To set addional message for transaction
 * ex insert some message
 */
Agent.prototype.setContext = function (context) {
  var transaction = this.currTransaction();
  if (!transaction) {
    return false
  }
  transaction.setContext(context)
  return true
}

/**
 * Set an external config in Agent 
 * @param {Object} options
 */
Agent.prototype.setConfig = function (options) {
  logger.info(appendMsg, "User config options:", options);

  try {
    Object.keys(options).forEach(key => {
      this.config[key] = options[key];
    });
  } catch (e) {
    logger.error(appendMsg, 'Agent setConfig ', e);
  }
}

Agent.prototype.captureError = function (err, option) {
  try {
    this.errorHandler.handle(err, option);
  } catch (e) {
    logger.error(appendMsg, 'error in captureError fn ', e);
  }
}

Agent.prototype.getProcessableError = function (err) {
  return this.errorHandler.getProcessableError(err);
}

Agent.prototype.getErrorOccuredLine = function (err) {
  return this.errorHandler.getErrorOccuredLine(err);
}

Agent.prototype.getErrorOccuredLineDatils = function (err) {
  return this.errorHandler.getErrorOccuredLineDatils(err);
}

Agent.prototype.sendInfraError = function (err, option) {
  if (!this.getConfig('enable_inframetrics')) {
    logger.warn(appendMsg, 'inframetrics is disabled, so ignoring the error ', err);
    return;
  }

  if (!this.getConfig('enable_infra_exceptions')) {
    logger.debug(appendMsg, 'Infra error is not enabled.');
    return;
  }

  this.infraManager.sendError(err, option);
}

Agent.prototype.flush = function (cb) {
  if (this._transport) {
    // TODO: Only bind the callback if the transport can't use AsyncResource from async hooks
    this._transport.flush(cb && this._instrumentation.bindFunction(cb))
  } else {
    // Log an *err* to provide a stack for the user.
    const err = new Error('cannot flush agent before it is started')
    this.logger.warn({ err }, err.message)
    if (cb) process.nextTick(cb)
  }
}

Agent.prototype.getConfig = function (key) {
  if (this.config[key] === false) return false;
  return this.config[key] || DEFAULT_CONFIG[key];
}

Agent.prototype.bindFunction = function (original) {
  return this._instrumentation.bindFunction(original);
}

// These are metrics about the agent itself -- separate from the metrics
// gathered on behalf of the using app and sent to APM server. Currently these
// are only useful for internal debugging of the APM agent itself.
//
Agent.prototype._getStats = function () {
  const stats = {};
  const runCtxMgr = null;
  if (runCtxMgr && runCtxMgr._runContextFromAsyncId) {
    stats.runContextFromAsyncIdSize = runCtxMgr._runContextFromAsyncId.size;
  }
  return stats;
}

function appInfo(_agent) {
  logger.info('eG Start time:', appUtils.formatDate());
  logger.info(appendMsg, 'Info:', info);
  logger.info(appendMsg, 'Config:', _agent.config);
  logger.info(appendMsg, 'Node.js argument', process.argv);
  logger.info(appendMsg, 'Enviroment Variables', process.env);
}